﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testEns_Common.h"
#include "../../Common/testEns_AccountUtility.h"
#include "../../Common/testEns_NetworkUtility.h"

namespace
{
    nn::socket::ConfigDefaultWithMemory g_SocketConfig;

    nn::os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_ThreadStack[64 * 1024];

    NN_ALIGNAS(4096) nn::Bit8 g_ServiceWorkMemory[nn::ens::RequiredMemorySizeMin + 4 * 1024 * 1024];
}

namespace
{
    void WorkerThread(void*) NN_NOEXCEPT
    {
        nn::ens::StartServiceLoop("acbaa", g_ServiceWorkMemory, sizeof (g_ServiceWorkMemory));
    }
}

class EnsBasic : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::account::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfig));

        nnt::ens::ReserveSockets();
        nnt::ens::InitializeLibcurl();
        nnt::ens::EnableCommunicationLogDump();
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
        nnt::ens::FinalizeLibcurl();
        nnt::ens::CancelSocketsReservation();

        nn::socket::Finalize();
    }

    virtual void SetUp() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Thread, WorkerThread, nullptr,
            g_ThreadStack, sizeof (g_ThreadStack), nn::os::DefaultThreadPriority + 1));

        nn::os::StartThread(&g_Thread);

        nn::nifm::SubmitNetworkRequestAndWait();
    }

    virtual void TearDown() NN_NOEXCEPT
    {
        nn::nifm::CancelNetworkRequest();

        nn::ens::StopServiceLoop();

        nn::os::DestroyThread(&g_Thread);
    }
};

TEST_F(EnsBasic, GenerateCredencial)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    nn::ens::AsyncContext contexts[4];
    nn::ens::Credential credentials[4] = {};

    for (int i = 0; i < NN_ARRAY_SIZE(contexts); i++)
    {
        nn::ens::GenerateCredential(&contexts[i], &credentials[i], token.Get());
    }

    for (int i = 0; i < NN_ARRAY_SIZE(contexts); i++)
    {
        contexts[i].GetEvent().Wait();

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

        NNT_ASSERT_RESULT_SUCCESS(contexts[i].GetResult());

#else

        // Generic ではローカルサーバ起動が必要なので、接続タイムアウトのみ除外する。
        if (contexts[i].GetResult().IsFailure() &&
            !nn::ens::ResultHttpErrorOperationTimedout::Includes(contexts[i].GetResult()))
        {
            NNT_ASSERT_RESULT_SUCCESS(contexts[i].GetResult());
        }

#endif

        if (contexts[i].GetResult().IsSuccess())
        {
            NN_LOG("UserId = %llu\n", credentials[i].userId.value);
            NN_LOG("Password = %s\n", credentials[i]._password);
        }
    }
}

TEST_F(EnsBasic, Cancel1)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    {
        nn::ens::AsyncContext context;
        nn::ens::Credential credential = {};

        nn::ens::GenerateCredential(&context, &credential, token.Get());
        context.Cancel();
        context.GetEvent().Wait();

        if (context.GetResult().IsFailure())
        {
            NNT_EXPECT_RESULT_FAILURE(nn::ens::ResultCanceledByUser, context.GetResult());
        }
    }
    {
        nn::ens::AsyncContext context;
        nn::ens::Credential credential = {};

        nn::ens::GenerateCredential(&context, &credential, token.Get());
        context.GetEvent().TimedWait(nn::TimeSpan::FromMilliSeconds(10));
        context.Cancel();
        context.GetEvent().Wait();

        if (context.GetResult().IsFailure())
        {
            NNT_EXPECT_RESULT_FAILURE(nn::ens::ResultCanceledByUser, context.GetResult());
        }
    }
}

TEST_F(EnsBasic, Cancel2)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    // 実行直後に非同期コンテキストを破棄する。
    {
        nn::ens::AsyncContext context;
        nn::ens::Credential credential = {};

        nn::ens::GenerateCredential(&context, &credential, token.Get());
    }
    {
        nn::ens::AsyncContext context;
        nn::ens::Credential credential = {};

        nn::ens::GenerateCredential(&context, &credential, token.Get());

        context.GetEvent().TimedWait(nn::TimeSpan::FromMilliSeconds(10));
    }
}
