﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../../Common/testEns_Common.h"

#include <nn/ens/ens_SearchQueryBuilder.h>

namespace
{
    template <size_t Size>
    class SearchQueryBuilderWithBuffer : public nn::ens::SearchQueryBuilder
    {
    public:
        //
        SearchQueryBuilderWithBuffer() NN_NOEXCEPT
            : nn::ens::SearchQueryBuilder(m_Buffer, sizeof (m_Buffer))
        {
        }

    private:
        //
        char m_Buffer[Size];
    };
}

TEST(EnsSearchQueryBuilder, Case1)
{
    {
        SearchQueryBuilderWithBuffer<1024> builder;

        builder.Set("name", "ABCDE");

        // &q[name]=ABCDE
        EXPECT_STREQ(builder.GetString(), "&q%5Bname%5D=ABCDE");
    }
    {
        SearchQueryBuilderWithBuffer<1024> builder;

        EXPECT_TRUE(builder.Set("name", "ABCDE"));

        EXPECT_TRUE(builder.SetMulti("tags", "1"));
        EXPECT_TRUE(builder.SetMulti("tags", "2"));
        EXPECT_TRUE(builder.SetMulti("tags", "3"));

        // &q[name]=ABCDE&q[tags][]=1&q[tags][]=2&q[tags][]=3
        EXPECT_STREQ(builder.GetString(), "&q%5Bname%5D=ABCDE&q%5Btags%5D%5B%5D=1&q%5Btags%5D%5B%5D=2&q%5Btags%5D%5B%5D=3");
    }
}

TEST(EnsSearchQueryBuilder, Case2)
{
    {
        SearchQueryBuilderWithBuffer<1024> builder;

        EXPECT_TRUE(builder.Set("name", "-._~"));

        // &q[name]=-._~
        EXPECT_STREQ(builder.GetString(), "&q%5Bname%5D=-._~");
    }
    {
        SearchQueryBuilderWithBuffer<1024> builder;

        EXPECT_TRUE(builder.Set("name", " !\"#$%&'()*+,-./:;<=>?@[\\]^_`{|}~"));

        // &q[name]=' !"#$%&'()*+,-./:;<=>?@[\]^_`{|}~'
        EXPECT_STREQ(builder.GetString(),
            "&q%5Bname%5D=%20%21%22%23%24%25%26%27%28%29%2A%2B%2C-.%2F%3A%3B%3C%3D%3E%3F%40%5B%5C%5D%5E_%60%7B%7C%7D~");
    }
    {
        SearchQueryBuilderWithBuffer<1024> builder;

        EXPECT_TRUE(builder.Set("name", u8"あいうえお"));

        // &q[name]=あいうえお
        EXPECT_STREQ(builder.GetString(), "&q%5Bname%5D=%E3%81%82%E3%81%84%E3%81%86%E3%81%88%E3%81%8A");
    }
    {
        SearchQueryBuilderWithBuffer<1024> builder;

        EXPECT_TRUE(builder.Set("name", u8"あAいIうUえEおO"));

        // &q[name]=あAいIうUえEおO
        EXPECT_STREQ(builder.GetString(), "&q%5Bname%5D=%E3%81%82A%E3%81%84I%E3%81%86U%E3%81%88E%E3%81%8AO");
    }
    {
        SearchQueryBuilderWithBuffer<1024> builder;

        EXPECT_TRUE(builder.SetMulti("tags1", "1"));
        EXPECT_TRUE(builder.SetMulti("tags2", "X"));
        EXPECT_TRUE(builder.SetMulti("tags1", "2"));
        EXPECT_TRUE(builder.SetMulti("tags2", "Y"));

        // &q[tags1][]=1&q[tags2][]=X&q[tags1][]=2&q[tags2][]=Y
        EXPECT_STREQ(builder.GetString(), "&q%5Btags1%5D%5B%5D=1&q%5Btags2%5D%5B%5D=X&q%5Btags1%5D%5B%5D=2&q%5Btags2%5D%5B%5D=Y");
    }
    {
        SearchQueryBuilderWithBuffer<1024> builder;

        EXPECT_TRUE(builder.Set("tags", "1"));

        EXPECT_TRUE(builder.SetMulti("tags", "1"));
        EXPECT_TRUE(builder.SetMulti("tags", "2"));
        EXPECT_TRUE(builder.SetMulti("tags", "3"));

        // 単体と複数での重複は許可する。
        // &q[tags]=1&q[tags][]=1&q[tags][]=2&q[tags][]=3
        EXPECT_STREQ(builder.GetString(), "&q%5Btags%5D=1&q%5Btags%5D%5B%5D=1&q%5Btags%5D%5B%5D=2&q%5Btags%5D%5B%5D=3");
    }
}

TEST(EnsSearchQueryBuilder, SimilarKeys)
{
    SearchQueryBuilderWithBuffer<1024> builder;

    EXPECT_TRUE(builder.Set("n", "ABCDE"));
    EXPECT_TRUE(builder.Set("na", "ABCDE"));
    EXPECT_TRUE(builder.Set("nam", "ABCDE"));
    EXPECT_TRUE(builder.Set("name", "ABCDE"));
    EXPECT_TRUE(builder.Set("name_", "ABCDE"));
    EXPECT_TRUE(builder.Set("N", "ABCDE"));
    EXPECT_TRUE(builder.Set("Na", "ABCDE"));
    EXPECT_TRUE(builder.Set("Nam", "ABCDE"));
    EXPECT_TRUE(builder.Set("Name", "ABCDE"));
    EXPECT_TRUE(builder.Set("Name_", "ABCDE"));
}

TEST(EnsSearchQueryBuilder, Invalid)
{
    SearchQueryBuilderWithBuffer<1024> builder;

    // 半角英数と "-" "_" 以外は指定不可
    EXPECT_FALSE(builder.Set("!", "123"));
    EXPECT_FALSE(builder.Set(u8"あいうえお", "123"));

    // 空文字列は指定不可
    EXPECT_FALSE(builder.Set("", "123"));
    EXPECT_FALSE(builder.Set("name", ""));
}

TEST(EnsSearchQueryBuilder, Overflow)
{
    {
        SearchQueryBuilderWithBuffer<sizeof ("&q%5Bname%5D=1")> builder;

        EXPECT_TRUE(builder.Set("name", "1"));

        // &q[name]=1
        EXPECT_STREQ(builder.GetString(), "&q%5Bname%5D=1");
    }
    {
        SearchQueryBuilderWithBuffer<sizeof ("&q%5Bname%5D=1") - 1> builder;

        EXPECT_FALSE(builder.Set("name", "1"));
    }

    {
        SearchQueryBuilderWithBuffer<sizeof ("&q%5Bname1%5D=1") + sizeof ("&q%5Bname2%5D=1") - 1> builder;

        EXPECT_TRUE(builder.Set("name1", "1"));
        EXPECT_TRUE(builder.Set("name2", "1"));

        // &q[name1]=1&q[name2]=1
        EXPECT_STREQ(builder.GetString(), "&q%5Bname1%5D=1&q%5Bname2%5D=1");
    }
    {
        SearchQueryBuilderWithBuffer<sizeof ("&q%5Bname1%5D=1") + sizeof ("&q%5Bname2%5D=1") - 2> builder;

        EXPECT_TRUE(builder.Set("name1", "1"));
        EXPECT_FALSE(builder.Set("name2", "1"));

        // &q[name1]=1
        EXPECT_STREQ(builder.GetString(), "&q%5Bname1%5D=1");
    }
}
