﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../../Common/testEns_Common.h"
#include "../../../Common/testEns_AccountUtility.h"
#include "../../../Common/testEns_NetworkUtility.h"
#include "../AcbaaCommon/testEns_ApiForAcbaa.h"

#include <nn/nn_Assert.h>
#include <nn/ens/ens_ApiForAcbaa.h>
#include <nn/ens/ens_SearchQueryBuilder.h>
#include <algorithm>
#include <numeric>

namespace
{
    const size_t MetadataBufferSize = 4096;

    const size_t BodyBufferSize = 64 * 1024;
}

namespace
{
    nn::socket::ConfigDefaultWithMemory g_SocketConfig;

    nn::os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_ThreadStack[64 * 1024];

    NN_ALIGNAS(4096) nn::Bit8 g_ServiceWorkMemory[nn::ens::RequiredMemorySizeMin + 4 * 1024 * 1024];

    nn::ens::Credential g_Credentials[2] = {nn::ens::InvalidCredential, nn::ens::InvalidCredential};
}

namespace
{
    void WorkerThread(void*) NN_NOEXCEPT
    {
        nn::ens::StartServiceLoop("acbaa", g_ServiceWorkMemory, sizeof (g_ServiceWorkMemory));
    }
}

class EnsAcbaaBenchmark : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::account::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfig));

        nnt::ens::ReserveSockets();
        nnt::ens::InitializeLibcurl();
        nnt::ens::EnableCommunicationLogDump();

        nn::ens::detail::LibrarySettings::SetResponseDumpEnabled(false);
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
        nnt::ens::FinalizeLibcurl();
        nnt::ens::CancelSocketsReservation();

        nn::socket::Finalize();
    }

    virtual void SetUp() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Thread, WorkerThread, nullptr,
            g_ThreadStack, sizeof (g_ThreadStack), nn::os::DefaultThreadPriority + 1));

        nn::os::StartThread(&g_Thread);

        nn::nifm::SubmitNetworkRequestAndWait();
    }

    virtual void TearDown() NN_NOEXCEPT
    {
        nn::nifm::CancelNetworkRequest();

        nn::ens::StopServiceLoop();

        nn::os::DestroyThread(&g_Thread);
    }

    void SendMessage(const nn::ens::Credential& from, const nn::ens::Credential& to, const char* pNsaIdToken) NN_NOEXCEPT
    {
        nn::Bit8 metadataBuffer[MetadataBufferSize] = {};
        size_t metadataSize = 0;

        // メタデータフィクスチャの取得
        {
            nn::ens::AsyncContext context;
            nn::ens::ReceiveBuffer metadata = {metadataBuffer, sizeof (metadataBuffer)};

            nnt::ens::GetMessageMetadataFixture(&context, &metadata);

            context.GetEvent().Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            metadataSize = metadata.receivedSize;
        }
        // メッセージ送信
        {
            // 現状の仕様では約 39,500 B となる。
            // 将来サイズが増えることを考慮して、40 KiB のボディを送信する。
            static nn::Bit8 s_BodyData[40 * 1024] = {};

            NN_STATIC_ASSERT(sizeof (s_BodyData) <= BodyBufferSize);

            nn::ens::SendBuffer metadata = {metadataBuffer, metadataSize};
            nn::ens::SendBuffer body = {s_BodyData, sizeof (s_BodyData)};

            nn::ens::AsyncContext context;

            nn::ens::SendMessage(&context, to.userId, metadata, body, from, pNsaIdToken);

            context.GetEvent().Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }
    }

    void PostMyDesign(nn::ens::MyDesignAuthorId* pOutId,
        const nn::ens::Credential& credential, const char* pNsaIdToken) NN_NOEXCEPT
    {
        nn::Bit8 metadataBuffer[MetadataBufferSize] = {};
        size_t metadataSize = 0;

        // メタデータフィクスチャの取得
        {
            nn::ens::AsyncContext context;
            nn::ens::ReceiveBuffer metadata = {metadataBuffer, sizeof (metadataBuffer)};

            nnt::ens::GetMyDesignMetadataFixture(&context, &metadata, nullptr);

            context.GetEvent().Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            metadataSize = metadata.receivedSize;
        }
        // 作者登録
        {
            nn::ens::AsyncContext context;

            nn::ens::MyDesignAuthorName name = {"name"};

            nn::ens::RegisterMyDesignAuthorProfile(&context, pOutId, name, credential, pNsaIdToken);

            context.GetEvent().Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }
        // マイデザイン投稿
        {
            // 現状の仕様では約 2300 B となる。
            // 将来サイズが増えることを考慮して、2.5 KiB のボディを送信する。
            static nn::Bit8 s_BodyData[2560] = {};

            NN_STATIC_ASSERT(sizeof (s_BodyData) <= BodyBufferSize);

            nn::ens::SendBuffer metadata = {metadataBuffer, metadataSize};
            nn::ens::SendBuffer body = {s_BodyData, sizeof (s_BodyData)};

            nn::ens::AsyncContext context;
            nn::ens::MyDesignId id;

            nn::ens::PostMyDesign(&context, &id, metadata, body, credential, pNsaIdToken);

            context.GetEvent().Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }
    }

    void DumpStatisticValue(nn::TimeSpan* pDurationList, int count, const char* pTestName) NN_NOEXCEPT
    {
        std::sort(pDurationList, pDurationList + count);

        NN_LOG("##teamcity[buildStatisticValue key='%s_Avg(ms)' value='%lld']\n",
            pTestName, std::accumulate(pDurationList, pDurationList + count, nn::TimeSpan()).GetMilliSeconds() / count);

        if ((count % 2) == 0)
        {
            NN_LOG("##teamcity[buildStatisticValue key='%s_Median(ms)' value='%lld']\n",
                pTestName, (pDurationList[count / 2 - 1] + pDurationList[count / 2]).GetMilliSeconds() / 2);
        }
        else
        {
            NN_LOG("##teamcity[buildStatisticValue key='%s_Median(ms)' value='%lld']\n",
                pTestName, pDurationList[count / 2].GetMilliSeconds() / 2);
        }
    }
};

TEST_F(EnsAcbaaBenchmark, GenarateCredentials)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    for (int i = 0; i < NN_ARRAY_SIZE(g_Credentials); i++)
    {
        nn::ens::AsyncContext context;

        nn::ens::GenerateCredential(&context, &g_Credentials[i], token.Get());

        context.GetEvent().Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
}

TEST_F(EnsAcbaaBenchmark, Message)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    // メッセージ送信 (0 -> 1)
    ASSERT_NO_FATAL_FAILURE(SendMessage(g_Credentials[0], g_Credentials[1], token.Get()));

    static nn::ens::MessageHeader s_Headers[2] = {};
    static char s_MetadataBuffers[NN_ARRAY_SIZE(s_Headers)][MetadataBufferSize];
    int count;
    int total;

    for (int i = 0; i < NN_ARRAY_SIZE(s_Headers); i++)
    {
        s_Headers[i].metadata.pBuffer = s_MetadataBuffers[i];
        s_Headers[i].metadata.bufferSize = sizeof (s_MetadataBuffers[i]);
    }

    // メッセージヘッダーリスト
    {
        nn::TimeSpan durations[10];

        for (int i = 0; i < NN_ARRAY_SIZE(durations); i++)
        {
            nn::ens::AsyncContext context;

            nn::os::Tick beginTick = nn::os::GetSystemTick();

            nn::ens::ReceiveMessageHeaderList(&context, &count, &total,
                s_Headers, NN_ARRAY_SIZE(s_Headers), 0, g_Credentials[1], token.Get());

            context.GetEvent().Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            durations[i] = (nn::os::GetSystemTick() - beginTick).ToTimeSpan();

            NN_LOG("nn::ens::ReceiveMessageHeaderList(%d) time = %lld ms\n", i, durations[i].GetMilliSeconds());
        }

        DumpStatisticValue(durations, NN_ARRAY_SIZE(durations), "ensReceiveMessageHeaderList");
    }

    ASSERT_EQ(count, 1);

    // メッセージボディ
    {
        nn::TimeSpan durations[10];

        for (int i = 0; i < NN_ARRAY_SIZE(durations); i++)
        {
            nn::ens::ReceiveBuffer body = {};
            static char s_Buffer[BodyBufferSize];

            body.pBuffer = s_Buffer;
            body.bufferSize = sizeof (s_Buffer);

            nn::ens::AsyncContext context;

            nn::os::Tick beginTick = nn::os::GetSystemTick();

            nn::ens::ReceiveMessageBody(&context, &body, s_Headers[0].id, g_Credentials[1], token.Get());

            context.GetEvent().Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            durations[i] = (nn::os::GetSystemTick() - beginTick).ToTimeSpan();

            NN_LOG("nn::ens::ReceiveMessageBody(%d) time = %lld ms\n", i, durations[i].GetMilliSeconds());
        }

        DumpStatisticValue(durations, NN_ARRAY_SIZE(durations), "ensReceiveMessageBody");
    }

    // 削除
    {
        nn::TimeSpan durations[10];

        for (int i = 0; i < NN_ARRAY_SIZE(durations); i++)
        {
            nn::ens::AsyncContext context;

            nn::os::Tick beginTick = nn::os::GetSystemTick();

            nn::ens::DeleteMessage(&context, &s_Headers[0].id, 1, "RECEIVED", g_Credentials[1], token.Get());

            context.GetEvent().Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            durations[i] = (nn::os::GetSystemTick() - beginTick).ToTimeSpan();

            NN_LOG("nn::ens::DeleteMessage(%d) time = %lld ms\n", i, durations[i].GetMilliSeconds());
        }

        DumpStatisticValue(durations, NN_ARRAY_SIZE(durations), "ensDeleteMessage");
    }
} // NOLINT(impl/function_size)

TEST_F(EnsAcbaaBenchmark, MyDesign)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    nn::ens::MyDesignAuthorId authorId = nn::ens::InvalidMyDesignAuthorId;

    // マイデザイン投稿
    ASSERT_NO_FATAL_FAILURE(PostMyDesign(&authorId, g_Credentials[0], token.Get()));

    static nn::ens::MyDesignHeader s_Headers[100] = {};
    static char s_MetadataBuffers[NN_ARRAY_SIZE(s_Headers)][MetadataBufferSize];
    int count;
    int total;

    for (int i = 0; i < NN_ARRAY_SIZE(s_Headers); i++)
    {
        s_Headers[i].metadata.pBuffer = s_MetadataBuffers[i];
        s_Headers[i].metadata.bufferSize = sizeof (s_MetadataBuffers[i]);
    }

    // マイデザインヘッダーリスト（自分）
    {
        nn::TimeSpan durations[10];

        for (int i = 0; i < NN_ARRAY_SIZE(durations); i++)
        {
            nn::ens::AsyncContext context;

            nn::os::Tick beginTick = nn::os::GetSystemTick();

            nn::ens::GetMyDesignHeaderList(&context, &count, &total,
                s_Headers, NN_ARRAY_SIZE(s_Headers), 0, authorId, g_Credentials[0], token.Get());

            context.GetEvent().Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            durations[i] = (nn::os::GetSystemTick() - beginTick).ToTimeSpan();

            NN_LOG("nn::ens::GetMyDesignHeaderList(%d) time = %lld ms\n", i, durations[i].GetMilliSeconds());
        }

        DumpStatisticValue(durations, NN_ARRAY_SIZE(durations), "ensGetMyDesignHeaderList(Self)");
    }

    ASSERT_EQ(count, 1);

    nn::ens::MyDesignId postedMyDesignId = s_Headers[0].id;

    const int Counts[] = {1, 25, 100};
    const int Offsets[] = {0, 100, 999};

    NN_SDK_ASSERT_LESS_EQUAL(Counts[NN_ARRAY_SIZE(Counts) - 1], static_cast<int>(NN_ARRAY_SIZE(s_Headers)));

    // マイデザインヘッダーリスト（検索）
    for (int c = 0; c < NN_ARRAY_SIZE(Counts); c++)
    {
        for (int f = 0; f < NN_ARRAY_SIZE(Offsets); f++)
        {
            nn::TimeSpan durations[10];

            for (int i = 0; i < NN_ARRAY_SIZE(durations); i++)
            {
                nn::ens::AsyncContext context;

                char searchQuery[nn::ens::SearchQueryLengthMax + 1] = {};
                nn::ens::SearchQueryBuilder builder(searchQuery, sizeof (searchQuery));

                ASSERT_TRUE(builder.Set("name", "ens"));

                nn::os::Tick beginTick = nn::os::GetSystemTick();

                nn::ens::GetMyDesignHeaderList(&context, &count, &total,
                    s_Headers, Counts[c], Offsets[f], builder.GetString(), g_Credentials[0], token.Get());

                context.GetEvent().Wait();

                NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

                durations[i] = (nn::os::GetSystemTick() - beginTick).ToTimeSpan();

                NN_LOG("nn::ens::GetMyDesignHeaderList(%d) time = %lld ms\n", i, durations[i].GetMilliSeconds());
                NN_LOG("    - Count = %d, Total = %d\n", count, total);
            }

            char name[128] = {};
            nn::util::SNPrintf(name, sizeof (name), "ensGetMyDesignHeaderList(Search_c%03d_f%03d)", Counts[c], Offsets[f]);

            DumpStatisticValue(durations, NN_ARRAY_SIZE(durations), name);
        }
    }
    // マイデザインヘッダーリスト（全体）
    for (int c = 0; c < NN_ARRAY_SIZE(Counts); c++)
    {
        for (int f = 0; f < NN_ARRAY_SIZE(Offsets); f++)
        {
            nn::TimeSpan durations[10];

            for (int i = 0; i < NN_ARRAY_SIZE(durations); i++)
            {
                nn::ens::AsyncContext context;

                nn::os::Tick beginTick = nn::os::GetSystemTick();

                nn::ens::GetMyDesignHeaderList(&context, &count, &total,
                    s_Headers, Counts[c], Offsets[f], g_Credentials[0], token.Get());

                context.GetEvent().Wait();

                NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

                durations[i] = (nn::os::GetSystemTick() - beginTick).ToTimeSpan();

                NN_LOG("nn::ens::GetMyDesignHeaderList(%d) time = %lld ms\n", i, durations[i].GetMilliSeconds());
                NN_LOG("    - Count = %d, Total = %d\n", count, total);
            }

            char name[128] = {};
            nn::util::SNPrintf(name, sizeof (name), "ensGetMyDesignHeaderList(Whole_c%03d_f%03d)", Counts[c], Offsets[f]);

            DumpStatisticValue(durations, NN_ARRAY_SIZE(durations), name);
        }
    }

    ASSERT_GE(count, 1);

    // マイデザインボディ
    {
        nn::TimeSpan durations[10];

        for (int i = 0; i < NN_ARRAY_SIZE(durations); i++)
        {
            nn::ens::ReceiveBuffer body = {};
            static char s_Buffer[BodyBufferSize];

            body.pBuffer = s_Buffer;
            body.bufferSize = sizeof (s_Buffer);

            nn::ens::AsyncContext context;

            nn::os::Tick beginTick = nn::os::GetSystemTick();

            nn::ens::GetMyDesignBody(&context, &body, s_Headers[0].id, g_Credentials[0], token.Get());

            context.GetEvent().Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            durations[i] = (nn::os::GetSystemTick() - beginTick).ToTimeSpan();

            NN_LOG("nn::ens::GetMyDesignBody(%d) time = %lld ms\n", i, durations[i].GetMilliSeconds());
        }

        DumpStatisticValue(durations, NN_ARRAY_SIZE(durations), "ensGetMyDesignBody");
    }

    // 削除
    {
        nn::TimeSpan durations[10];

        for (int i = 0; i < NN_ARRAY_SIZE(durations); i++)
        {
            nn::ens::AsyncContext context;

            nn::os::Tick beginTick = nn::os::GetSystemTick();

            nn::ens::DeleteMyDesign(&context, postedMyDesignId, g_Credentials[0], token.Get());

            context.GetEvent().Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            durations[i] = (nn::os::GetSystemTick() - beginTick).ToTimeSpan();

            NN_LOG("nn::ens::DeleteMyDesign(%d) time = %lld ms\n", i, durations[i].GetMilliSeconds());
        }

        DumpStatisticValue(durations, NN_ARRAY_SIZE(durations), "ensDeleteMyDesign");
    }
} // NOLINT(impl/function_size)
