﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nnt.h>
#include <nn/nn_Common.h>
#include <nn/account.h>
#include <nn/account/account_Result.h>

/**
 * @brief   ネットワークサービスアカウントの ID トークンを取得するマクロです。
 *
 * @param[in]   Variable    nnt::ens::NsaIdTokenGetter の変数名
 *
 * @details
 *  アカウントのインデックス = 0 で NsaIdTokenGetter のインスタンスを作成します。@n
 *  このマクロ使用後、 Variable.Get() で ID トークンが取得できます。
 *
 *  失敗時は GoogleTest フレームワークの ASSERT で止まります。
 */
#define NNT_ENS_ENSURE_NSA_ID_TOKEN(Variable) \
    nnt::ens::NsaIdTokenGetter Variable; \
    ASSERT_TRUE(Variable.IsValid())

namespace nnt { namespace ens {

/**
 * @brief   ネットワークサービスアカウントの ID トークンを取得します。
 *
 * @param[out]  pOutTokenActualSize 取得したトークンのサイズ
 * @param[out]  pOutToken           トークン格納先
 * @param[in]   tokenSize           pOutToken のサイズ
 * @param[in]   accountIndex        取得対象となるアカウントのインデックス
 *
 * @pre
 *  - pOutTokenActualSize != nullptr
 *  - pOutToken != nullptr
 *  - tokenSize >= nn::account::NetworkServiceAccountIdTokenLengthMax
 *
 * @details
 *  失敗時は GoogleTest フレームワークの ASSERT で止まります。
 */
void GetNsaIdToken(size_t* pOutTokenActualSize, char* pOutToken, size_t tokenSize, int accountIndex = 0) NN_NOEXCEPT;

/**
 * @brief   ネットワークサービスアカウントの ID を取得します。
 *
 * @param[out]  pOutNsaId       ネットワークサービスアカウントの ID
 * @param[in]   accountIndex    取得対象となるアカウントのインデックス
 *
 * @pre
 *  - pOutNsaId != nullptr
 *
 * @details
 *  失敗時は GoogleTest フレームワークの ASSERT で止まります。
 */
void GetNsaId(nn::account::NetworkServiceAccountId* pOutNsaId, int accountIndex = 0) NN_NOEXCEPT;

/**
 * @brief   ネットワークサービスアカウントの ID トークン取得をサポートするユーティリティクラス
 *
 * @details
 */
class NsaIdTokenGetter
{
public:
    /**
     * @brief   コンストラクタ
     *
     * @param[in]   accountIndex    取得対象となるアカウントのインデックス
     *
     * @details
     *  ネットワークサービスアカウントの ID トークンを取得します。
     *
     *  失敗時は GoogleTest フレームワークの ASSERT で止まります。
     */
    explicit NsaIdTokenGetter(int accountIndex = 0) NN_NOEXCEPT
        : m_TokenSize(0)
        , m_HasFatalFailure(false)
    {
        std::memset(m_Token, 0, sizeof (m_Token));
        nnt::ens::GetNsaIdToken(&m_TokenSize, m_Token, sizeof (m_Token), accountIndex);

        m_HasFatalFailure = ::testing::Test::HasFatalFailure();
    }

    /**
     * @brief   ネットワークサービスアカウントの ID トークンを取得します。
     *
     * @return  ネットワークサービスアカウントの ID トークン
     *
     * @details
     */
    const char* Get() const NN_NOEXCEPT
    {
        return m_Token;
    }

    /**
     * @brief   ネットワークサービスアカウントの ID トークンのサイズを取得します。
     *
     * @return  ネットワークサービスアカウントの ID トークンのサイズ
     *
     * @details
     */
    size_t GetSize() const NN_NOEXCEPT
    {
        return m_TokenSize;
    }

    /**
     * @brief   ネットワークサービスアカウントの ID トークンが有効かどうかを判定します。
     *
     * @return  ネットワークサービスアカウントの ID トークンが有効かどうか
     *
     * @details
     */
    bool IsValid() const NN_NOEXCEPT
    {
        return !m_HasFatalFailure;
    }

private:
    //
    char m_Token[nn::account::NetworkServiceAccountIdTokenLengthMax];
    size_t m_TokenSize;
    //
    bool m_HasFatalFailure;
};

}}
