﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "CecManager.h"
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/cec/cec_Api.h>
#include <nn/settings/system/settings_Tv.h>

namespace nns {

const int64_t PollingIntervalMilliSeconds = 1000;
const int ThreadStackSize = 16 * 1024;
NN_ALIGNAS(4096) uint8_t s_ThreadStack[ThreadStackSize];
volatile bool s_IsCommandListenerRunning;

void CecCommandListener(void* arg)
{
    CecManager* cec = reinterpret_cast<CecManager*>(arg);
    while (s_IsCommandListenerRunning)
    {
        nn::os::WaitSystemEvent(&cec->m_SystemEvent);
        nn::cec::GetBusEventType(&cec->m_BusEventType);
        NN_LOG("  --> CEC bus Event:[%d]\n", cec->m_BusEventType);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    }
}

NN_IMPLICIT CecManager::CecManager() NN_NOEXCEPT
:
m_IsInitialized(false),
m_BusEventType(nn::cec::BusEventType_Ignore)
{
    nn::settings::system::TvSettings settings;
    nn::settings::system::GetTvSettings(&settings);
    if (settings.flags.Test<nn::settings::system::TvFlag::AllowsCec>())
    {
        auto result = nn::cec::Initialize(&m_SystemEvent);
        m_IsInitialized = result.IsSuccess();
    }
    else
    {
        NN_LOG("Warning: CEC is disabled in settings parameter.\n");
    }
}

CecManager::~CecManager() NN_NOEXCEPT
{
    if (s_IsCommandListenerRunning)
    {
        s_IsCommandListenerRunning = false;
        nn::os::WaitThread(&m_CecThread);
        nn::os::DestroyThread(&m_CecThread);
        NN_LOG("Stop CEC command listener.\n");
    }
    if (m_IsInitialized)
    {
        nn::cec::Finalize(&m_SystemEvent);
    }
    m_IsInitialized = false;
}

bool CecManager::SetCommandListener() NN_NOEXCEPT
{
    if (m_IsInitialized)
    {
        if (!s_IsCommandListenerRunning)
        {
            // CECコマンドを監視する。
            nn::Result result = nn::os::CreateThread(
                &m_CecThread, CecCommandListener, reinterpret_cast<void*>(this),
                s_ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority);
            s_IsCommandListenerRunning = result.IsSuccess();
            if (s_IsCommandListenerRunning)
            {
                NN_LOG("Start CEC status listener.\n");
                nn::os::StartThread(&m_CecThread);
            }
            else
            {
                NN_LOG("Cannot create listener to listen to CEC command.\n");
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            return true;
        }
        else
        {
            NN_LOG("Warning: CEC command listener has been started already.\n");
        }
    }
    else
    {
        NN_LOG("Warning: CEC is disabled in settings parameter.\n");
    }
    return false;
}

bool CecManager::SendOneTouchPlay() NN_NOEXCEPT
{
    if (m_IsInitialized)
    {
        auto result = nn::cec::PerformOneTouchPlay();
        return result.IsSuccess();
    }
    NN_LOG("Warning: Cannot send one touch play because CEC is disabled.\n");
    return false;
}

bool CecManager::SendTvStandby() NN_NOEXCEPT
{
    if (m_IsInitialized)
    {
        auto result = nn::cec::PerformGoStandby(true);
        return result.IsSuccess();
    }
    NN_LOG("Warning: Cannot send TV standby because CEC is disabled.\n");
    return false;
}

bool CecManager::SendTvVolumeUp() NN_NOEXCEPT
{
    if (m_IsInitialized)
    {
        auto result = nn::cec::PerformSendRemoteControlCommand(true, nn::cec::RemoteControlVolumeUp);
        if (result.IsSuccess())
        {
            result = nn::cec::PerformSendRemoteControlCommand(false, 0);
            return result.IsSuccess();
        }
        NN_LOG("Warning: Cannot send CEC release command.\n");
        return false;
    }
    NN_LOG("Warning: Cannot send volume up because CEC is disabled.\n");
    return false;
}

bool CecManager::SendTvVolumeDown() NN_NOEXCEPT
{
    if (m_IsInitialized)
    {
        auto result = nn::cec::PerformSendRemoteControlCommand(true, nn::cec::RemoteControlVolumeDown);
        if (result.IsSuccess())
        {
            result = nn::cec::PerformSendRemoteControlCommand(false, 0);
            return result.IsSuccess();
        }
        NN_LOG("Warning: Cannot send CEC release command.\n");
        return false;
    }
    NN_LOG("Warning: Cannot send volume down because CEC is disabled.\n");
    return false;
}

bool CecManager::SendTvVolumeMute() NN_NOEXCEPT
{
    if (m_IsInitialized)
    {
        auto result = nn::cec::PerformSendRemoteControlCommand(true, nn::cec::RemoteControlVolumeMute);
        if (result.IsSuccess())
        {
            result = nn::cec::PerformSendRemoteControlCommand(false, 0);
            return result.IsSuccess();
        }
        NN_LOG("Warning: Cannot send CEC release command.\n");
        return false;
    }
    NN_LOG("Warning: Cannot send volume mute because CEC is disabled.\n");
    return false;
}

nn::cec::BusEventType CecManager::GetStoredBusEventType() NN_NOEXCEPT
{
    return m_BusEventType;
}

}
