﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "GfxJpegViewer.h"
#include <cstdlib>
#include <cstring>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/os.h>
#include <nn/fs.h>
#include <nn/init.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nn/image/image_JpegDecoder.h>
#include <nn/util/util_ScopeExit.h>

#if NN_GFX_IS_TARGET_NVN
    #include <nvn/nvn.h>
    #include <nvn/nvn_FuncPtrInline.h>
#endif

#include <Common/SetupGraphics.h>

namespace nns {

namespace {

// 頂点
// (x, y, z; u, v) を頂点数分並べる
const float Vertices[] =
{
    -1.0f, -1.0f, 0.0f, 0.0f, 0.0f,
    -1.0f, +1.0f, 0.0f, 0.0f, 1.0f,
    +1.0f, -1.0f, 0.0f, 1.0f, 0.0f,
    +1.0f, +1.0f, 0.0f, 1.0f, 1.0f,
};

// インデクス
// (i0, i1, i2) をポリゴン数分並べる
const int Indices[] =
{
    0, 2, 1,
    2, 3, 1,
};

int g_BufferDescriptorBaseIndex = 0;
int g_TextureDescriptorBaseIndex = 0;
int g_SamplerDescriptorBaseIndex = 0;

//-----------------------------------------------------------------------------
// メモリ

nn::util::BytePtr g_pMemoryHeap( NULL );
nn::util::BytePtr g_pMemory( NULL );

enum MemoryPoolType
{
    MemoryPoolType_CpuUncached_GpuCached,
    MemoryPoolType_CpuInvisible_GpuCached_Compressible,

    MemoryPoolType_End
};

const size_t MemoryPoolSize[ MemoryPoolType_End ] =
{
    64 * 1024 * 1024,
    64 * 1024 * 1024
};

void* g_pPoolMemory[ MemoryPoolType_End ] = {};
ptrdiff_t g_MemoryPoolOffset[ MemoryPoolType_End ] = {};

size_t g_MaxScratchMemory = 0;
ptrdiff_t g_ScratchMemoryOffset = 0;

//-----------------------------------------------------------------------------
// 画像ファイルのパス
const char* g_ImageSourcePath = "Contents:/Image.jpg";
// テクスチャの幅の最小単位[px]。読み込んだ画像の幅はこの値の整数倍までパディングが詰められる。
const int g_ImageLineAlignment = 1;

uint8_t* g_pImageData = NULL;
size_t g_ImageDataSize = 0;
nn::image::Dimension g_ImageDimension = {};
int g_ImageStride = 0;
bool g_IsBitmap = false;

const int RenderWidth = 1280;
const int RenderHeight = 720;

//-----------------------------------------------------------------------------

nn::gfx::Device* g_pDevice;
nns::ConfigSwitcher* g_pConfigSwitcher;

void* ReadResource( const char* filename )
{
    nn::Result result;
    nn::fs::FileHandle hFile;

    int64_t fileSize = 0;
    result = nn::fs::OpenFile( &hFile, filename, nn::fs::OpenMode_Read );
    NN_ASSERT( result.IsSuccess() );

    result = nn::fs::GetFileSize( &fileSize, hFile );
    NN_ASSERT( result.IsSuccess() );

    nn::util::BinaryFileHeader fileHeader;
    size_t readSize;
    result = nn::fs::ReadFile( &readSize, hFile, 0, &fileHeader, sizeof( nn::util::BinaryFileHeader ) );
    NN_ASSERT( result.IsSuccess() );
    NN_ASSERT( readSize == sizeof( nn::util::BinaryFileHeader ) );
    size_t alignment = fileHeader.GetAlignment();

    g_pMemory.AlignUp( alignment );
    void* pBuffer = g_pMemory.Get();
    result = nn::fs::ReadFile( &readSize, hFile, 0, pBuffer, static_cast< size_t >( fileSize ) );
    NN_ASSERT( result.IsSuccess() );
    NN_ASSERT( readSize == static_cast< size_t >( fileSize ) );
    g_pMemory.Advance( static_cast< ptrdiff_t >( fileSize ) );

    nn::fs::CloseFile( hFile );

    return pBuffer;
}

void SetImageFormat()
{
    // 末尾の文字だけチェックしてフォーマットを判定。
    int length = strnlen(g_ImageSourcePath, 512);
    g_IsBitmap = ('g' != g_ImageSourcePath[length - 1]);
}

void ReadBitmapImage(uint8_t* fileData, size_t dataSize)
{
    uint32_t bmpSize = *reinterpret_cast<uint32_t*>(&fileData[2]);
    uint32_t bmpOffset = *reinterpret_cast<uint32_t*>(&fileData[10]);
    g_ImageDataSize = static_cast<int64_t>(bmpSize - bmpOffset);
    g_ImageDimension.width = *reinterpret_cast<uint32_t*>(&fileData[18]);
    g_ImageDimension.height = *reinterpret_cast<uint32_t*>(&fileData[22]);
    g_ImageStride = 4 * g_ImageLineAlignment * ((g_ImageDimension.width + g_ImageLineAlignment - 1) / g_ImageLineAlignment);
    g_ImageDataSize = bmpSize;
    g_pImageData = reinterpret_cast<uint8_t*>(malloc(g_ImageDataSize));
    NN_ASSERT_NOT_NULL(g_pImageData);
    memcpy(g_pImageData, &fileData[bmpOffset], g_ImageDataSize);
}

void ReadJpegImage(void* pCompressed, size_t compressedSize)
{
    nn::image::JpegStatus result;
    nn::image::JpegDecoder decoder;
    decoder.SetImageData(pCompressed, compressedSize);
    result = decoder.Analyze();
    NN_ASSERT(result == nn::image::JpegStatus_Ok);
    size_t workSize = decoder.GetAnalyzedWorkBufferSize();
    void* workBuffer = malloc(workSize);
    NN_ASSERT_NOT_NULL(workBuffer);
    g_ImageDimension = decoder.GetAnalyzedDimension();
    g_ImageStride = 4 * g_ImageLineAlignment * ((g_ImageDimension.width + g_ImageLineAlignment - 1) / g_ImageLineAlignment);
    g_ImageDataSize = g_ImageStride * g_ImageDimension.height;
    g_pImageData = reinterpret_cast<uint8_t*>(malloc(g_ImageDataSize));
    memset(g_pImageData, 0, g_ImageDataSize);
    result = decoder.Decode(g_pImageData, g_ImageDataSize, g_ImageLineAlignment, workBuffer, workSize);
    NN_ASSERT(result == nn::image::JpegStatus_Ok);
    free(workBuffer);
    workBuffer = NULL;
    workSize = 0;
}

void InitializeImage()
{
    // ファイルから読み込み
    SetImageFormat();
    NN_LOG("Load %s :[%s]\n", g_IsBitmap ? "BitmapImage" : "JpegImage", g_ImageSourcePath);
    void* pCompressed = NULL;
    size_t compressedSize = 0;
    {
        nn::Result result;
        nn::fs::FileHandle file = {};
        result = nn::fs::OpenFile(&file, g_ImageSourcePath, nn::fs::OpenMode_Read);
        NN_ASSERT(result.IsSuccess());
        int64_t fileSize = 0;
        result = nn::fs::GetFileSize(&fileSize, file);
        NN_ASSERT(result.IsSuccess());
        compressedSize = static_cast<size_t>(fileSize);
        pCompressed = malloc(compressedSize);
        NN_ASSERT_NOT_NULL(pCompressed);
        result = nn::fs::ReadFile(file, 0, pCompressed, compressedSize);
        NN_ASSERT(result.IsSuccess());
        nn::fs::CloseFile(file);
    }
    // 画像を展開
    if (g_IsBitmap)
    {
        ReadBitmapImage(reinterpret_cast<uint8_t*>(pCompressed), compressedSize);
    }
    else
    {
        ReadJpegImage(pCompressed, compressedSize);
    }

    free(pCompressed);
    NN_LOG("Image  W%d x H%d\n", g_ImageDimension.width, g_ImageDimension.height);
}
void FinalizeImage()
{
    free(g_pImageData);
    g_pImageData = NULL;
    g_ImageDataSize = 0;
}

nn::vi::Display* g_pDisplay;
nn::vi::Layer* g_pLayer;
nn::gfx::MemoryPool g_MemoryPool[ MemoryPoolType_End ];
void InitializeMemoryPool()
{
    const int MemoryPoolProperty[ MemoryPoolType_End ] =
    {
        nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached,
        nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached
            | nn::gfx::MemoryPoolProperty_Compressible
    };

    nn::gfx::MemoryPool::InfoType info;
    for( int idxMemoryPool = 0; idxMemoryPool < MemoryPoolType_End; ++idxMemoryPool )
    {
        info.SetDefault();
        info.SetMemoryPoolProperty( MemoryPoolProperty[ idxMemoryPool ] );
        size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment( g_pDevice, info );
        size_t granularity = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity( g_pDevice, info );
        g_pPoolMemory[ idxMemoryPool ] = malloc( MemoryPoolSize[ idxMemoryPool ] );
        void* pPoolMemoryAligned = nn::util::BytePtr(
            g_pPoolMemory[ idxMemoryPool ] ).AlignUp( alignment ).Get();
        size_t memoryPoolSizeAligned = nn::util::align_down( MemoryPoolSize[ idxMemoryPool ], granularity );
        info.SetPoolMemory( pPoolMemoryAligned, memoryPoolSizeAligned );
        g_MemoryPool[ idxMemoryPool ].Initialize( g_pDevice, info );
        g_MemoryPoolOffset[ idxMemoryPool ] = 0;
    }
}

// スワップチェーンを初期化
nn::gfx::SwapChain g_SwapChain;
void InitializeSwapChain()
{
    nn::gfx::SwapChain::InfoType info;

    info.SetDefault();
    info.SetLayer( g_pLayer );
    info.SetWidth( RenderWidth );
    info.SetHeight( RenderHeight );
    info.SetFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb );
    info.SetBufferCount( 2 );
    if( NN_STATIC_CONDITION( nn::gfx::SwapChain::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ];
        size_t size = g_SwapChain.CalculateScanBufferSize( g_pDevice, info );
        offset = nn::util::align_up( offset, nn::gfx::SwapChain::GetScanBufferAlignment( g_pDevice, info ) );
        g_SwapChain.Initialize( g_pDevice, info,
            &g_MemoryPool[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ], offset, size );
        offset += size;
    }
    else
    {
        g_SwapChain.Initialize( g_pDevice, info, NULL, 0, 0 );
    }
}

// キューを初期化
nn::gfx::Queue g_Queue;
void InitializeQueue()
{
    nn::gfx::Queue::InfoType info;
    info.SetDefault();
    info.SetCapability( nn::gfx::QueueCapability_Graphics );
    g_Queue.Initialize( g_pDevice, info );
}

// コマンドバッファを初期化
nn::gfx::CommandBuffer g_CommandBuffer;
void InitializeCommandBuffer()
{
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability( nn::gfx::QueueCapability_Graphics );
    info.SetCommandBufferType( nn::gfx::CommandBufferType_Direct );
    g_CommandBuffer.Initialize( g_pDevice, info );
}

// ビューポートシザーを初期化
nn::gfx::ViewportScissorState g_ViewportScissor;
void InitializeViewport()
{
    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled( true );
    nn::gfx::ViewportStateInfo viewportInfo;
    {
        viewportInfo.SetDefault();
        viewportInfo.SetWidth( static_cast< float >( RenderWidth ) );
        viewportInfo.SetHeight( static_cast< float >( RenderHeight ) );
    }
    nn::gfx::ScissorStateInfo scissorInfo;
    {
        scissorInfo.SetDefault();
        scissorInfo.SetWidth( RenderWidth );
        scissorInfo.SetHeight( RenderHeight );
    }
    info.SetViewportStateInfoArray( &viewportInfo, 1 );
    info.SetScissorStateInfoArray( &scissorInfo, 1 );
    g_ViewportScissor.Initialize( g_pDevice, info );
}

// Initialize the rasterizer state
nn::gfx::RasterizerState g_RasterizerState;
static void InitializeRasterizerState()
{
    nn::gfx::RasterizerState::InfoType info;
    info.SetDefault();
    info.SetCullMode( nn::gfx::CullMode_None );
    info.SetPrimitiveTopologyType( nn::gfx::PrimitiveTopologyType_Triangle );
    info.SetScissorEnabled( true );
    info.SetDepthClipEnabled( false );
    g_RasterizerState.Initialize( g_pDevice, info );
}

// Initialize the blend state
nn::gfx::BlendState g_BlendState;
static void InitializeBlendState()
{
    nn::gfx::BlendState::InfoType info;
    info.SetDefault();
    nn::gfx::BlendTargetStateInfo targetInfo;
    {
        targetInfo.SetDefault();
    };
    info.SetBlendTargetStateInfoArray( &targetInfo, 1 );
    size_t size = nn::gfx::BlendState::GetRequiredMemorySize( info );
    g_pMemory.AlignUp( nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
    g_BlendState.SetMemory( g_pMemory.Get(), size );
    g_pMemory.Advance( size );
    g_BlendState.Initialize( g_pDevice, info );
}

// Initialize the depth stencil state
nn::gfx::DepthStencilState g_DepthStencilState;
static void InitializeDepthStencilState()
{
    nn::gfx::DepthStencilState::InfoType info;
    info.SetDefault();
    info.SetDepthTestEnabled( false );
    info.SetDepthWriteEnabled( false );
    g_DepthStencilState.Initialize( g_pDevice, info );
}

// Initialize the vertex state
nn::gfx::VertexState g_VertexState;
static void InitializeVertexState()
{
    nn::gfx::VertexState::InfoType info;
    info.SetDefault();
    uint16_t stride = sizeof( float ) * 5;
    nn::gfx::VertexAttributeStateInfo attribs[ 2 ];
    {
        // i_Position
        attribs[ 0 ].SetDefault();
        attribs[ 0 ].SetBufferIndex( 0 );
        attribs[ 0 ].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
        attribs[ 0 ].SetOffset( 0 );
        attribs[ 0 ].SetShaderSlot( 0 );
    }
    {
        // i_TexCoord
        attribs[ 1 ].SetDefault();
        attribs[ 1 ].SetBufferIndex( 0 );
        attribs[ 1 ].SetFormat( nn::gfx::AttributeFormat_32_32_Float );
        attribs[ 1 ].SetOffset( sizeof( float ) * 3 );
        attribs[ 1 ].SetShaderSlot( 1 );
    }
    nn::gfx::VertexBufferStateInfo buffer;
    {
        buffer.SetDefault();
        buffer.SetStride( stride );
    }
    info.SetVertexAttributeStateInfoArray( attribs, 2 );
    info.SetVertexBufferStateInfoArray( &buffer, 1 );
    size_t size = nn::gfx::VertexState::GetRequiredMemorySize( info );
    g_pMemory.AlignUp( nn::gfx::VertexState::RequiredMemoryInfo_Alignment );
    g_VertexState.SetMemory( g_pMemory.Get(), size );
    g_pMemory.Advance( size );
    g_VertexState.Initialize( g_pDevice, info, NULL );
}

// 頂点用のバッファを初期化
nn::gfx::Buffer g_VertexBuffer;
void InitializeVertexBuffer()
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize( sizeof( Vertices ) );
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_VertexBuffer );
    if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        offset = nn::util::align_up( offset, nn::gfx::Buffer::GetBufferAlignment( g_pDevice, info ) );
        g_VertexBuffer.Initialize( g_pDevice, info,
            &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, info.GetSize() );
        offset += info.GetSize();
    }
    else
    {
        g_VertexBuffer.Initialize( g_pDevice, info, NULL, 0, 0 );
    }

    void* pMapped = g_VertexBuffer.Map();
    memcpy( pMapped, Vertices, info.GetSize() );
    g_VertexBuffer.Unmap();
}

// インデクス用のバッファを初期化
nn::gfx::Buffer g_IndexBuffer;
void InitializeIndexBuffer()
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize( sizeof( Indices ) );
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_IndexBuffer );
    if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        offset = nn::util::align_up( offset, nn::gfx::Buffer::GetBufferAlignment( g_pDevice, info ) );
        g_IndexBuffer.Initialize( g_pDevice, info,
            &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, info.GetSize() );
        offset += info.GetSize();
    }
    else
    {
        g_IndexBuffer.Initialize( g_pDevice, info, NULL, 0, 0 );
    }

    void* pMapped = g_IndexBuffer.Map();
    memcpy( pMapped, Indices, info.GetSize() );
    g_IndexBuffer.Unmap();
}

// 定数バッファ用のバッファを初期化
nn::gfx::Buffer g_ConstantBuffer;
void InitializeConstantBuffer()
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize( sizeof( float ) * 4 );
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_ConstantBuffer );
    if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        offset = nn::util::align_up( offset, nn::gfx::Buffer::GetBufferAlignment( g_pDevice, info ) );
        g_ConstantBuffer.Initialize( g_pDevice, info,
            &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, info.GetSize() );
        offset += info.GetSize();
    }
    else
    {
        g_ConstantBuffer.Initialize( g_pDevice, info, NULL, 0, 0 );
    }
}

// サンプラを初期化
nn::gfx::Sampler g_Sampler;
void InitializeSampler()
{
    nn::gfx::Sampler::InfoType info;
    info.SetDefault();
    info.SetFilterMode( nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint );
    info.SetAddressU( nn::gfx::TextureAddressMode_Mirror );
    info.SetAddressV( nn::gfx::TextureAddressMode_Mirror );
    info.SetAddressW( nn::gfx::TextureAddressMode_Mirror );
    g_Sampler.Initialize( g_pDevice, info );
}

nn::gfx::DescriptorPool g_BufferDescriptorPool;
void InitializeBufferDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_BufferView );
    info.SetSlotCount( g_BufferDescriptorBaseIndex + 1 );
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( g_pDevice, info );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( g_pDevice, info ) );
    g_BufferDescriptorPool.Initialize( g_pDevice, info,
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, size );
    offset += size;
}

nn::gfx::DescriptorPool g_SamplerDescriptorPool;
void InitializeSamplerDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_Sampler );
    info.SetSlotCount( g_SamplerDescriptorBaseIndex + 1 );
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( g_pDevice, info );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( g_pDevice, info ) );
    g_SamplerDescriptorPool.Initialize( g_pDevice, info,
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, size );
    offset += size;
}

nn::gfx::DescriptorPool g_TextureDescriptorPool;
void InitializeTextureDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_TextureView );
    info.SetSlotCount( g_TextureDescriptorBaseIndex + 2 );
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( g_pDevice, info );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( g_pDevice, info ) );
    g_TextureDescriptorPool.Initialize( g_pDevice, info,
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, size );
    offset += size;
}

nn::gfx::Texture g_Texture;
void InitializeTexture()
{
    nn::gfx::Texture::InfoType info;

    // Create temporary linear buffer
    nn::gfx::Buffer tempBuffer;
    size_t tempBufferSize = static_cast<size_t>(g_ImageStride * g_ImageDimension.height);
    {
        nn::gfx::MemoryPool& pool = g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ];
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetGpuAccessFlags( nn::gfx::GpuAccess_Read );
        info.SetSize( tempBufferSize );
        size_t alignment = nn::gfx::Buffer::GetBufferAlignment(g_pDevice, info);
        offset = nn::util::align_up(offset, alignment);
        tempBuffer.Initialize( g_pDevice, info, &pool, offset, tempBufferSize );
        offset += tempBufferSize;
    }

    // Configure the optimal texture
    info.SetDefault();
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_Texture );
    info.SetWidth( g_ImageStride / 4 );
    info.SetHeight( g_ImageDimension.height );
    info.SetImageStorageDimension( nn::gfx::ImageStorageDimension_2d );
    info.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb );
    info.SetMipCount( 1 );
    info.SetTileMode( nn::gfx::TileMode_Linear );

    // Allocate space for the optimal texture
    {
        nn::gfx::MemoryPool& pool = g_MemoryPool[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ];
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ];
        offset = nn::util::align_up( offset,
            nn::gfx::Texture::CalculateMipDataAlignment( g_pDevice, info ) );

        size_t size = nn::gfx::Texture::CalculateMipDataSize( g_pDevice, info );
        g_Texture.Initialize( g_pDevice, info, &pool, offset, size );
        offset += size;
    }

    // Generate the linear texture w/ CPU writes
    uint8_t* pixels = tempBuffer.Map< uint8_t >();
    if (g_IsBitmap)
    {
        int bmpIndex = 0;
        for (int index = 0; index < tempBufferSize; index += 4)
        {
            pixels[index + 0] = g_pImageData[bmpIndex + 2];
            pixels[index + 1] = g_pImageData[bmpIndex + 1];
            pixels[index + 2] = g_pImageData[bmpIndex + 0];
            pixels[index + 3] = 0xFF;
            bmpIndex += 3;
        }
    }
    else
    {
        for(int y = 0; y < g_ImageDimension.height; y++)
        {
            int revY = g_ImageDimension.height - 1 - y;
            memcpy(pixels + y * g_ImageStride, g_pImageData + revY * g_ImageStride, g_ImageStride);
        }
    }
    g_pConfigSwitcher->ReferImagePixel(pixels, g_ImageDimension.width, g_ImageDimension.height);
    tempBuffer.Unmap();

    // Use GPU to tile texture
    {
        const size_t CommandMemorySize = 1024;
        const size_t ControlMemorySize = 256;
        g_CommandBuffer.Reset();
        nn::gfx::MemoryPool& pool = g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ];
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        offset = nn::util::align_up( offset,
            nn::gfx::CommandBuffer::GetCommandMemoryAlignment( g_pDevice ) );
        g_CommandBuffer.AddCommandMemory( &pool, offset, CommandMemorySize );
        offset += CommandMemorySize;
        g_pMemory.AlignUp( nn::gfx::CommandBuffer::GetControlMemoryAlignment( g_pDevice ) );
        g_CommandBuffer.AddControlMemory( g_pMemory.Get(), ControlMemorySize );
        g_pMemory.Advance( ControlMemorySize );
        g_CommandBuffer.Begin();

        nn::gfx::TextureCopyRegion dstRegion;
        dstRegion.SetDefault();
        dstRegion.SetWidth( g_ImageStride / 4 );
        dstRegion.SetHeight( g_ImageDimension.height );
        dstRegion.SetDepth( 1 );
        g_CommandBuffer.CopyBufferToImage( &g_Texture, dstRegion, &tempBuffer, 0 );

        g_CommandBuffer.End();
        g_Queue.ExecuteCommand( &g_CommandBuffer, NULL );
        g_Queue.Sync();
    }

    tempBuffer.Finalize( g_pDevice );
}

// テクスチャビューを初期化
nn::gfx::TextureView g_TextureView;
void InitializeTextureBufferView()
{
    nn::gfx::TextureView::InfoType info;
    info.SetDefault();
    info.SetImageDimension( nn::gfx::ImageDimension_2d );
    info.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb );
    info.SetTexturePtr( &g_Texture );
    info.SetChannelMapping( nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Green,
        nn::gfx::ChannelMapping_Blue, nn::gfx::ChannelMapping_Alpha );
    g_TextureView.Initialize( g_pDevice, info );
}

nn::gfx::ResShaderFile* g_pResShaderFile;
void InitializeResShaderFile()
{
    void* pResource = ReadResource( "Contents:/GfxJpegViewerShader.bnsh" );
    g_pResShaderFile = nn::gfx::ResShaderFile::ResCast( pResource );
    nn::gfx::ResShaderContainer* pContainer = g_pResShaderFile->GetShaderContainer();
    NN_ASSERT_NOT_NULL( pContainer );
    pContainer->Initialize( g_pDevice );

    for( int idxVariation = 0, variationCount = pContainer->GetShaderVariationCount();
        idxVariation < variationCount; ++idxVariation )
    {
        nn::gfx::ResShaderProgram* pProgram = pContainer->GetResShaderProgram( idxVariation );
        nn::gfx::ShaderInitializeResult result = pProgram->Initialize( g_pDevice );
        NN_ASSERT_EQUAL( result, nn::gfx::ShaderInitializeResult_Success );
        NN_UNUSED( result );
    }

#if NN_GFX_IS_TARGET_NVN
    g_MaxScratchMemory = nn::gfx::NvnGetMaxRecommendedScratchMemorySize( g_pDevice, &g_pResShaderFile, 1 );
    int scratchMemoryAlignment;
    nvnDeviceGetInteger( g_pDevice->ToData()->pNvnDevice,
        NVN_DEVICE_INFO_SHADER_SCRATCH_MEMORY_ALIGNMENT, &scratchMemoryAlignment );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ];
    offset = nn::util::align_up( offset, scratchMemoryAlignment );
    g_ScratchMemoryOffset = offset;
    offset += g_MaxScratchMemory;
#endif
    NN_UNUSED( g_MaxScratchMemory );
    NN_UNUSED( g_ScratchMemoryOffset );
}

void FinalizeResShaderFile()
{
    auto pContainer = g_pResShaderFile->GetShaderContainer();
    NN_ASSERT_NOT_NULL( pContainer );

    for( int idxVariation = 0, variationCount = pContainer->GetShaderVariationCount();
        idxVariation < variationCount; ++idxVariation )
    {
        nn::gfx::ResShaderProgram* pProgram = pContainer->GetResShaderProgram( idxVariation );
        pProgram->Finalize( g_pDevice );
    }

    pContainer->Finalize( g_pDevice );
}

void InitializeGfxObjects()
{
    InitializeMemoryPool();
    InitializeSwapChain();

    // メモリ
    g_pMemoryHeap.Reset( malloc( 1024 * 1024 ) );
    g_pMemory = g_pMemoryHeap;

    // 画像の読み込み
    InitializeImage();

    //InitializeDevice();

#if NN_GFX_IS_TARGET_NVN
    nn::gfx::Device::DataType& deviceData = nn::gfx::AccessorToData( *g_pDevice );
    nvnDeviceGetInteger( deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &g_TextureDescriptorBaseIndex );
    nvnDeviceGetInteger( deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &g_SamplerDescriptorBaseIndex );
#endif

    InitializeQueue();

    InitializeCommandBuffer();
    InitializeViewport();

    InitializeRasterizerState();
    InitializeBlendState();
    InitializeDepthStencilState();
    InitializeVertexState();

    InitializeVertexBuffer();
    InitializeIndexBuffer();
    InitializeConstantBuffer();
    InitializeSampler();

    InitializeBufferDescriptorPool();
    InitializeSamplerDescriptorPool();
    InitializeTextureDescriptorPool();

    InitializeTexture();
    InitializeTextureBufferView();
    InitializeResShaderFile();

    NN_ASSERT( g_pMemoryHeap.Distance( g_pMemory.Get() ) < 1024 * 1024 );
    for( int idxMemoryPool = 0; idxMemoryPool < MemoryPoolType_End; ++idxMemoryPool )
    {
        NN_ASSERT( static_cast< size_t >( g_MemoryPoolOffset[ idxMemoryPool ] )
            < MemoryPoolSize[ idxMemoryPool ] );
    }
}

void MakeCommand()
{
    nn::gfx::GpuAddress gpuAddress;

    g_CommandBuffer.Reset();
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::CommandBuffer::GetCommandMemoryAlignment( g_pDevice ) );
    // ワンタイムのコマンドバッファには nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuUncached のメモリプールを使うことがより適しています
    g_CommandBuffer.AddCommandMemory(
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, 1024 * 1024 );
    g_pMemory.AlignUp( 256 );
    g_CommandBuffer.AddControlMemory( g_pMemory.Get(), 256 );
    g_CommandBuffer.Begin();
    {
        nn::gfx::ColorTargetView* pTarget = g_SwapChain.AcquireNextScanBufferView();

        g_VertexBuffer.GetGpuAddress( &gpuAddress );
        g_CommandBuffer.SetVertexBuffer( 0, gpuAddress, sizeof( float ) * 5, sizeof( Vertices ) );

        g_CommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_ShaderCode | nn::gfx::GpuAccess_Descriptor );

        g_CommandBuffer.SetDescriptorPool( &g_BufferDescriptorPool );
        g_CommandBuffer.SetDescriptorPool( &g_TextureDescriptorPool );
        g_CommandBuffer.SetDescriptorPool( &g_SamplerDescriptorPool );

        g_CommandBuffer.ClearColor( pTarget, 0.3f, 0.1f, 0.1f, 1.0f, NULL );

#if NN_GFX_IS_TARGET_NVN
        nvnCommandBufferSetShaderScratchMemory( g_CommandBuffer.ToData()->pNvnCommandBuffer,
            g_MemoryPool[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ].ToData()->pNvnMemoryPool,
            g_ScratchMemoryOffset, g_MaxScratchMemory );
#endif

        nn::gfx::ResShaderContainer* pContainer = g_pResShaderFile->GetShaderContainer();
        int variation = 0;
        nn::gfx::ResShaderProgram* pProgram = pContainer->GetResShaderProgram( variation );
        nn::gfx::Shader* pShader = pProgram->GetShader();
        g_CommandBuffer.SetShader( pShader, nn::gfx::ShaderStageBit_All );

        g_CommandBuffer.SetRenderTargets( 1, &pTarget, NULL );

        g_CommandBuffer.SetViewportScissorState( &g_ViewportScissor );

        g_CommandBuffer.SetRasterizerState( &g_RasterizerState );
        g_CommandBuffer.SetBlendState( &g_BlendState );
        g_CommandBuffer.SetDepthStencilState( &g_DepthStencilState );
        g_CommandBuffer.SetVertexState( &g_VertexState );

        g_CommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer
            | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer );

        nn::gfx::DescriptorSlot constantBufferDescriptor;
        nn::gfx::DescriptorSlot textureDescriptor;
        nn::gfx::DescriptorSlot samplerDescriptor;
        g_BufferDescriptorPool.GetDescriptorSlot( &constantBufferDescriptor, g_BufferDescriptorBaseIndex );
        g_TextureDescriptorPool.GetDescriptorSlot( &textureDescriptor, g_TextureDescriptorBaseIndex );
        g_SamplerDescriptorPool.GetDescriptorSlot( &samplerDescriptor, g_SamplerDescriptorBaseIndex );

        int slotMat = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,
            nn::gfx::ShaderInterfaceType_ConstantBuffer, "Mat" );
        int slotTex = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,
            nn::gfx::ShaderInterfaceType_Sampler, "tex" );
        if( slotMat >= 0 )
        {
            g_CommandBuffer.SetConstantBuffer( slotMat,
                nn::gfx::ShaderStage_Pixel, constantBufferDescriptor );
        }
        if( slotTex >= 0 )
        {
            g_CommandBuffer.SetTextureAndSampler( slotTex,
                nn::gfx::ShaderStage_Pixel, textureDescriptor, samplerDescriptor );
        }

        g_IndexBuffer.GetGpuAddress( &gpuAddress );
        g_CommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList,
            nn::gfx::IndexFormat_Uint32, gpuAddress, sizeof(Indices) / sizeof(Indices[0]), 0 );
    }
    g_pConfigSwitcher->Draw(&g_CommandBuffer);
    g_CommandBuffer.End();
}

void FinalizeGfxObjects()
{
    // 各オブジェクトを破棄
    FinalizeResShaderFile();

    g_Sampler.Finalize( g_pDevice );
    g_TextureView.Finalize( g_pDevice );
    g_Texture.Finalize( g_pDevice );
    g_ConstantBuffer.Finalize( g_pDevice );
    g_IndexBuffer.Finalize( g_pDevice );
    g_VertexBuffer.Finalize( g_pDevice );

    g_RasterizerState.Finalize( g_pDevice );
    g_BlendState.Finalize( g_pDevice );
    g_DepthStencilState.Finalize( g_pDevice );
    g_VertexState.Finalize( g_pDevice );

    g_BufferDescriptorPool.Finalize( g_pDevice );
    g_TextureDescriptorPool.Finalize( g_pDevice );
    g_SamplerDescriptorPool.Finalize( g_pDevice );

    g_ViewportScissor.Finalize( g_pDevice );
    g_CommandBuffer.Finalize( g_pDevice );
    g_SwapChain.Finalize( g_pDevice );
    g_Queue.Finalize( g_pDevice );
    for( int idxMemoryPool = 0; idxMemoryPool < MemoryPoolType_End; ++idxMemoryPool )
    {
        g_MemoryPoolOffset[idxMemoryPool] = 0;
        g_MemoryPool[ idxMemoryPool ].Finalize( g_pDevice );
        free( g_pPoolMemory[ idxMemoryPool ] );
    }
    //g_Device.Finalize();

    // 画像を解放
    FinalizeImage();

    free( g_pMemoryHeap.Get() );
}

void* Allocate( size_t size )
{
    return malloc( size );
}

void Deallocate( void* ptr, size_t )
{
    free( ptr );
}

}// anonymous namespace


void ShowImage(const char* pImageSourcePath, SetupGraphics* pGfxSetup, ConfigSwitcher* pSwitcher)
{
    g_ImageSourcePath = (nullptr != pImageSourcePath) ? pImageSourcePath : g_ImageSourcePath;
    g_pLayer = pGfxSetup->GetLayer();
    g_pDisplay = pGfxSetup->GetDisplay();
    g_pDevice = pGfxSetup->GetDevice();
    g_pConfigSwitcher = pSwitcher;

    //nn::vi::Initialize();
    //InitializeLayer();

    //nn::gfx::Initialize();
    InitializeGfxObjects();

    g_pConfigSwitcher->Initialize(g_pDevice, g_pDisplay,
        &g_TextureDescriptorPool, &g_SamplerDescriptorPool,
        g_TextureDescriptorBaseIndex + 2, g_SamplerDescriptorBaseIndex + 1);

    g_BufferDescriptorPool.BeginUpdate();
    {
        nn::gfx::GpuAddress gpuAddress;
        g_ConstantBuffer.GetGpuAddress( &gpuAddress );
        g_BufferDescriptorPool.SetBufferView( g_BufferDescriptorBaseIndex, gpuAddress, sizeof( float ) * 4 );
    }
    g_BufferDescriptorPool.EndUpdate();

    g_TextureDescriptorPool.BeginUpdate();
    {
        g_TextureDescriptorPool.SetTextureView( g_TextureDescriptorBaseIndex, &g_TextureView);
    }
    g_TextureDescriptorPool.EndUpdate();

    g_SamplerDescriptorPool.BeginUpdate();
    {
        g_SamplerDescriptorPool.SetSampler( g_SamplerDescriptorBaseIndex, &g_Sampler );
    }
    g_SamplerDescriptorPool.EndUpdate();

    // 毎フレームのレンダリング
    for( int frame = 0; frame < 60 * 1 || !g_pConfigSwitcher->IsSceneSwitchedOn(); ++frame )
    {
        MakeCommand();

        // 定数バッファを更新
        float* pConstantBuffer = g_ConstantBuffer.Map< float >();
        {
            pConstantBuffer[ 0 ] = 0.0f;
            pConstantBuffer[ 1 ] = 0.0f;
            pConstantBuffer[ 2 ] = 0.0f;
            pConstantBuffer[ 3 ] = 0.0f;
        }
        g_ConstantBuffer.Unmap();

        // コマンドの実行
        g_Queue.ExecuteCommand( &g_CommandBuffer, NULL );

        // 結果の表示
        g_Queue.Present( &g_SwapChain, 1 );
        g_Queue.Sync();
    }
    g_pConfigSwitcher->Finalize();

    FinalizeGfxObjects();
    //nn::gfx::Finalize();

    //nn::vi::DestroyLayer(g_pLayer);
    //nn::vi::CloseDisplay(g_pDisplay);
    //nn::vi::Finalize();

}
} // nns
