﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "AudioRenderer.h"
#include <cstdlib>
#include <cmath>

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/os/os_Thread.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include <nn/nn_TimeSpan.h>

#include <nn/audio.h>
#include <nns/audio/audio_WavFormat.h>
#include <nn/audioctrl.h>

namespace nns { namespace audio {

namespace {

const int CharBufferLength = 512;

// choose rendering engine sample rate
const int RenderRate = 48000;  // or 32000;
const int RenderCount = (RenderRate / 200);

const int SineFrequency = 440;

NN_ALIGNAS(4096) char g_WorkBuffer[8 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[14 * 1024 * 1024];

nn::mem::StandardAllocator g_Allocator;
nn::mem::StandardAllocator g_WaveBufferAllocator;

nn::os::ThreadType g_Thread;
const int ThreadStackSize = 16 * 1024;
NN_ALIGNAS(4096) uint8_t g_ThreadStack[ThreadStackSize];
volatile bool g_IsThreadRunning;

nn::os::SystemEvent g_SystemEvent;
nn::audio::AudioRendererHandle g_Handle;
nn::audio::AudioRendererConfig g_Config;

void* g_Data[ChannelCountMax];
nn::audio::VoiceType g_Voice[ChannelCountMax];
nn::audio::SubMixType g_SubMix[ChannelCountMax];

void* g_ConfigBuffer;
void* g_RendererBuffer;

bool g_IsInitialized;
int g_ChannelCount;

bool g_IsMute;

void* Allocate(size_t size)
{
    return g_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    g_Allocator.Free(p);
}

void AudioRendererListener(void* arg)
{
    bool isMutePreviously = g_IsMute;
    while (g_IsThreadRunning)
    {
        g_SystemEvent.Wait();
        bool isSetMute = (!isMutePreviously && g_IsMute);
        bool isUnsetMute = (isMutePreviously && !g_IsMute);
        isMutePreviously = g_IsMute;
        for (int i1=0; i1<g_ChannelCount; ++i1)
        {
            if (nullptr != g_Data[i1])
            {
                if (isSetMute)
                {
                    // ミュート設定
                    nn::audio::SetVoiceVolume(&g_Voice[i1], 0.0f);
                }
                else if (isUnsetMute)
                {
                    // ミュート解除
                    nn::audio::SetVoiceVolume(&g_Voice[i1], 1.0f);
                }
                if (const nn::audio::WaveBuffer* pWaveBuffer = nn::audio::GetReleasedWaveBuffer(&g_Voice[i1]))
                {
                    nn::audio::AppendWaveBuffer(&g_Voice[i1], pWaveBuffer);
                }
            }
        }

        nn::Result result = nn::audio::RequestUpdateAudioRenderer(g_Handle, &g_Config);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
}

void SetVoiceToSubMix(nn::audio::VoiceType* pVoice, nn::audio::SubMixType* pSubMix, void* data, std::size_t dataSize, int samplingRate) NN_NOEXCEPT
{
    nn::audio::AcquireVoiceSlot(&g_Config, pVoice, samplingRate, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
    nn::audio::SetVoiceDestination(&g_Config, pVoice, pSubMix);

    nn::audio::WaveBuffer waveBuffer;
    waveBuffer.buffer = data;
    waveBuffer.size = dataSize;
    waveBuffer.startSampleOffset = 0;
    waveBuffer.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t));
    waveBuffer.loop = true;
    waveBuffer.isEndOfStream = false;
    waveBuffer.pContext = nullptr;
    waveBuffer.contextSize = 0;

    nn::audio::AppendWaveBuffer(pVoice, &waveBuffer);
    nn::audio::SetVoicePlayState(pVoice, nn::audio::VoiceType::PlayState_Play);
    nn::audio::SetVoiceMixVolume(pVoice, pSubMix, 0.707f / 2, 0, 0);
}

std::size_t GenerateSineWave(void** data, int sampleRate, int frequency, int sampleCount) NN_NOEXCEPT
{
    // g_WaveBufferAllocator が管理するメモリ領域はすべて waveBufferMemoryPool メモリプールに追加されています。
    int16_t* p = static_cast<int16_t*>(g_WaveBufferAllocator.Allocate(sampleCount * sizeof(int16_t), nn::audio::BufferAlignSize));
    NN_ABORT_UNLESS_NOT_NULL(p);
    const float Pi = 3.1415926535897932384626433f;
    for (auto i1 = 0; i1 < sampleCount; ++i1)
    {
        p[i1] = static_cast<int16_t>(std::numeric_limits<int16_t>::max() * sinf(2 * Pi * frequency * i1 / sampleRate));
    }
    *data = p;
    return sampleCount * sizeof(int16_t);
}

std::size_t ReadWavFile(nns::audio::WavFormat* format, void** data, const char* filename) NN_NOEXCEPT
{
    nn::fs::FileHandle g_Handle;
    nn::Result result = nn::fs::OpenFile(&g_Handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    int64_t size;

    result = nn::fs::GetFileSize(&size, g_Handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    *data = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size), nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*data);

    // DATA チャンクを読む必要がありますが、ここではそれが 1024 バイト以内に見つかると仮定しています
    const std::size_t WavHeaderDataSize = 1024;

    result = nn::fs::ReadFile(g_Handle, 0, *data, WavHeaderDataSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nns::audio::WavResult wavResult = nns::audio::ParseWavFormat(format, *data, WavHeaderDataSize);
    NN_ABORT_UNLESS_EQUAL(wavResult, nns::audio::WavResult_Success);
    NN_ABORT_UNLESS_EQUAL(format->channelCount, 1);    // 1ch を仮定しています
    NN_ABORT_UNLESS_EQUAL(format->bitsPerSample, 16);  // 16bit PCM を仮定しています

    result = nn::fs::ReadFile(g_Handle, static_cast<std::size_t>(format->dataOffset), *data, static_cast<std::size_t>(format->dataSize));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::fs::CloseFile(g_Handle);

    return static_cast<std::size_t>(format->dataSize);
}

}

void InitializeAudioRenderer(int channels) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!g_IsInitialized, "Audio renderer has been initialized already.");
    NN_SDK_REQUIRES(2 == channels || 6 == channels, "The number of channels is unsupported.");
    g_ChannelCount = channels;

    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
    g_WaveBufferAllocator.Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));

    nn::audioctrl::SetAudioOutputMode(nn::audioctrl::AudioTarget_Tv,
        (2 == channels) ? nn::audioctrl::AudioOutputMode_Pcm2ch : nn::audioctrl::AudioOutputMode_Pcm6ch);

    // レンダラのパラメータを指定します。
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = RenderRate;
    parameter.sampleCount = RenderCount;
    parameter.mixBufferCount = ChannelCountMax + g_ChannelCount; // FinalMix() + SubMix()
    parameter.voiceCount = 36;
    parameter.subMixCount = g_ChannelCount;
    parameter.sinkCount = 1;
    parameter.effectCount = 0;
    parameter.performanceFrameCount = 0;

    // パラメータがシステムでサポートされているかどうかを確認します。
    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(parameter),
        "Invalid AudioRendererParameter specified."
    );

    // レンダラのワークバッファを準備します。
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    g_RendererBuffer = g_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(g_RendererBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::audio::OpenAudioRenderer(&g_Handle, &g_SystemEvent, parameter, g_RendererBuffer, workBufferSize)
    );

    // AudioRendererConfig を初期化します。
    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    g_ConfigBuffer = g_Allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(g_ConfigBuffer);
    nn::audio::InitializeAudioRendererConfig(&g_Config, parameter, g_ConfigBuffer, configBufferSize);

    // ミックスバッファとオーディオバスの関係を定義します。
    int8_t mainBus[ChannelCountMax];
    for (int i1=0; i1<ChannelCountMax; ++i1)
    {
        mainBus[i1] = i1;
    }
    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&g_Config, &finalMix, ChannelCountMax);

    for (auto& mix : g_SubMix)
    {
        nn::audio::AcquireSubMix(&g_Config, &mix, parameter.sampleRate, 1);
    }

    // レンダラの出力先を用意します。
    nn::audio::DeviceSinkType deviceSink;
    const char* audioOutName = "MainAudioOut";
    nn::Result result = nn::audio::AddDeviceSink(&g_Config, &deviceSink, &finalMix, mainBus, g_ChannelCount, audioOutName);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    for (int i1=0; i1<g_ChannelCount; ++i1)
    {
        // SubMix を FinalMix へ接続します
        nn::audio::SetSubMixDestination(&g_Config, &g_SubMix[i1], &finalMix);
        // SubMix の 0 番目のバッファを FinalMix の mainBus へ ボリューム 0.5f でミックスするように設定します
        nn::audio::SetSubMixMixVolume(&g_SubMix[i1], &finalMix, 0.5f, 0, mainBus[i1]);
    }

    // 設定したパラメータをレンダラに反映させます。
    result = nn::audio::RequestUpdateAudioRenderer(g_Handle, &g_Config);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // レンダリングを開始します。
    result = nn::audio::StartAudioRenderer(g_Handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // WaveBuffer に追加するサンプルデータを保持するためのメモリプールを準備します。
    nn::audio::MemoryPoolType waveBufferMemoryPool;
    bool rt = AcquireMemoryPool(&g_Config, &waveBufferMemoryPool, g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    NN_ABORT_UNLESS(rt);
    rt = RequestAttachMemoryPool(&waveBufferMemoryPool);
    NN_ABORT_UNLESS(rt);

    g_IsMute = false;
    g_IsInitialized = true;
}

void SetWaveFile(const char* waveFile, int channelIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_IsInitialized, "Audio renderer is not initialized.");
    NN_SDK_REQUIRES_LESS_EQUAL(0, channelIndex);
    NN_SDK_REQUIRES_GREATER(g_ChannelCount, channelIndex);

    // Voice で再生する波形を準備します。
    int samplingRate = RenderRate;
    std::size_t size = 0;

    // ファイル名が null ポインタの(ファイル名がない)場合は音を出さない。
    if (nullptr != waveFile)
    {
        if (!std::strncmp(waveFile, "Contents:/sin", CharBufferLength))
        {
            size = GenerateSineWave(&g_Data[channelIndex], samplingRate, SineFrequency, samplingRate);
        }
        else
        {
            nns::audio::WavFormat format;
            size = ReadWavFile(&format, &g_Data[channelIndex], waveFile);
            samplingRate = format.sampleRate;
        }
        SetVoiceToSubMix(&g_Voice[channelIndex], &g_SubMix[channelIndex], g_Data[channelIndex], size, samplingRate);
    }
}

void StartAudioRenderer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_IsInitialized, "Audio renderer is not initialized.");

    nn::Result result = nn::os::CreateThread(
        &g_Thread, AudioRendererListener, nullptr,
        g_ThreadStack, ThreadStackSize, nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread()));
    g_IsThreadRunning = result.IsSuccess();
    if (g_IsThreadRunning)
    {
        NN_LOG("Start thread to output audio.\n");
        nn::os::StartThread(&g_Thread);
    }
    else
    {
        NN_LOG("Cannot create thread to output audio.\n");
    }
}

void FinalizeAudioRenderer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_IsInitialized, "Audio renderer is not initialized.");

    if (g_IsThreadRunning)
    {
        g_IsThreadRunning = false;
        nn::os::WaitThread(&g_Thread);
        nn::os::DestroyThread(&g_Thread);
    }

    // レンダリングを終了します。
    nn::audio::StopAudioRenderer(g_Handle);
    nn::audio::CloseAudioRenderer(g_Handle);
    nn::os::DestroySystemEvent(g_SystemEvent.GetBase());

    for (auto& data : g_Data)
    {
        if (data)
        {
            g_WaveBufferAllocator.Free(data);
            data = nullptr;
        }
    }
    if (g_ConfigBuffer)
    {
        g_Allocator.Free(g_ConfigBuffer);
        g_ConfigBuffer = nullptr;
    }
    if (g_RendererBuffer)
    {
        g_Allocator.Free(g_RendererBuffer);
        g_RendererBuffer = nullptr;
    }

    g_WaveBufferAllocator.Finalize();
    g_Allocator.Finalize();
    g_IsInitialized = false;
}

bool IsInitialized() NN_NOEXCEPT
{
    return g_IsInitialized;
}

bool IsMute() NN_NOEXCEPT
{
    return g_IsMute;
}

void MuteAudio(bool isEnabled) NN_NOEXCEPT
{
    g_IsMute = isEnabled;
}

}} // nns::audio
