﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/vi_Result.h>
#include "testDisplay_MockPhysicalDisplay.h"
#include "master/detail/visrv_IModeFilter.h"

const int MockPhysicalDisplay::DisplayLogicalWidth = 1280;
const int MockPhysicalDisplay::DisplayLogicalHeight = 720;

MockPhysicalDisplay::MockPhysicalDisplay()
    : IPhysicalDisplay(&m_Filter)
    , m_State(nn::vi::HotplugState_Disconnected)
    , m_CurrentMode { 0, 0, 0.f, nn::vi::StereoMode_None }
    , m_GetModeResult(nn::ResultSuccess())
    , m_Underscan(0)
    , m_Luma(0.f)
    , m_Power(nn::vi::PowerState_Off)
    , m_CmuMode(nn::vi::CmuMode_Default)
    , m_Contrast(0.f)
    , m_ValidRgbRangesSize(0)
    , m_Alpha(1.f)
    , m_Gamma(2.2f)
    , m_LayerStack(nn::vi::LayerStack_Default)
    , m_IsVsyncEventSupported(false)
    , m_IsHotplugEventSupported(false)
    , m_IsModeChangedEventSupported(false)
    , m_LogicalWidth(MockPhysicalDisplay::DisplayLogicalWidth)
    , m_LogicalHeight(MockPhysicalDisplay::DisplayLogicalHeight)
{
}

nn::Result MockPhysicalDisplay::Open() NN_NOEXCEPT
{
    return nn::ResultSuccess();
}

void MockPhysicalDisplay::Close() NN_NOEXCEPT
{
}

nn::Result MockPhysicalDisplay::GetHotplugState(nn::vi::HotplugStateType* pOutState) const NN_NOEXCEPT
{
    *pOutState = m_State;

    return nn::ResultSuccess();
}

nn::Result MockPhysicalDisplay::GetMode(nn::vi::DisplayModeInfo* pOutMode) const NN_NOEXCEPT
{
    if( m_GetModeResult.IsSuccess() )
    {
        *pOutMode = m_CurrentMode;
    }

    return m_GetModeResult;
}

int MockPhysicalDisplay::ListModes(nn::vi::DisplayModeInfo* pOutModes, int modeCountMax) const NN_NOEXCEPT
{
    int count = 0;
    for( const nn::vi::DisplayModeInfoInternal& mode : m_AvailableModes )
    {
        if( count >= modeCountMax )
        {
            break;
        }

        if( pFilter->IsValid(mode) )
        {
            pOutModes[count++] = mode.clientInfo;
        }
    }

    return count;
}

nn::Result MockPhysicalDisplay::SetMode(const nn::vi::DisplayModeInfo& mode) NN_NOEXCEPT
{
    nn::vi::DisplayModeInfoInternal newMode;
    newMode.clientInfo = mode;
    newMode.ratio = nn::vi::AspectRatio_16_9;

    if( pFilter->IsValid(newMode) )
    {
        m_CurrentMode = mode;
    }
    else
    {
        return nn::vi::ResultNotSupported();
    }

    return nn::ResultSuccess();
}

void MockPhysicalDisplay::SetHotplugState(nn::vi::HotplugStateType state) NN_NOEXCEPT
{
    m_State = state;
}

void MockPhysicalDisplay::SetAvailableModes(const ModeList& modes) NN_NOEXCEPT
{
    m_AvailableModes = modes;
}

void MockPhysicalDisplay::SetGetModeResult(const nn::Result& result) NN_NOEXCEPT
{
    m_GetModeResult = result;
}

int MockPhysicalDisplay::ListRgbRanges(nn::vi::RgbRangeType* pOutRanges, int rgbCountMax) const NN_NOEXCEPT
{
    for( int i = 0; i < rgbCountMax && i < m_ValidRgbRangesSize; ++i )
    {
        pOutRanges[i] = m_ValidRgbRanges[i];
    }

    return m_ValidRgbRangesSize;
}

nn::Result MockPhysicalDisplay::GetRgbRange(nn::vi::RgbRangeType* pOutRange) const NN_NOEXCEPT
{
    *pOutRange = m_CurrentRgb;
    return nn::ResultSuccess();
}

nn::Result MockPhysicalDisplay::SetRgbRange(nn::vi::RgbRange range) NN_NOEXCEPT
{
    m_CurrentRgb = range;
    return nn::ResultSuccess();
}

nn::Result MockPhysicalDisplay::SetUnderscan(int underscan) NN_NOEXCEPT
{
    m_Underscan = underscan;

    return nn::ResultSuccess();
}

nn::Result MockPhysicalDisplay::GetUnderscan(int* pOutUnderscan) const NN_NOEXCEPT
{
    *pOutUnderscan = m_Underscan;

    return nn::ResultSuccess();
}

nn::Result MockPhysicalDisplay::SetAlpha(float alpha) NN_NOEXCEPT
{
    m_Alpha = alpha;
    return nn::ResultSuccess();
}

nn::Result MockPhysicalDisplay::GetPowerState(nn::vi::PowerStateType* pOutState) NN_NOEXCEPT
{
    *pOutState = m_Power;

    return nn::ResultSuccess();
}

nn::Result MockPhysicalDisplay::SetPowerState(nn::vi::PowerState state) NN_NOEXCEPT
{
    m_Power = state;

    return nn::ResultSuccess();
}

nn::Result MockPhysicalDisplay::SetLayerStack(nn::vi::LayerStackType id) NN_NOEXCEPT
{
    m_LayerStack = id;
    return nn::ResultSuccess();
}

nn::Result MockPhysicalDisplay::SetCmuLuma(float luma) NN_NOEXCEPT
{
    m_Luma = luma;

    return nn::ResultSuccess();
}

nn::Result MockPhysicalDisplay::GetCmuLuma(float* pOutLuma) const NN_NOEXCEPT
{
    *pOutLuma = m_Luma;

    return nn::ResultSuccess();
}

bool MockPhysicalDisplay::IsHotplugEventSupported() const NN_NOEXCEPT
{
    return m_IsHotplugEventSupported;
}

bool MockPhysicalDisplay::IsVsyncEventSupported() const NN_NOEXCEPT
{
    return m_IsVsyncEventSupported;
}

bool MockPhysicalDisplay::IsModeChangedEventSupported() const NN_NOEXCEPT
{
    return m_IsModeChangedEventSupported;
}

nn::Result MockPhysicalDisplay::GetLogicalResolution(int* pOutWidth, int* pOutHeight) const NN_NOEXCEPT
{
    *pOutWidth = m_LogicalWidth;
    *pOutHeight = m_LogicalHeight;

    return nn::ResultSuccess();
}

nn::Result MockPhysicalDisplay::GetCmuMode(nn::vi::CmuModeType* pOutMode) const NN_NOEXCEPT
{
    *pOutMode = m_CmuMode;
    return nn::ResultSuccess();
}

nn::Result MockPhysicalDisplay::SetCmuMode(nn::vi::CmuMode mode) NN_NOEXCEPT
{
    m_CmuMode = mode;
    return nn::ResultSuccess();
}

nn::Result MockPhysicalDisplay::GetContrastRatio(float* pOutRatio) const NN_NOEXCEPT
{
    *pOutRatio = m_Contrast;
    return nn::ResultSuccess();
}

nn::Result MockPhysicalDisplay::SetContrastRatio(float ratio) NN_NOEXCEPT
{
    m_Contrast = ratio;
    return nn::ResultSuccess();
}

nn::Result MockPhysicalDisplay::GetGamma(float* pOutGamma) const NN_NOEXCEPT
{
    *pOutGamma = m_Gamma;
    return nn::ResultSuccess();
}

nn::Result MockPhysicalDisplay::SetGamma(float gamma) NN_NOEXCEPT
{
    m_Gamma = gamma;
    return nn::ResultSuccess();
}

void MockPhysicalDisplay::SetValidRgbRanges(const nn::vi::RgbRange* pRanges, int rgbRangeCountMax) NN_NOEXCEPT
{
    int i;
    for( i = 0; i < rgbRangeCountMax && i < sizeof(m_ValidRgbRanges) / sizeof(m_ValidRgbRanges[0]); ++i )
    {
        m_ValidRgbRanges[i] = pRanges[i];
    }

    m_ValidRgbRangesSize = i;
}

float MockPhysicalDisplay::GetAlpha() const NN_NOEXCEPT
{
    return m_Alpha;
}

nn::Result MockPhysicalDisplay::GetLayerStack(nn::vi::LayerStackType* pOutStack) const NN_NOEXCEPT
{
    *pOutStack = m_LayerStack;
    return nn::ResultSuccess();
}

void MockPhysicalDisplay::SetIsHotplugEventSupported(bool isSupported) NN_NOEXCEPT
{
    m_IsHotplugEventSupported = isSupported;
}

void MockPhysicalDisplay::SetIsVsyncEventSupported(bool isSupported) NN_NOEXCEPT
{
    m_IsVsyncEventSupported = isSupported;
}

void MockPhysicalDisplay::SetIsModeChangedEventSupported(bool isSupported) NN_NOEXCEPT
{
    m_IsModeChangedEventSupported = isSupported;
}

void MockPhysicalDisplay::SetLogicalResolution(int width, int height) NN_NOEXCEPT
{
    m_LogicalWidth = width;
    m_LogicalHeight = height;
}

nn::Result MockPhysicalDisplay::SetViewport(int x, int y, int width, int height) NN_NOEXCEPT
{
    return nn::ResultSuccess();
}

nn::Result MockPhysicalDisplay::GetCompositorErrorInfo(nn::vi::CompositorError* pOutErrorInfo, int* pOutLength, int errorID) NN_NOEXCEPT
{
    NN_UNUSED(pOutErrorInfo);
    NN_SDK_ASSERT_NOT_NULL(pOutLength);

    *pOutLength = 0;

    return nn::ResultSuccess();
}
