﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <nn/nn_Log.h>
#include <nnt.h>
#include <nn/vi.private.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/viUtil/testVi_Fixture.h>
#include <nnt/viUtil/testVi_EdidFixture.h>

namespace
{
    #ifdef NN_BUILD_CONFIG_OS_WIN32
        // SIGLO-83050: Each iteration on Windows 7 takes quite a bit more time due to
        //              training the vsync timer.
        //              Once Windows 7 isn't supported, then this can be removed.
        const int StressTestCount = 1024;
    #else
        const int StressTestCount = 8192;
    #endif

    const char* IcosaDisplayNames[] = { "Default", "External", "Internal", "Edid", "Null" };
    const char* CopperDisplayNames[] = { "Default", "External", "Edid", "Null" };
    const char* HoagDisplayNames[] = { "Default", "Internal", "Null" };
    const char* WindowsDisplayNames[] = { "Default" };

    class DisplayStressFixture : public nnt::vi::util::EdidFixture
                               , public ::testing::WithParamInterface<const char*>
                               , public nnt::vi::util::Fixture
    {
    };

    class DisplayListFixture : public nnt::vi::util::EdidFixture
                             , public ::testing::WithParamInterface<std::pair<const char**, size_t>>
                             , public nnt::vi::util::Fixture
    {
    };
}

INSTANTIATE_TEST_CASE_P(DisplayStress_Icosa,
                        DisplayStressFixture,
                        ::testing::ValuesIn(IcosaDisplayNames));

INSTANTIATE_TEST_CASE_P(DisplayStress_Copper,
                        DisplayStressFixture,
                        ::testing::ValuesIn(CopperDisplayNames));

INSTANTIATE_TEST_CASE_P(DisplayStress_Hoag,
                        DisplayStressFixture,
                        ::testing::ValuesIn(HoagDisplayNames));

INSTANTIATE_TEST_CASE_P(DisplayStress_Windows,
                        DisplayStressFixture,
                        ::testing::ValuesIn(WindowsDisplayNames));

INSTANTIATE_TEST_CASE_P(AllDisplayStress_Icosa,
                        DisplayListFixture,
                        ::testing::Values(std::make_pair(IcosaDisplayNames, sizeof(IcosaDisplayNames) / sizeof(IcosaDisplayNames[0]))));

INSTANTIATE_TEST_CASE_P(AllDisplayStress_Copper,
                        DisplayListFixture,
                        ::testing::Values(std::make_pair(CopperDisplayNames, sizeof(CopperDisplayNames) / sizeof(CopperDisplayNames[0]))));

INSTANTIATE_TEST_CASE_P(AllDisplayStress_Hoag,
                        DisplayListFixture,
                        ::testing::Values(std::make_pair(HoagDisplayNames, sizeof(HoagDisplayNames) / sizeof(HoagDisplayNames[0]))));

INSTANTIATE_TEST_CASE_P(AllDisplayStress_Windows,
                        DisplayListFixture,
                        ::testing::Values(std::make_pair(WindowsDisplayNames, sizeof(WindowsDisplayNames) / sizeof(WindowsDisplayNames[0]))));

TEST_P(DisplayStressFixture, OpenClose)
{
    nn::vi::Initialize();

    for( int i = 0; i < StressTestCount; ++i )
    {
        nn::vi::Display* pDisplay;
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, GetParam()));
        nn::vi::CloseDisplay(pDisplay);
    }
}

TEST_P(DisplayStressFixture, OpenFinalize)
{
    for( int i = 0; i < StressTestCount; ++i )
    {
        nn::vi::Initialize();

        nn::vi::Display* pDisplay;
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, GetParam()));

        nn::vi::Finalize();
    }
}

TEST_P(DisplayListFixture, OpenAllClose)
{
    nn::vi::Initialize();

    for( int i = 0; i < StressTestCount; ++i )
    {
        std::vector<nn::vi::Display*> displays;
        displays.resize(GetParam().second);

        for( size_t j = 0; j < GetParam().second; ++j )
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&displays[j], GetParam().first[j]));
        }

        for( size_t j = 0; j < GetParam().second; ++j )
        {
            nn::vi::CloseDisplay(displays[j]);
        }
    }
}

TEST_P(DisplayListFixture, OpenAllFinalize)
{
    for( int i = 0; i < StressTestCount; ++i )
    {
        nn::vi::Initialize();

        std::vector<nn::vi::Display*> displays;
        displays.resize(GetParam().second);

        for( size_t j = 0; j < GetParam().second; ++j )
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&displays[j], GetParam().first[j]));
        }

        nn::vi::Finalize();
    }
}
