﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/vi.private.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nv/nv_MemoryManagement.h>
#include <nn/vi/buffer/vi_Buffer.h>
#include <nn/pcm/pcm.h>


enum class Color : std::uint16_t
{
        Grey  =  0xF888,
        White =  0xFFFF,
        Red   =  0xF00F,
        Green =  0xF0F0
};

static void* NvAllocate(size_t size, size_t alignment, void* userPtr)
{
        NN_UNUSED(userPtr);
        return std::aligned_alloc(alignment, size);
}

static void NvFree(void* ptr, void* userPtr)
{
        NN_UNUSED(userPtr);
        std::free(ptr);
}

static void* NvReallocate(void* ptr, size_t size, void* userPtr)
{
        NN_UNUSED(userPtr);
        return std::realloc(ptr, size);
}

        static void
fillRect(void *ptr,
                Color colour,
                int xpos,
                int ypos,
                int width,
                int height,
                int stride)
{
        int xmax = xpos + width;
        int ymax = ypos + height;
        for (int y=ypos; y < ymax; ++y)
        {
                std::uint16_t *pixel = reinterpret_cast<std::uint16_t*>(ptr);
                pixel += y * stride;
                for (int x = xpos; x < xmax; ++x)
                        pixel[x] = static_cast<std::underlying_type<Color>::type>(colour);
        }
}

void setPixels(void *ptr, int stride)
{
        static int m_counter = 0;


        static const int speed[] =
        {
                1, 2, 3, 8, 16, 32
        };

        fillRect(ptr,  Color::Grey, 0, 0, 256, 256, stride);

        int x = 16;
        int y = 208;
        int b = 6;

        // C
        fillRect(ptr, Color::White, x,   y,       b,   b * 5, stride);
        fillRect(ptr, Color::White, x + b, y, b * 4, b, stride);
        fillRect(ptr, Color::White, x + b, y + (b * 4), b * 4, b, stride);
        x += b * 6;

        // H
        fillRect(ptr, Color::White, x,   y,       b,   b * 5, stride);
        fillRect(ptr, Color::White, x + (b * 4),   y,       b,   b * 5, stride);
        fillRect(ptr, Color::White, x + b,   y + (b * 2),       b * 3,   b, stride);
        x += b * 6;

        // A
        fillRect(ptr, Color::White, x,   y,       b,   b * 5, stride);
        fillRect(ptr, Color::White, x + (b * 4),   y,       b,   b * 5, stride);
        fillRect(ptr, Color::White, x + b,   y,       b * 3,   b, stride);
        fillRect(ptr, Color::White, x + b,   y + (b * 2),       b * 3,   b, stride);

        x += b * 6;

        // R
        fillRect(ptr, Color::White, x,   y,       b,   b * 5, stride);
        fillRect(ptr, Color::White, x + (b * 4),   y,       b,   b * 3, stride);
        fillRect(ptr, Color::White, x + b,   y,       b * 3,   b, stride);
        fillRect(ptr, Color::White, x + b,   y + b * 2,       b * 3,   b, stride);
        fillRect(ptr, Color::White, x + b * 3,   y + b * 3,       b,   b, stride);
        fillRect(ptr, Color::White, x + b * 3,   y + b * 4,       b,   b, stride);
        fillRect(ptr, Color::White, x + b * 4,   y + b * 4,       b,   b, stride);
        x += b * 6;

        // G
        fillRect(ptr, Color::White, x,   y,       b,   b * 5, stride);
        fillRect(ptr, Color::White, x + (b * 4),   y + (b * 3),       b,   b, stride);
        fillRect(ptr, Color::White, x + b, y, b * 4, b, stride);
        fillRect(ptr, Color::White, x + b, y + (b * 4), b * 4, b, stride);
        fillRect(ptr, Color::White, x + b * 2, y + (b * 2), b * 3, b, stride);
        x += b * 6;

        // E
        fillRect(ptr, Color::White, x,   y,       b,   b * 5, stride);
        fillRect(ptr, Color::White, x + b, y, b * 4, b, stride);
        fillRect(ptr, Color::White, x + b, y + (b * 4), b * 4, b, stride);
        fillRect(ptr, Color::White, x + b,   y + (b * 2),       b * 4,   b, stride);
        x += b * 6;


        int s = (speed[0] * m_counter) % 208;
        x = s + 16;
        y = 80;
        if(x < 48)
                fillRect(ptr, Color::Red, 16, y, x, 64, stride);
        else
                fillRect(ptr, Color::Green, 16, y, x, 64, stride);

        ++m_counter;
}

extern "C" void nnMain()
{
        const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
        nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, nullptr);
        nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, nullptr);
        nv::InitializeGraphics( std::malloc( GraphicsSystemMemorySize ), GraphicsSystemMemorySize );

        nn::vi::Initialize();

        nn::pcm::Initialize();

        nn::vi::Display* pDisplay;

        nn::Result result = nn::vi::OpenDefaultDisplay(&pDisplay);
        NN_ASSERT(result.IsSuccess(), "Failed to open default display.");

        nn::vi::Layer* pLayer1;

        const int LayerWidth = 256;
        const int LayerHeight = 256;

        nn::vi::buffer::BufferInfo bufInfo;

        bufInfo.width = LayerWidth;
        bufInfo.height = LayerHeight;
        bufInfo.format = nn::vi::PixelFormat_Rgba4444;
        bufInfo.bufferCount = 2;

        nn::vi::LayerSettings settings;
        nn::vi::SetLayerSettingsDefaults(&settings);
        settings.Reset<nn::vi::LayerFlags::Fullscreen>();

        // On layer creation, it will be positioned in the top left corner with a size matching that of width and height
        // parameters.
        result = nn::vi::CreateLayer(&pLayer1, pDisplay, settings);
        NN_ASSERT(result.IsSuccess(), "Failed to create layer.");

        nn::vi::SetLayerSize(pLayer1, LayerWidth, LayerHeight);
        nn::vi::SetLayerZ(pLayer1, 255);

        int displayWidth;
        int displayHeight;

        result = nn::vi::GetDisplayLogicalResolution(&displayWidth, &displayHeight, pDisplay);
        NN_ASSERT(result.IsSuccess(), "Failed to query display resolution.");

        nn::vi::SetLayerPosition(pLayer1, (displayWidth - LayerWidth) / 2, (displayHeight - LayerHeight) / 2);

        nn::vi::buffer::BufferQueue bufQueue;
        result = bufQueue.Initialize(pLayer1,bufInfo );
        NN_ASSERT(result.IsSuccess(), "Failed to initialize bufferqueue object.");
        size_t size;
        size = bufQueue.GetRequiredMemorySize(bufInfo);
        NN_LOG("Required Memory Size: %d\n", size);

        size_t alignment;
        alignment = bufQueue.GetRequiredMemoryAlignment(bufInfo);
        NN_LOG("Required Allignment: %d\n", alignment);

        for(int i=0; i < bufInfo.bufferCount; i++)
        {
                void* pMemory = std::aligned_alloc(alignment, size);
                bufQueue.SetScanBuffer(i, pMemory, size);
        }

        size_t stride;
        stride = bufQueue.GetStride(bufInfo);

        void* bufferAddr;
        nn::vi::buffer::BufferQueueHandle* buffer;
        for (;;)
        {
                bufQueue.DequeueBuffer(&buffer);
                bufQueue.GetScanBufferAddress(&bufferAddr,buffer);

                setPixels(bufferAddr, stride);

                bufQueue.QueueBuffer(buffer);

#ifdef MEASURE_POWER
                int power = nn::pcm::ReadCurrentPower(nn::pcm::MeasuringPoint_Gpu);
                NN_LOG("PCM Test: power = %d mW (@ GPU)\n", power);
#endif
        }

        bufQueue.Finalize();
        nn::vi::Finalize();
}
