<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Verify TV is unaffected by powering on and off the test SDEV

    .DESCRIPTION
        Powers on and off the test SDEV.  Verifies that CEC and nothing else causes the TV to turn on or change active source.
#>

param
(
    [Parameter(Mandatory=$True)]
    [Int]
    # Number of iterations
    $Iterations,
    
    [Parameter(Mandatory=$False)]
    [String]
    # The artifact path
    $ArtifactPath,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The ip address of the test target
    $TestAddress,
    
    [Parameter(Mandatory=$True)]
    [Int]
    # The tone to be emitted by the test target
    $TestTone,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The IP Address of the power source for the HDMI splitter
    $HDMIPowerIP,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The offset number of the power source for the HDMI splitter
    $HDMIPowerNum,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The ip address of the microphone server
    $ServerAddress
    
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

# The target time for TVs in general is 14 seconds (2 seconds more than the library timeout)
$PowerOffTV = $False
$HDCPTargetTime = [TimeSpan]::FromMilliseconds(14000)

# Note: Most of the above parameters are used in Common.ps1
. "${scriptDirectoryPath}\Common.ps1"

function Time-Tone([System.TimeSpan]$target)
{
    Write-Host "Timing how long it takes the tone to start playing (against $target)..."
    $stopwatch = [diagnostics.stopwatch]::StartNew()
    $attemptCounter = 1
    do
    {
        if ($stopwatch.elapsed -ge $target)
        {
            #Cleanup
            Write-Error "xxx TONE WAS NOT DETECTED WITHIN TARGET TIME (see SIGLO-69359) xxx"
            
            Write-Error "Aborting"
            exit 1
        }
        
        Start-Sleep -s 0.1

        & $CheckToneExe -activeTone $TestTone -ipAddr $ServerAddress > $null
        
        $attemptCounter = $attemptCounter + 1
    }
    while ($lastexitcode -ne 0)
    
    $elapsed = $stopwatch.elapsed
    Write-Host "  (detected after $elapsed)"
}

function Wait-Tone
{
    Write-Host "Making sure the tone is playing..."
    $attemptCounter = 1
    do
    {
        if ($attemptCounter -ge 60)
        {
            Cleanup
            Write-Error "xxx TONE WAS NOT DETECTED FOR 60 SECONDS xxx"
            
            Write-Error "Aborting"
            exit 1
        }

        Write-Host "- Attempt #${attemptCounter} -"
        
        Start-Sleep -s 1

        & $CheckToneExe -activeTone $TestTone -ipAddr $ServerAddress > $null
        
        $attemptCounter = $attemptCounter + 1
    }
    while ($lastexitcode -ne 0)
}

function Wait-No-Tone
{
    Write-Host "Making sure the tone isn't playing..."
    $attemptCounter = 1
    do
    {
        if ($attemptCounter -ge 30)
        {
            Cleanup
            Write-Error "xxx TONE WAS STILL DETECTED AFTER 30 SECONDS xxx"
            
            Write-Error "Aborting"
            exit 1
        }

        Write-Host "- Attempt #${attemptCounter} -"
        
        Start-Sleep -s 1

        & $CheckToneExe -silentTone $TestTone -ipAddr $ServerAddress > $null
        
        $attemptCounter = $attemptCounter + 1
    }
    while ($lastexitcode -ne 0)
}

Initialize -initRemoteControl $False

###############################################################################################
# Make sure the SDEV is the active source on the TV
Write-Host "Resetting SDEV..."
$ret = & $ControlTargetExe reset -t $TestAddress
Write-Host "Connecting to SDEV..."
$ret = & $ControlTargetExe connect -t $TestAddress

###############################################################################################
for($i = 0; $i -lt $Iterations; $i++)
{
    Write-Host "===Iteration $i ==="

    if ($i % 10 -eq 0)
    {
        Write-Host "Playing tone with HDCP..."
        $ret = & $RunOnTargetExe "${scriptRootPath}\Tests\Outputs\NX-NXFP2-a64\Tests\HDCPDisplayTest\Develop\HDCPDisplayTest.nsp" -t $TestAddress --no-wait -- -frequency $TestTone -amplitude $TestAmplitude

        Wait-Tone
    }
    
###############################################################################################
    # Dock/undock

    Write-Host "Undocking..."
    & $ControlTargetExe disable-cradle -t $TestAddress

    Wait-No-Tone

    Write-Host "Docking..."
    & $ControlTargetExe enable-cradle -t $TestAddress

    Time-Tone $HDCPTargetTime
}
###############################################################################################

Cleanup

Write-Host "Passed!"
