﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <algorithm>

#include <nv/nv_MemoryManagement.h>
#include <nv/nv_ServiceName.h>

#include "VideoConfigHelper.h"

namespace
{

void* NvAllocateFunction(size_t size, size_t alignment, void* userPtr)
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}
void NvFreeFunction(void* addr, void* userPtr)
{
    NN_UNUSED(userPtr);
    free(addr);
}
void* NvReallocateFunction(void* addr, size_t newSize, void* userPtr)
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}

const float fpsMargin = 0.01f;

}

/**
* @brief   1920x1080@60fps[CEA],width=1920,height>1080[DTD]
*/
TEST(DisplayConfig, Cea1080p_DtdWe1920Hg1080)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("cea_1080p_dtd_1920x1200.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1920, video.GetActiveDisplayWidth());
    EXPECT_EQ(1080, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_TRUE(video.IsActiveDisplayCea());
    EXPECT_FALSE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   上記でRangeLimitedの下限が60Hz
 */
TEST(DisplayConfig, Cea1080p_DtdWe1920Hg1080_lower60)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("cea_1080p_dtd_1920x1200_rangelimit_lower60.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1920, video.GetActiveDisplayWidth());
    EXPECT_EQ(1080, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_TRUE(video.IsActiveDisplayCea());
    EXPECT_FALSE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   上記でRangeLimitedの上限が60Hz
 */
TEST(DisplayConfig, Cea1080p_DtdWe1920Hg1080_upper60)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("cea_1080p_dtd_1920x1200_rangelimit_upper60.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1920, video.GetActiveDisplayWidth());
    EXPECT_EQ(1080, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_TRUE(video.IsActiveDisplayCea());
    EXPECT_FALSE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   1280x720@60fps[CEA],width>1280[DTD]
 */
TEST(DisplayConfig, Cea720p_DtdWg1280)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("cea_720p_dtd_1440x1080.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1280, video.GetActiveDisplayWidth());
    EXPECT_EQ(720, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_TRUE(video.IsActiveDisplayCea());
    EXPECT_FALSE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   1280x720@60fps, width<1280,height>720の非対応解像度
 */
TEST(DisplayConfig, Cea720p_DtdWl1280Hg720)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("cea_720p_dtd_1024x768.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1280, video.GetActiveDisplayWidth());
    EXPECT_EQ(720, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_TRUE(video.IsActiveDisplayCea());
    EXPECT_FALSE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   上記でRangeLimitedの下限が60Hz
 */
TEST(DisplayConfig, Cea720p_DtdWl1280Hg720_lower60)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("cea_720p_dtd_1024x768_rangelimit_lower60.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1280, video.GetActiveDisplayWidth());
    EXPECT_EQ(720, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_TRUE(video.IsActiveDisplayCea());
    EXPECT_FALSE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   上記でRangeLimitedの上限が60Hz
 */
TEST(DisplayConfig, Cea720p_DtdWl1280Hg720_upper60)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("cea_720p_dtd_1024x768_rangelimit_upper60.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1280, video.GetActiveDisplayWidth());
    EXPECT_EQ(720, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_TRUE(video.IsActiveDisplayCea());
    EXPECT_FALSE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   1920x1080@60fps[CEA],1920x1080@59.9xfps[DTD]
 */
TEST(DisplayConfig, Cea1080p_Dtd1920x1080)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("cea_1080p_dtd_1920x1080.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1920, video.GetActiveDisplayWidth());
    EXPECT_EQ(1080, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_TRUE(video.IsActiveDisplayCea());
    EXPECT_FALSE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   1280x720@60fps[CEA],1920x1080@59.9xfps[DTD]
 */
TEST(DisplayConfig, Cea720p_Dtd1920x1080)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("cea_720p_dtd_1920x1080.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1280, video.GetActiveDisplayWidth());
    EXPECT_EQ(720, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_TRUE(video.IsActiveDisplayCea());
    EXPECT_FALSE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   720x480@60fps[CEA],1920x1080@59.9xfps[DTD]
 */
TEST(DisplayConfig, Cea480p_Dtd1920x1080)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("cea_480p_dtd_1920x1080.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_TRUE(video.IsActiveDisplayCea());
    EXPECT_FALSE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   1280x720@60fps[CEA],1920x1080@60fps[DTD]
 */
TEST(DisplayConfig, Cea720p_Dtd1920x1080F60)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("cea_720p_dtd_1920x1080F60.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1920, video.GetActiveDisplayWidth());
    EXPECT_EQ(1080, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_FALSE(video.IsActiveDisplayCea());
    EXPECT_TRUE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   720x480@60fps[CEA],1920x1080@60fps[DTD]
 */
TEST(DisplayConfig, Cea480p_Dtd1920x1080F60)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("cea_480p_dtd_1920x1080F60.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1920, video.GetActiveDisplayWidth());
    EXPECT_EQ(1080, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_FALSE(video.IsActiveDisplayCea());
    EXPECT_TRUE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   1280x720@60fps[DTD],1920x1080@59.9xfps[DTD]
 */
TEST(DisplayConfig, Dtd1280x720F60_Dtd1920x1080)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_1280x720F60_dtd_1920x1080.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1920, video.GetActiveDisplayWidth());
    EXPECT_EQ(1080, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_FALSE(video.IsActiveDisplayCea());
    EXPECT_TRUE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   720x480@60fps[DTD],1920x1080@59.9xfps[DTD]
 */
TEST(DisplayConfig, Dtd720x480F60_Dtd1920x1080)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_720x480F60_dtd_1920x1080.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1920, video.GetActiveDisplayWidth());
    EXPECT_EQ(1080, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_FALSE(video.IsActiveDisplayCea());
    EXPECT_TRUE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   1920x1080@59.9xfps[DTD]
 */
TEST(DisplayConfig, Dtd1920x1080)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_1920x1080.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1920, video.GetActiveDisplayWidth());
    EXPECT_EQ(1080, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_FALSE(video.IsActiveDisplayCea());
    EXPECT_TRUE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
* @brief   上記でRangeLimitedの下限が60Hz
*/
TEST(DisplayConfig, Dtd1920x1080_lower60)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_1920x1080_rangelimit_lower60.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_TRUE(video.IsActiveDisplayCea());
    EXPECT_FALSE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
* @brief   上記でRangeLimitedの上限が60Hz
*/
TEST(DisplayConfig, Dtd1920x1080_upper59)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_1920x1080_rangelimit_upper59.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_TRUE(video.IsActiveDisplayCea());
    EXPECT_FALSE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   1280x720@59.9xfps[DTD]
 */
TEST(DisplayConfig, Dtd1280x720)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_1280x720.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1280, video.GetActiveDisplayWidth());
    EXPECT_EQ(720, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_FALSE(video.IsActiveDisplayCea());
    EXPECT_TRUE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   720x480@59.9xfps[DTD]
 */
TEST(DisplayConfig, Dtd720x480)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_720x480.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_TRUE(video.IsActiveDisplayCea());
    EXPECT_FALSE(video.IsActiveDisplayDetailed());
    EXPECT_EQ(4, video.GetDisplayInfoCount());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   width>1920,720<=height<=1080[DTD]
 */
TEST(DisplayConfig, DtdWg1920H720t1080)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_2048x1080.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_TRUE(video.IsActiveDisplayCea());
    EXPECT_FALSE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   width<=1920,height>1080[DTD]
 */
TEST(DisplayConfig, DtdWle1920Hg1080)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_1600x1200.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_TRUE(video.IsActiveDisplayCea());
    EXPECT_FALSE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   width<=1920,height<720[DTD]
 */
TEST(DisplayConfig, DtdWle1920Hl720)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_1136x640.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_TRUE(video.IsActiveDisplayCea());
    EXPECT_FALSE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

#if 0
/**
* @brief   width=1920,720<=height<=1080@60Hz[DTD]
*/
TEST(DisplayConfig, DtdWe1920H720t1080)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_1920x????.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1920, video.GetActiveDisplayWidth());
    EXPECT_EQ(1200, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_FALSE(video.IsActiveDisplayCea());
    EXPECT_TRUE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}
#endif

/**
 * @brief   width<=1920,height=1080@60Hz[DTD]
 */
TEST(DisplayConfig, DtdWle1920He1080F60)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_1440x1080F60.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1440, video.GetActiveDisplayWidth());
    EXPECT_EQ(1080, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_FALSE(video.IsActiveDisplayCea());
    EXPECT_TRUE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   上記で59.9xfpsな解像度
 */
TEST(DisplayConfig, DtdWle1920He1080)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_1440x1080.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1440, video.GetActiveDisplayWidth());
    EXPECT_EQ(1080, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_FALSE(video.IsActiveDisplayCea());
    EXPECT_TRUE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   1600x900 でのアスペクト比チェック。
 */
TEST(DisplayConfig, DtdWe1600He900)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_1600x900.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1600, video.GetActiveDisplayWidth());
    EXPECT_EQ(900, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_FALSE(video.IsActiveDisplayCea());
    EXPECT_TRUE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

#if 0
/**
 * @brief   width<=1920,height=720@60Hz[DTD]
 */
TEST(DisplayConfig, DtdWle1920He720)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_????x720.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1920, video.GetActiveDisplayWidth());
    EXPECT_EQ(1200, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_FALSE(video.IsActiveDisplayCea());
    EXPECT_TRUE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}
#endif

/**
 * @brief   16:9の対応解像度[DTD],16:10の対応解像度[DTD]
 */
TEST(DisplayConfig, AspectRatio_Dtd16x9_Dtd16x10)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_1280x720_dtd_1280x800.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1280, video.GetActiveDisplayWidth());
    EXPECT_EQ(720, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_FALSE(video.IsActiveDisplayCea());
    EXPECT_TRUE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   16:10の対応解像度[DTD],4:3の対応解像度[DTD]
 */
TEST(DisplayConfig, AspectRatio_Dtd16x10_Dtd4x3)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_1280x800_dtd_1280x960.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1280, video.GetActiveDisplayWidth());
    EXPECT_EQ(800, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_FALSE(video.IsActiveDisplayCea());
    EXPECT_TRUE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   16:10の対応解像度[DTD],15:9の対応解像度[DTD]
 */
TEST(DisplayConfig, AspectRatio_Dtd16x10_Dtd15x9)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_1280x800_dtd_1280x768.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1280, video.GetActiveDisplayWidth());
    EXPECT_EQ(800, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_FALSE(video.IsActiveDisplayCea());
    EXPECT_TRUE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   16:9の2つの対応解像度[DTD]
 */
TEST(DisplayConfig, AspectRatioComparison_Dtd16x9)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_1136x640_dtd_1366x768.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1366, video.GetActiveDisplayWidth());
    EXPECT_EQ(768, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_FALSE(video.IsActiveDisplayCea());
    EXPECT_TRUE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   16:9の2つの対応解像度[DTD] + Extra resolutions.
 */
TEST(DisplayConfig, AspectRatioComparison_Dtd16x9_Extra)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_1136x640_dtd_1366x768_ex.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1024, video.GetActiveDisplayWidth());
    EXPECT_EQ(768, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_FALSE(video.IsActiveDisplayCea());
    EXPECT_FALSE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   16:10の2つの対応解像度[DTD]
 */
TEST(DisplayConfig, AspectRatioComparison_Dtd16x10)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_1280x800_dtd_1440x900.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1440, video.GetActiveDisplayWidth());
    EXPECT_EQ(900, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_FALSE(video.IsActiveDisplayCea());
    EXPECT_TRUE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   4:3の[DTD]と15:9の対応解像度[DTD](画素数:4:3<15:9)
 */
TEST(DisplayConfig, AspectRatioComparison_Dtd4x3_l_Dtd15x9)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_1024x768_dtd_1280x768.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1280, video.GetActiveDisplayWidth());
    EXPECT_EQ(768, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_FALSE(video.IsActiveDisplayCea());
    EXPECT_TRUE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}

/**
 * @brief   4:3の[DTD]と15:9の対応解像度[DTD](画素数:4:3>15:9)
 */
TEST(DisplayConfig, AspectRatioComparison_Dtd4x3_g_Dtd15x9)
{
    nns::VideoConfigHelper video;
    ASSERT_TRUE(video.IsExternalDisplayConnected());
    ASSERT_TRUE(video.SetVirtualEdid("dtd_1152x864_dtd_1280x768.bin"));
    video.ShowDisplayInfo();

    // Should not be negative ID.
    EXPECT_LE(0, video.GetActiveDisplayId());

    // Check resolution.
    EXPECT_EQ(1152, video.GetActiveDisplayWidth());
    EXPECT_EQ(864, video.GetActiveDisplayHeight());
    float fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    EXPECT_FALSE(video.IsActiveDisplayCea());
    EXPECT_TRUE(video.IsActiveDisplayDetailed());

    // Try to choose 480p.
    ASSERT_TRUE(video.SetMandatoryResolution());
    EXPECT_EQ(720, video.GetActiveDisplayWidth());
    EXPECT_EQ(480, video.GetActiveDisplayHeight());
    fps = video.GetActiveDisplayFps();
    EXPECT_LT(60.0f - fpsMargin, fps);
    EXPECT_GT(60.0f + fpsMargin, fps);
    ASSERT_TRUE(video.ResetPreviousResolution());
}


extern "C" void nnMain()
{
    int argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();
    ::testing::InitGoogleTest(&argc, argv);
    {
        const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
        nv::SetGraphicsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::SetGraphicsDevtoolsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::SetGraphicsServiceName("nvdrv:t");
        nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);
    }

    nnt::Exit(RUN_ALL_TESTS());
}
