﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <windows.h>
#include <cstdio>
#include <memory>
#include <vector>

#include <nn/util/util_Compression.h>

#include "CRCHelper.h"

#define clamp(n) (unsigned char)((n) >= 255 ? 255 : ((n) < 0 ? 0 : (n)))

namespace PNGHelper
{

// Writes out a PNG file from the compressed data
HRESULT WritePNG(const char* pFile, int width, int height, int size, BYTE* pData)
{
    BYTE pngHeader[] =    { 0x89, 'P', 'N', 'G', 0x0D, 0x0A, 0x1A, 0x0A,    // 8: Basic PNG header
                            0, 0, 0, 0x0D, 'I', 'H', 'D', 'R',              // 8: Start of IHDR chunk
                            0, 0, 0, 0, 0, 0, 0, 0,                         // 8: Width and height (to be set below)
                            8, 2, 0, 0, 0,                                  // 5: 8-bit Truecolor (24-bit total) with default compression/filter and no interlacing
                            0, 0, 0, 0,                                     // 4: CRC (to be set below)
                            0, 0, 0, 0, 'I', 'D', 'A', 'T' };               // 8: Start of IDAT chunk (length to be filled in)

    pngHeader[16] = (BYTE)((width & 0xFF000000) >> 24);
    pngHeader[17] = (BYTE)((width & 0x00FF0000) >> 16);
    pngHeader[18] = (BYTE)((width & 0x0000FF00) >> 8);
    pngHeader[19] = (BYTE)((width & 0x000000FF) >> 0);
    pngHeader[20] = (BYTE)((height & 0xFF000000) >> 24);
    pngHeader[21] = (BYTE)((height & 0x00FF0000) >> 16);
    pngHeader[22] = (BYTE)((height & 0x0000FF00) >> 8);
    pngHeader[23] = (BYTE)((height & 0x000000FF) >> 0);

    int crc = ComputeCRC(&pngHeader[12], 13);
    pngHeader[29] = (BYTE)((crc & 0xFF000000) >> 24);
    pngHeader[30] = (BYTE)((crc & 0x00FF0000) >> 16);
    pngHeader[31] = (BYTE)((crc & 0x0000FF00) >> 8);
    pngHeader[32] = (BYTE)((crc & 0x000000FF) >> 0);
    pngHeader[33] = (BYTE)((size & 0xFF000000) >> 24);
    pngHeader[34] = (BYTE)((size & 0x00FF0000) >> 16);
    pngHeader[35] = (BYTE)((size & 0x0000FF00) >> 8);
    pngHeader[36] = (BYTE)((size & 0x000000FF) >> 0);

    BYTE pngFooter[] =    { 0, 0, 0, 0,                                     // 4: IDAT CRC (to be set below)
                            0, 0, 0, 0, 'I', 'E', 'N', 'D', 0xAE, 0x42, 0x60, 0x82 };   // 12: IEND chunk

    crc = ComputeCRC(&pngFooter[0], 4);
    crc = ComputeCRC(pData, size, crc);
    pngFooter[0] = (BYTE)((crc & 0xFF000000) >> 24);
    pngFooter[1] = (BYTE)((crc & 0x00FF0000) >> 16);
    pngFooter[2] = (BYTE)((crc & 0x0000FF00) >> 8);
    pngFooter[3] = (BYTE)((crc & 0x000000FF) >> 0);

    HRESULT hr;
    HANDLE hFile = CreateFileA(pFile, GENERIC_WRITE, 0, nullptr, CREATE_ALWAYS, 0, nullptr);

    if (hFile)
    {
        DWORD cbWritten = 0;
        BOOL result = WriteFile(hFile, &pngHeader, sizeof(pngHeader), &cbWritten, nullptr);
        if (result)
        {
            result = WriteFile(hFile, pData, size, &cbWritten, nullptr);
        }
        if (result)
        {
            result = WriteFile(hFile, &pngFooter, sizeof(pngFooter), &cbWritten, nullptr);
        }

        CloseHandle(hFile);

        hr = result ? S_OK : HRESULT_FROM_WIN32(GetLastError());
    }
    else
    {
        hr = HRESULT_FROM_WIN32(GetLastError());
    }

    return hr;
}

// Performs phase 2 (compression stage) to make a PNG
size_t CompressPhase1ToPhase2(void* pDst, size_t dstSize, const void* pSrc, size_t srcSize)
{
    // Rather than write a compression algorithm, use the one provided by nn::util
    int workSize = nn::util::CompressZlibWorkBufferSizeDefault;
    std::vector<unsigned char> pWork;
    pWork.resize(workSize);

    size_t compressedSize;
    if(!nn::util::CompressZlib( &compressedSize, pDst, dstSize, pSrc, srcSize, &pWork[0], workSize, 1, 1))
    {
        return 0;
    }

    return compressedSize;
}

// Copies an 8-bit YUV bitmap into a phase 1 (pre-compression) PNG
void Convert8BitYUVtoPhase1(BYTE* bytes, int width, int height, BYTE** out, int* outSize)
{
    // 24 bit bitmap
    // +1 byte per row for PNG data
    // +1 for 32 bit -> 24 bit overrun
    BYTE* pBitmap = new BYTE[(width + 1) * height * 3 + 1];
    *outSize = (width + 1) * height * 3;

    // Note: The source is inverted vertically
    // Note 2: Need to go from BGR to RGB
    BYTE* pOut = pBitmap;
    BYTE* pIn = bytes;

    const int yBitmap = 0;
    const int yBytes = 0;
    const int xBitmap = 6;
    const int xBytes = 4;

    for (int y = 0; y < height; ++y)
    {
        // Always use the "Sub" filter
        pOut[0] = 1;
        ++pOut;

        BYTE old0 = 0;
        BYTE old1 = 0;
        BYTE old2 = 0;

        for (int x = 0; x < width; x += 2)
        {
            int Y = pIn[1] * 298;
            int U = pIn[0];
            int V = pIn[2];

            int a = 409 * V - 56992;
            int b = 36064 - 100 * U - 208 * V;
            int c = 516 * U - 70688;

            BYTE temp0 = clamp(((Y + a) >> 8));
            pOut[0] = temp0 - old0;

            BYTE temp1 = clamp(((Y + b) >> 8));
            pOut[1] = temp1 - old1;

            BYTE temp2 = clamp(((Y + c) >> 8));
            pOut[2] = temp2 - old2;

            Y = pIn[3] * 298;

            old0 = clamp(((Y + a) >> 8));
            pOut[3] = old0 - temp0;

            old1 = clamp(((Y + b) >> 8));
            pOut[4] = old1 - temp1;

            old2 = clamp(((Y + c) >> 8));
            pOut[5] = old2 - temp2;

            pOut += xBitmap;
            pIn += xBytes;
        }

        pOut += yBitmap;
        pIn += yBytes;
    }

    *out = pBitmap;
}

// Copies a 32-bit BGRA bitmap into a phase 1 (pre-compression) PNG
void ConvertBGRAtoPhase1(BYTE* bytes, int width, int height, BYTE** out, int* outSize)
{
    // 24 bit bitmap
    // +1 byte per row for PNG data
    // +1 for 32 bit -> 24 bit overrun
    BYTE* pBitmap = new BYTE[(width + 1) * height * 3 + 1];
    *outSize = (width + 1) * height * 3;

    // Note: The source is inverted vertically
    // Note 2: Need to go from BGR to RGB
    BYTE* pOut = pBitmap;
    BYTE* pIn = bytes;

    const int yBitmap = 0;
    const int yBytes = 0;

    const int xBitmap = 3;
    const int xBytes = 4;

    for (int y = 0; y < height; ++y)
    {
        // Always use the "None" filter
        pOut[0] = 0;
        ++pOut;

        for (int x = 0; x < width; ++x)
        {
            pOut[0] = pIn[2];
            pOut[1] = pIn[1];
            pOut[2] = pIn[0];

            pOut += xBitmap;
            pIn += xBytes;
        }

        pOut += yBitmap;
        pIn += yBytes;
    }

    *out = pBitmap;
}

}
