﻿using System;
using System.IO;
using System.Reflection;

namespace TargetUtility
{
    class EnvironmentInfo
    {
        public static string NintendoSdkRoot { get { return EnvironmentInfoImpl.Instance.NintendoSdkRoot; } }
        public static bool IsRepository { get { return EnvironmentInfoImpl.Instance.IsRepository; } }
        public static bool IsPackage { get { return EnvironmentInfoImpl.Instance.IsPackage; } }
        public static string LogServer { get { return EnvironmentInfoImpl.Instance.NintendoSdkRoot; } }
        public static string ControlTarget { get { return EnvironmentInfoImpl.Instance.ControlTarget; } }
        public static string RunOnTarget { get { return EnvironmentInfoImpl.Instance.RunOnTarget; } }
        public static string TargetManager { get { return EnvironmentInfoImpl.Instance.TargetManager; } }
        public static string InitializeSdevFull { get { return EnvironmentInfoImpl.Instance.InitializeSdevFull; } }
        public static string InitializeSdevExe { get { return EnvironmentInfoImpl.Instance.InitializeSdevExe; } }
        public static string WriteBootConfig { get { return EnvironmentInfoImpl.Instance.WriteBootConfig; } }
        public static string WriteInvalidBootConfig { get { return EnvironmentInfoImpl.Instance.WriteInvalidBootConfig; } }
        public static string Nact { get { return EnvironmentInfoImpl.Instance.Nact; } }
        public static string DevMenuCommand { get { return EnvironmentInfoImpl.Instance.DevMenuCommand; } }
        public static string SettingsManager { get { return EnvironmentInfoImpl.Instance.SettingsManager; } }
        public static string LogDirectory { get { return EnvironmentInfoImpl.Instance.LogDirectory; } }

        public static void SetNintendoSdkRoot(string nintendoSdkRoot)
        {
            EnvironmentInfoImpl.Instance.SetNintendoSdkRoot(nintendoSdkRoot);
        }

        internal class EnvironmentInfoImpl
        {
            private static EnvironmentInfoImpl instance = new EnvironmentInfoImpl();
            public static EnvironmentInfoImpl Instance { get { return instance; } }

            public string NintendoSdkRoot { get; private set; }
            public bool IsRepository { get; private set; }
            public bool IsPackage { get { return !IsRepository; } }
            public string LogServer { get; private set; }
            public string ControlTarget { get; private set; }
            public string RunOnTarget { get; private set; }
            public string TargetManager { get; private set; }
            public string InitializeSdevFull { get; private set; }
            public string InitializeSdevExe { get; private set; }
            public string WriteBootConfig { get; private set; }
            public string WriteInvalidBootConfig { get; private set; }
            public string Nact { get; private set; }
            public string DevMenuCommand { get; private set; }
            public string SettingsManager { get; private set; }

            private string LogDirectoryValue = string.Empty;
            private object LogDirectoryLock = new object();
            public string LogDirectory
            {
                get
                {
                    lock (LogDirectoryLock)
                    {
                        if (string.IsNullOrEmpty(LogDirectoryValue))
                        {
                            LogDirectoryValue =
                                Path.Combine(Path.GetDirectoryName(
                                    Assembly.GetEntryAssembly().Location), DateTime.Now.ToString("yyyy-MM-dd_HH-mm-ss"));
                            Directory.CreateDirectory(LogDirectoryValue);
                        }
                        return LogDirectoryValue;
                    }
                }
            }

            private string FindNintendoSdkRoot()
            {
                string startDirectory = Path.GetDirectoryName(Assembly.GetEntryAssembly().Location);
                const string sdkRootMarkFileName = "NintendoSdkRootMark";
                var currentDirectory = startDirectory;
                var driveRootDirectory = Path.GetPathRoot(startDirectory);

                while (currentDirectory != driveRootDirectory)
                {
                    if (File.Exists(Path.Combine(currentDirectory, sdkRootMarkFileName)))
                    {
                        return currentDirectory;
                    }
                    currentDirectory = Path.GetFullPath(Path.Combine(currentDirectory, ".."));
                }

                throw new FileNotFoundException(
                    $"Not found NintendoSdkRootMark\n" +
                    $"Path={startDirectory}");
            }

            private EnvironmentInfoImpl()
            {
                SetNintendoSdkRoot(FindNintendoSdkRoot());
            }

            public void SetNintendoSdkRoot(string nintendoSdkRoot)
            {
                if (!Directory.Exists(nintendoSdkRoot))
                {
                    throw new DirectoryNotFoundException(
                        $"Not found the specified directory\n" +
                        $"Path={nintendoSdkRoot}");
                }

                const string sdkRootMarkFileName = "NintendoSdkRootMark";
                if (!File.Exists(
                    Path.Combine(nintendoSdkRoot, sdkRootMarkFileName)))
                {
                    throw new FileNotFoundException(
                        $"Not found NintendoSdkRootMark\n" +
                        $"Path={nintendoSdkRoot}");
                }

                NintendoSdkRoot = nintendoSdkRoot;

                Environment.SetEnvironmentVariable("NINTENDO_SDK_ROOT", NintendoSdkRoot);

                IsRepository = File.Exists(Path.Combine(NintendoSdkRoot, @"SigloRootMark"));

                TargetManager = Path.Combine(
                    NintendoSdkRoot, @"Externals\Oasis\bin\NintendoTargetManager.exe");
                if (!File.Exists(TargetManager))
                {
                    TargetManager = Path.Combine(
                        NintendoSdkRoot, @"Resources\Firmwares\NX\OasisTM\NintendoTargetManager.exe");
                }

                LogServer = Path.Combine(
                    NintendoSdkRoot, @"Tools\Log\LogServer\LogServer.exe");

                ControlTarget = Path.Combine(
                    NintendoSdkRoot, @"Tools\CommandLineTools\ControlTarget.exe");

                RunOnTarget = Path.Combine(
                    NintendoSdkRoot, @"Tools\CommandLineTools\RunOnTarget.exe");

                InitializeSdevFull = Path.Combine(
                    NintendoSdkRoot, @"Integrate\Scripts\NX\Initialize\Initialize-Sdev-Full.ps1");

                InitializeSdevExe = Path.Combine(
                    NintendoSdkRoot, @"Tools\CommandLineTools\InitializeSdev.exe");

                WriteBootConfig = Path.Combine(
                    NintendoSdkRoot, @"Integrate\Scripts\NX\BootConfig\Write-ProperBootConfig.ps1");

                WriteInvalidBootConfig = Path.Combine(
                    NintendoSdkRoot, @"Integrate\Scripts\NX\BootConfig\Write-InvalidBootConfig.ps1");

                Nact = Path.Combine(
                    NintendoSdkRoot, @"Integrate\CommandLineTools\nact.exe");

                DevMenuCommand = Path.Combine(
                    NintendoSdkRoot, @"Programs\Eris\Outputs\NX-NXFP2-a64\TargetTools\DevMenuCommand\Develop\DevMenuCommand.nsp");

                SettingsManager = Path.Combine(
                    NintendoSdkRoot, @"Programs\Eris\Outputs\NX-NXFP2-a64\TargetTools\SettingsManager\Develop\SettingsManager.nsp");
            }
        }
    }
}
