﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Log.h>
#include <nn/diag/diag_Backtrace.h>
#include <nn/diag/diag_Symbol.h>
#include <nn/diag/diag_AbortObserverSystemApi.h>
#include <nn/diag/diag_AbortObserverForSystem.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs.h>

#include <nn/err/err_Result.h>

namespace {
    void WriteTempFile(std::string path, const void* buffer, size_t size)
    {
        nn::fs::MountHostRoot();
        NN_UTIL_SCOPE_EXIT{ nn::fs::UnmountHostRoot(); };
        auto deleteResult = nn::fs::DeleteFile(path.c_str());
        if (!(deleteResult.IsSuccess() || deleteResult <= nn::fs::ResultPathNotFound()))
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(deleteResult);
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::CreateFile(path.c_str(), size));

        nn::fs::FileHandle file;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::OpenFile(&file, path.c_str(), nn::fs::OpenMode::OpenMode_Write)
        );
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(file); };
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::WriteFile(file, 0, buffer, size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush))
        );

    }

    void ReadTempFile(std::string path, void* buffer, size_t size)
    {
        nn::fs::MountHostRoot();
        NN_UTIL_SCOPE_EXIT{ nn::fs::UnmountHostRoot(); };

        nn::fs::FileHandle file;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::OpenFile(&file, path.c_str(), nn::fs::OpenMode::OpenMode_Read)
        );
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(file); };
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::ReadFile(file, 0, buffer, size)
        );

    }

    const char AbortTempFile[] = "\\Temp.bin";
    std::string GetTempFilePath()
    {
        std::string path;
        path = nnt::GetHostArgv()[1];
        path += AbortTempFile;

        return path;
    }

    nn::diag::SdkAbortObserverHolder g_AbortObserverHolder;
    void TestAbortObserver(const nn::diag::SdkAbortInfo& info)
    {
        WriteTempFile(GetTempFilePath(), &info.result, sizeof(info.result));
    }
} // namespace

TEST(SystemProcessAbortTest, Basic)
{
    nn::diag::InitializeSdkAbortObserverHolder(&g_AbortObserverHolder, TestAbortObserver);
    nn::diag::RegisterSdkAbortObserver(&g_AbortObserverHolder);

    auto errResult = nn::err::ResultSystemProgramAbort();
    EXPECT_DEATH(
        NN_ABORT_UNLESS_RESULT_SUCCESS(errResult), "");

    nn::Result result;
    ReadTempFile(GetTempFilePath(), &result, sizeof(result));

    NNT_EXPECT_RESULT_FAILURE(nn::err::ResultSystemProgramAbort, result);
}
