﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>

#include <nn/ddsf/ddsf_IEventHandler.h>
#include <nn/ddsf/ddsf_EventHandlerLooper.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

// #define NNT_DDSF_ENABLE_VERBOSE_LOG
#ifdef NNT_DDSF_ENABLE_VERBOSE_LOG
#define NNT_DDSF_LOG(...) NN_LOG(__VA_ARGS__)
#else
#define NNT_DDSF_LOG(...)
#endif

namespace {

// 割り込みハンドラスレッドに与えるスタックリソース
const size_t InterruptThreadStackSize = nn::os::StackRegionAlignment;
NN_ALIGNAS(nn::os::StackRegionAlignment) char g_InterruptThreadStack[InterruptThreadStackSize];
nn::os::ThreadType g_InterruptThread;

static int g_HandlerIdBase = 0;

void LoopHandler(void* p)
{
    auto& looper = *reinterpret_cast<nn::ddsf::EventHandlerLooper*>(p);
    NN_ABORT_UNLESS(!looper.IsRunningOnEventHandlerContext()); // このコンテキスト上で EXPECT_TRUE できないので仕方なく ABORT_UNLESS
    looper.LoopAuto();
    NN_ABORT_UNLESS(!looper.IsRunningOnEventHandlerContext()); // このコンテキスト上で EXPECT_TRUE できないので仕方なく ABORT_UNLESS
};

class TestEventHandler : public nn::ddsf::IEventHandler
{
public:
    TestEventHandler() NN_NOEXCEPT :
        m_Event(nn::os::EventClearMode_AutoClear),
        m_Counter(0),
        m_CounterMutex(false),
        m_pLooperToUnregisterOnNextHandleEvent(nullptr)
    {
        Initialize(m_Event.GetBase());

        m_HandlerId = g_HandlerIdBase++;

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::os::CreateThread(
                &m_SelfSignalThread, SelfSignalThread, this,
                m_SelfSignalThreadStack, SelfSignalThreadStackSize, nn::os::DefaultThreadPriority - 1)
        );
    }

    int GetCounter() const NN_NOEXCEPT
    {
        return m_Counter;
    }

    void StartSelfSignalThread() NN_NOEXCEPT
    {
        m_IsSelfSignalThreadActive = true;
        nn::os::StartThread(&m_SelfSignalThread);
    }

    void StopSelfSignalThread() NN_NOEXCEPT
    {
        m_IsSelfSignalThreadActive = false;
        NNT_DDSF_LOG("[%d] Stopping\n", m_HandlerId);
        nn::os::WaitThread(&m_SelfSignalThread);
        NNT_DDSF_LOG("[%d] *** Self signaling has stopped ***\n", m_HandlerId);
    }

    void Signal() NN_NOEXCEPT
    {
        m_Event.Signal();
    }

    void HandleEvent() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_Event.TryWait()); // 自分がシグナル状態でなければ呼ばれないことをテストしつつ AutoClear によってフラグを落とす
        {
            std::lock_guard<decltype(m_CounterMutex)> lock(m_CounterMutex);
            while ( m_Counter > 0 )
            {
                --m_Counter;
                NNT_DDSF_LOG("[%d] Counted down: %d\n", m_HandlerId, m_Counter);
            }
        }
        if ( m_pLooperToUnregisterOnNextHandleEvent )
        {
            // HandleEvent 中に自分自身を unregister するケース
            NN_ABORT_UNLESS(m_pLooperToUnregisterOnNextHandleEvent->IsRunningOnEventHandlerContext()); // このコンテキスト上で EXPECT_TRUE できないので仕方なく ABORT_UNLESS
            m_pLooperToUnregisterOnNextHandleEvent->UnregisterHandler(this);
            m_pLooperToUnregisterOnNextHandleEvent = nullptr;
            NNT_DDSF_LOG("[%d] Unregistered on handle event\n", m_HandlerId);
        }
    }

    void SetUnregisterNextFlag(nn::ddsf::EventHandlerLooper& looper) NN_NOEXCEPT
    {
        m_pLooperToUnregisterOnNextHandleEvent = &looper;
    }
    void WaitUnregisterNextFlagCleared() NN_NOEXCEPT
    {
        int allowedRetries = 20;
        while ( m_pLooperToUnregisterOnNextHandleEvent )
        {
            allowedRetries--;
            NN_ABORT_UNLESS_GREATER(allowedRetries, 0);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20));
        }
    }

private:
    static void SelfSignalThread(void* p) NN_NOEXCEPT
    {
        auto& handler = *reinterpret_cast<TestEventHandler*>(p);
        handler.SelfSignalLoop();
    };

    void SelfSignalLoop() NN_NOEXCEPT
    {
        while ( m_IsSelfSignalThreadActive )
        {
            const int WaitMilliseconds = (m_HandlerId + 1) * 3;
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(WaitMilliseconds));

            {
                std::lock_guard<decltype(m_CounterMutex)> lock(m_CounterMutex);
                ++m_Counter;
                NNT_DDSF_LOG("[%d] Counted up: %d\n", m_HandlerId, m_Counter);
            }
            Signal();
        }
    }

private:
    int m_HandlerId{ 0 };

    nn::os::Event m_Event{ nn::os::EventClearMode_AutoClear };

    int m_Counter{ 0 };
    nn::os::Mutex m_CounterMutex{ false };

    nn::ddsf::EventHandlerLooper* m_pLooperToUnregisterOnNextHandleEvent{ nullptr };

    // 自分で自分にシグナルしまくるスレッド
    static constexpr size_t SelfSignalThreadStackSize = nn::os::StackRegionAlignment;
    NN_ALIGNAS(nn::os::StackRegionAlignment) char m_SelfSignalThreadStack[SelfSignalThreadStackSize];
    nn::os::ThreadType m_SelfSignalThread;
    bool m_IsSelfSignalThreadActive{ false };
};

}

TEST(EventHandler, WaitLoopEnterExit)
{
    nn::ddsf::EventHandlerLooper looper;
    looper.Initialize();

    static TestEventHandler handler; // TORIAEZU: サイズが大きいのでスタックから取らない

    looper.RegisterHandler(&handler);
    handler.StartSelfSignalThread(); // 自分で自分に非同期にシグナルを繰り返す

    // わざとループスレッドの優先度を低くする
    NNT_ASSERT_RESULT_SUCCESS(
        nn::os::CreateThread(
            &g_InterruptThread, LoopHandler, &looper,
            g_InterruptThreadStack, InterruptThreadStackSize, nn::os::LowestThreadPriority)
    );
    nn::os::StartThread(&g_InterruptThread);

    looper.WaitLoopEnter();
    EXPECT_TRUE(looper.IsLooping()); // WaitLoopEnter 後は立っていることが保証される

    handler.StopSelfSignalThread();
    looper.UnregisterHandler(&handler);

    // 割り込みハンドラスレッドのループを終了
    looper.RequestStop();
    looper.WaitLoopExit();
    EXPECT_FALSE(looper.IsLooping()); // WaitLoopExit 後は落ちていることが保証される

    looper.Finalize();

    nn::os::WaitThread(&g_InterruptThread);
    nn::os::DestroyThread(&g_InterruptThread);
}

TEST(EventHandler, MultipleHandler)
{
    nn::ddsf::EventHandlerLooper looper;
    looper.Initialize();

    EXPECT_FALSE(looper.IsRunningOnEventHandlerContext());

    static TestEventHandler s_Handlers[10]; // TORIAEZU: サイズが大きいのでスタックから取らない
    static TestEventHandler s_MoreHandlers[10]; // TORIAEZU: サイズが大きいのでスタックから取らない

    for ( auto&& handler : s_Handlers )
    {
        EXPECT_TRUE(handler.IsInitialized());
        EXPECT_FALSE(handler.IsRegistered());
        looper.RegisterHandler(&handler);
        EXPECT_TRUE(handler.IsRegistered());
        handler.StartSelfSignalThread(); // 自分で自分に非同期にシグナルを繰り返す
    }

    NNT_ASSERT_RESULT_SUCCESS(
        nn::os::CreateThread(
            &g_InterruptThread, LoopHandler, &looper,
            g_InterruptThreadStack, InterruptThreadStackSize, nn::os::DefaultThreadPriority - 2)
        );
    nn::os::StartThread(&g_InterruptThread);

    // 割り込みスレッドがすでに実行中でもハンドラを追加登録できる
    for ( auto&& handler : s_MoreHandlers )
    {
        EXPECT_TRUE(handler.IsInitialized());
        EXPECT_FALSE(handler.IsRegistered());
        looper.RegisterHandler(&handler);
        EXPECT_TRUE(handler.IsRegistered());
        handler.StartSelfSignalThread();
    }

    EXPECT_FALSE(looper.IsRunningOnEventHandlerContext());
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5000));

    // ハンドラの解除
    for ( auto&& handler : s_Handlers )
    {
        handler.StopSelfSignalThread();
    }
    for ( auto&& handler : s_Handlers )
    {
        handler.SetUnregisterNextFlag(looper); // 次の HandleEvent 内で Unregister させる
        handler.Signal(); // Unregister させる

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20));

        EXPECT_FALSE(handler.IsRegistered());
        EXPECT_EQ(0, handler.GetCounter());
    }

    // ハンドラの解除
    for ( auto&& handler : s_MoreHandlers )
    {
        handler.StopSelfSignalThread();
    }
    for ( auto&& handler : s_MoreHandlers )
    {
        looper.UnregisterHandler(&handler);
        EXPECT_FALSE(handler.IsRegistered());
        EXPECT_EQ(0, handler.GetCounter());
    }

    // 割り込みハンドラスレッドのループを終了
    looper.RequestStop();
    nn::os::WaitThread(&g_InterruptThread);
    nn::os::DestroyThread(&g_InterruptThread);

    looper.Finalize();
}
