﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  アドレス変換 API のテスト
//---------------------------------------------------------------------------

#include <cstring>

#include "../Common/test_Pragma.h"

#include <nn/TargetConfigs/build_Compiler.h>

#include <nnt/nntest.h>

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/dd.h>

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_Result.h>
#include <nn/svc/svc_MemoryMapSelect.h>

namespace nnt { namespace dd { namespace physicalAddress {

const int singleSize = nn::os::MemoryPageSize;
const int doubleSize = nn::os::MemoryPageSize * 2;
const int largeSize = nn::os::MemoryPageSize * 10;
NN_ALIGNAS(4096) char singlePage[ singleSize ];
NN_ALIGNAS(4096) char doublePage[ doubleSize ];
NN_ALIGNAS(4096) char largePage[ largeSize ];

// API が呼び出せるかどうかのみのテスト
TEST(PhysicalAddress, CallTestPhysicalAddressMultiQuery)
{
    nn::Result result;
    nn::dd::PhysicalMemoryInfo pmi;
    nn::dd::InitializePhysicalMemoryInfo(&pmi, doublePage, doubleSize);
    NN_LOG("Query Physical Address: addr=0x%x size=%d\n\n", doublePage, doubleSize);
    while ((result = nn::dd::QueryNextPhysicalMemoryInfo(&pmi)).IsSuccess())
    {
        NN_LOG("Physical Address=0x%llx\n", nn::dd::GetPhysicalAddress(&pmi));
        NN_LOG("Virtual Address=0x%x\n", nn::dd::GetVirtualAddress(&pmi));
        NN_LOG("Size=%d\n", nn::dd::GetSize(&pmi));
        NN_LOG("\n");
    }
    EXPECT_TRUE(nn::dd::ResultEndOfQuery::Includes(result));
}

TEST(PhysicalAddress, CallTestPhysicalAddressSingleQuery)
{
    nn::dd::PhysicalAddress address;
    nn::Result result = nn::dd::QuerySinglePhysicalAddress(&address, singlePage, singleSize);
    EXPECT_TRUE(result.IsSuccess());

    if (result.IsSuccess())
    {
        NN_LOG("Physical Address=0x%llx\n", address);
        NN_LOG("Virtual Address=0x%x\n", singlePage);
        NN_LOG("Size=%d\n", singleSize);
    }
    else
    {
        NN_LOG("result=%d\n", result);
    }
}

// 物理アドレスと仮想アドレスで下位ビットが一致しているか確認
bool CheckOffset(nn::dd::PhysicalAddress physicalAddress, const void* virtualAddress)
{
    uintptr_t lowBitMask     = nn::os::MemoryPageSize - 1;
    uintptr_t physicalOffset = static_cast<uintptr_t>(physicalAddress)     & lowBitMask;
    uintptr_t virtualOffset  = reinterpret_cast<uintptr_t>(virtualAddress) & lowBitMask;
    return physicalOffset == virtualOffset;
}

// 正しく領域が取得できることを確認するテスト
void CheckPhysicalMemorySingleRegion(bool topAligned, bool bottomAligned)
{
    const int offset = 1;

    // 期待する物理アドレスを求めておく
    nn::dd::PhysicalAddress expectPhysicalAddress;
    nn::Result result = nn::dd::QuerySinglePhysicalAddress(&expectPhysicalAddress, singlePage, singleSize);
    EXPECT_TRUE(result.IsSuccess());

    // テスト用仮想メモリ領域を設定＆テスト条件によって値を修正
    char* bufferTop   = singlePage;
    size_t bufferSize = singleSize;
    if (!topAligned)
    {
        expectPhysicalAddress += offset;
        bufferTop             += offset;
        bufferSize            -= offset;
    }
    if (!bottomAligned)
    {
        bufferSize -= offset;
    }

    // QueryNextPhysicalMemoryInfo() による確認
    int regionCount = 0;
    nn::dd::PhysicalMemoryInfo pmi;
    nn::dd::InitializePhysicalMemoryInfo(&pmi, bufferTop, bufferSize);
    while ((result = nn::dd::QueryNextPhysicalMemoryInfo(&pmi)).IsSuccess())
    {
        NN_LOG("Single page: top %s, bottom %s\n", topAligned?"aligned":"not aligned", bottomAligned?"aligned":"not aligned");
        NN_LOG("Physical Address=0x%llx\n", nn::dd::GetPhysicalAddress(&pmi));
        NN_LOG("Virtual Address=0x%x\n", nn::dd::GetVirtualAddress(&pmi));
        NN_LOG("Size=%d\n", nn::dd::GetSize(&pmi));
        NN_LOG("\n");

        // 期待通りの値になっているか確認
        EXPECT_TRUE(CheckOffset(nn::dd::GetPhysicalAddress(&pmi), nn::dd::GetVirtualAddress(&pmi)));
        EXPECT_EQ(nn::dd::GetPhysicalAddress(&pmi), expectPhysicalAddress);
        EXPECT_EQ(nn::dd::GetVirtualAddress(&pmi),  bufferTop);
        EXPECT_EQ(nn::dd::GetSize(&pmi),            bufferSize);
        ++regionCount;
    }
    EXPECT_TRUE(nn::dd::ResultEndOfQuery::Includes(result));
    EXPECT_EQ(regionCount, 1);

    // QuerySinglePhysicalAddress() による確認
    nn::dd::PhysicalAddress physicalAddress;
    result = nn::dd::QuerySinglePhysicalAddress(&physicalAddress, bufferTop, bufferSize);
    EXPECT_TRUE(result.IsSuccess());

    NN_LOG("Physical Address=0x%llx (QuerySingle)\n", physicalAddress);

    // 期待通りの値になっているか確認
    EXPECT_TRUE(CheckOffset(physicalAddress, bufferTop));
    EXPECT_EQ(physicalAddress, expectPhysicalAddress);
}

// ページの真ん中の領域の物理アドレスを得る
//
// +--------+
// |        |
// ++++++++++
// |        |
// +--------+
TEST(PhysicalAddress, SinglePageTopAndBottomNotAligned)
{
    //CheckPhysicalMemoryRegion(singlePage + offset, singleSize - offset * 2, 1, singlePage + offset, singleSize - offset * 2);
    CheckPhysicalMemorySingleRegion(false, false);
}

// ページの上寄りの領域の物理アドレスを得る（先頭アドレスのみページアライメイト）
//
// ++++++++++ <- page top aligned
// ++++++++++
// ++++++++++
// |        |
// +--------+
TEST(PhysicalAddress, SinglePageTopAligned)
{
    CheckPhysicalMemorySingleRegion(true, false);
}

// ページの下寄りの領域の物理アドレスを得る（末尾アドレスのみページアライメイト）
//
// +--------+
// |        |
// ++++++++++
// ++++++++++
// ++++++++++ <- page bottom aligned
TEST(PhysicalAddress, SinglePageBottomAligned)
{
    CheckPhysicalMemorySingleRegion(false, true);
}

// ページのぴったりな領域の物理アドレスを得る（先頭・末尾アドレスともにページアライメイト）
//
// ++++++++++ <- page top aligned
// ++++++++++
// ++++++++++
// ++++++++++
// ++++++++++ <- page bottom aligned
TEST(PhysicalAddress, SinglePageTopAndBottomAligned)
{
    CheckPhysicalMemorySingleRegion(true, true);
}

#if 0  // Alias 領域に対する QueryPhysicalMemory ができなくなったので廃止
// 管理単位の異なる２ページにまたがる領域の物理アドレスを得る
//
// +--------+
// |        |
// ++++++++++
// ++++++++++
// ++++++++++ <- page boundary
// ++++++++++
// ++++++++++
// |        |
// +--------+
TEST(PhysicalAddress, DoublePageTopAndBottomNotAligned)
{
    const int offset = 1;

    nn::Result result;
    nn::dd::PhysicalMemoryInfo pmi;

    char* aliasTop = reinterpret_cast<char*>((NN_SVC_ADDR_STACK_BEGIN + NN_SVC_ADDR_STACK_END) / 2);

    // テスト用仮想メモリ領域
    void*  bufferTop  = aliasTop + offset;
    size_t bufferSize = doubleSize - offset * 2;

    // 期待値
    nn::dd::PhysicalAddress expectPhysicalAddress[2];
    void*                   expectVirtualAddress[2] = {aliasTop + offset, aliasTop + nn::os::MemoryPageSize};
    size_t                  expectSize[2]           = {nn::os::MemoryPageSize - offset, nn::os::MemoryPageSize - offset};

    // Alias を使って 2 つの物理メモリ領域にマップされた仮想メモリ領域をつくる
    result = nn::svc::MapMemory(
                        reinterpret_cast<uintptr_t>(aliasTop),
                        reinterpret_cast<uintptr_t>(doublePage),
                        nn::os::MemoryPageSize);
    EXPECT_TRUE(result.IsSuccess());
    result = nn::svc::MapMemory(
                        reinterpret_cast<uintptr_t>(aliasTop) + nn::os::MemoryPageSize,
                        reinterpret_cast<uintptr_t>(largePage),
                        nn::os::MemoryPageSize);
    EXPECT_TRUE(result.IsSuccess());

    // doublePage の属性を変更し管理領域を 2 つに分ける
    // ※ 現状、これでは単一のメモリ領域として扱われるので保留
    //nn::os::SetMemoryPermission(reinterpret_cast<uintptr_t>(doublePage + nn::os::MemoryPageSize), nn::os::MemoryPageSize, nn::os::MemoryPermission::MemoryPermission_ReadOnly);

    // 期待する物理アドレスを求めておく（１ページ目）
    nn::dd::InitializePhysicalMemoryInfo(&pmi, aliasTop, doubleSize);
    result = nn::dd::QueryNextPhysicalMemoryInfo(&pmi);
    EXPECT_TRUE(result.IsSuccess());
    expectPhysicalAddress[0] = nn::dd::GetPhysicalAddress(&pmi) + offset;

    // 期待する物理アドレスを求めておく（２ページ目）
    result = nn::dd::QuerySinglePhysicalAddress(&expectPhysicalAddress[1], aliasTop + nn::os::MemoryPageSize, nn::os::MemoryPageSize);
    EXPECT_TRUE(result.IsSuccess());

    // 複数ページで管理される物理アドレスを取得
    int regionCount = 0;
    nn::dd::InitializePhysicalMemoryInfo(&pmi, bufferTop, bufferSize);
    while ((result = nn::dd::QueryNextPhysicalMemoryInfo(&pmi)).IsSuccess())
    {
        NN_LOG("Page %d / 2\n", regionCount + 1);
        NN_LOG("Physical Address=0x%llx\n", nn::dd::GetPhysicalAddress(&pmi));
        NN_LOG("Virtual Address=0x%x\n", nn::dd::GetVirtualAddress(&pmi));
        NN_LOG("Size=%d\n", nn::dd::GetSize(&pmi));

        // 期待値と一致するか確認
        EXPECT_TRUE(CheckOffset(nn::dd::GetPhysicalAddress(&pmi), nn::dd::GetVirtualAddress(&pmi)));
        EXPECT_EQ(nn::dd::GetPhysicalAddress(&pmi), expectPhysicalAddress[regionCount]);
        EXPECT_EQ(nn::dd::GetVirtualAddress(&pmi),  expectVirtualAddress[regionCount]);
        EXPECT_EQ(nn::dd::GetSize(&pmi),            expectSize[regionCount]);

        ++regionCount;
    }
    EXPECT_TRUE(nn::dd::ResultEndOfQuery::Includes(result));
    EXPECT_EQ(regionCount, 2);
}
#endif

#if 1  // SetMemoryPermission でメモリ管理領域が分かれるようになったのでこちらでテスト
TEST(PhysicalAddress, CheckRegionCount)
{
    // doublePage の属性を変更し管理領域を 10 個に分ける
    for (uintptr_t offset = 0; offset < largeSize; offset += nn::os::MemoryPageSize * 2)
    {
        nn::os::SetMemoryPermission(reinterpret_cast<uintptr_t>(largePage) + offset, nn::os::MemoryPageSize, nn::os::MemoryPermission::MemoryPermission_ReadOnly);
    }

    // largePage の物理アドレスを取得
    int regionCount = 0;
    nn::Result result;
    nn::dd::PhysicalMemoryInfo pmi;
    nn::dd::InitializePhysicalMemoryInfo(&pmi, largePage, largeSize);
    while ((result = nn::dd::QueryNextPhysicalMemoryInfo(&pmi)).IsSuccess())
    {
        NN_LOG("Page %d / 10\n", regionCount + 1);
        NN_LOG("Physical Address=0x%llx\n", nn::dd::GetPhysicalAddress(&pmi));
        NN_LOG("Virtual Address=0x%x\n", nn::dd::GetVirtualAddress(&pmi));
        NN_LOG("Size=%d\n", nn::dd::GetSize(&pmi));

        ++regionCount;
    }
    EXPECT_TRUE(nn::dd::ResultEndOfQuery::Includes(result));

    // 管理領域が 10 個あることを確認
    EXPECT_EQ(regionCount, 10);
}
#endif

}}} // namespace nnt::dd::physicalAddress
