﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//-----------------------------------------------------------------------------
//  デバイスアドレス空間の API のテスト
//-----------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_SdkText.h>

#include <nnt/nntest.h>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/dd.h>
#include <nn/dd/dd_Result.h>

#if defined(NN_BUILD_CONFIG_OS_HORIZON) && \
    (defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || \
     defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || \
     defined(NN_BUILD_CONFIG_SOC_TEGRA_X1))

namespace nnt { namespace dd { namespace deviceAddressSpace {

const uint64_t      AddressSpaceSize  = 4ull * 1024ull * 1024ull * 1024ull;
NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY uint8_t MappingRegion1[11 * 1024 * 1024];
NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY uint8_t MappingRegion2[15 * 1024 * 1024];

const nn::dd::DeviceVirtualAddress deviceAddressTop1 = 0x40025000ull;
const nn::dd::DeviceVirtualAddress deviceAddressTop2 = 0x80000000ull;

namespace {

void TestIncrementalMapDas(nn::dd::DeviceAddressSpaceType* pDevice, nn::dd::ProcessHandle handle, uint64_t processAddress, uint64_t size, uint64_t deviceAddress, nn::dd::MemoryPermission devicePermission)
{
    nn::dd::DeviceAddressSpaceMapInfo info;
    nn::dd::InitializeDeviceAddressSpaceMapInfo(&info, pDevice, handle, processAddress, size, deviceAddress, devicePermission);

    for (;;)
    {
        size_t mappedSize;
        auto result = nn::dd::MapNextDeviceAddressSpaceRegion(&mappedSize, &info);
        EXPECT_TRUE( result.IsSuccess() );
        if (mappedSize == 0)
        {
            break;
        }
        NN_LOG("MapNextDeviceAddressSpaceRegion(size=0x%012llx)", size);

        NN_LOG(" ... adrs=0x%012llx size=0x%012p devAdrs=0x%012llx\n",
                    nn::dd::GetMappedProcessAddress(&info),
                    nn::dd::GetMappedSize(&info),
                    nn::dd::GetMappedDeviceVirtualAddress(&info) );
        EXPECT_TRUE( nn::dd::GetMappedSize(&info) == mappedSize );

        nn::dd::UnmapDeviceAddressSpaceRegion(&info);
    }
}

}   // anonymous namespace

//-----------------------------------------------------------------------------
//  デバイスアドレス機能は、Attach と Map が分離しており、
//  順不同で行なうことができるため、これらの順番を変えた動作テストを行なう。
//  Map と Attach で２パターン、Unmap と Detach で２パターンなので、
//  合計４パターンの順序で呼出しを行なうテストを実施する。
//
// デバイスアドレス空間機能の動作テスト１
//  Create → Attach → Map → Unmap → Detach → Destroy が成功するか。
//
TEST(DeviceAddressSpace, NormalDeviceAddressSpaceTest1)
{
    NN_LOG(NN_TEXT("SEQ1> Create -> Attach -> Map -> Unamp -> Detach -> Destroy のテスト\n"));

    nn::dd::DeviceAddressSpaceType  device;

    // ミラーリングの前準備
    NN_LOG("GetCurrentProcessHandle()\n");
    auto currentProcessHandle = nn::dd::GetCurrentProcessHandle();
    EXPECT_TRUE( currentProcessHandle != nn::os::InvalidNativeHandle );

    NN_LOG("CreateDeviceAddressSpace()\n");
    auto result = nn::dd::CreateDeviceAddressSpace(&device, AddressSpaceSize);
    EXPECT_TRUE( result.IsSuccess() );

    // 試しアタッチには今後も使われる可能性が低い Sata を使う
    NN_LOG("AttachDeviceAddressSpace()\n");
    result = nn::dd::AttachDeviceAddressSpace(&device, nn::dd::DeviceName_Sata);
    EXPECT_TRUE( result.IsSuccess() );

    // インクリメンタル版のマップ操作
    uint64_t address  = reinterpret_cast<uint64_t>(MappingRegion1);
    nn::dd::DeviceVirtualAddress deviceAddress1 = deviceAddressTop1;

    TestIncrementalMapDas(&device, currentProcessHandle, address, sizeof(MappingRegion1), deviceAddress1, nn::dd::MemoryPermission_ReadWrite);

    // 下位 22bit Aligned 版のマップ操作
    nn::dd::DeviceVirtualAddress deviceAddress2 = deviceAddressTop2 + (reinterpret_cast<uint64_t>(MappingRegion2) & 0x3fffff);
    NN_LOG("MapDeviceAddressSpaceAligned() cpuAdrs=0x%p\n", MappingRegion2);
    NN_LOG("                               devAdrs=0x%p\n", deviceAddress2);
    result = nn::dd::MapDeviceAddressSpaceAligned(
                                    &device,
                                    currentProcessHandle,
                                    reinterpret_cast<uint64_t>(MappingRegion2),
                                    sizeof(MappingRegion2),
                                    deviceAddress2,
                                    nn::dd::MemoryPermission_ReadWrite);
    EXPECT_TRUE( result.IsSuccess() );

    // ここからはデバイスアドレス空間にミラーリングされた状態

    // TORIAEZU:
    //  理想的には、ここで特定デバイスを操作して、
    //  指定したデバイスアドレス空間へのアクセスが正しく行なえることを
    //  テストすべき。汎用 DMA などのデバイスのアクセス方法が分かれば
    //  対応を検討する。

    // ミラーリングの後処理
    NN_LOG("UnmapDeviceAddressSpace()\n");
    nn::dd::UnmapDeviceAddressSpace(&device,
                                    currentProcessHandle,
                                    reinterpret_cast<uint64_t>(MappingRegion2),
                                    sizeof(MappingRegion2),
                                    deviceAddress2);

    NN_LOG("DetachDeviceAddressSpace()\n");
    nn::dd::DetachDeviceAddressSpace(&device, nn::dd::DeviceName_Sata);

    NN_LOG("DestroyDeviceAddressSpace()\n");
    nn::dd::DestroyDeviceAddressSpace(&device);
}


//-----------------------------------------------------------------------------
// デバイスアドレス空間機能の動作テスト２
//  Create → Attach → Map → Detach → Unmap → Destroy が成功するか。
//
TEST(DeviceAddressSpace, NormalDeviceAddressSpaceTest2)
{
    NN_LOG(NN_TEXT("SEQ2> Create -> Attach -> Map -> Detach -> Unamp -> Destroy のテスト\n"));

    nn::dd::DeviceAddressSpaceType   device;

    // ミラーリングの前準備
    NN_LOG("GetCurrentProcessHandle()\n");
    auto currentProcessHandle = nn::dd::GetCurrentProcessHandle();
    EXPECT_TRUE( currentProcessHandle != nn::os::InvalidNativeHandle );

    NN_LOG("CreateDeviceAddressSpace()\n");
    auto result = nn::dd::CreateDeviceAddressSpace(&device, AddressSpaceSize);
    EXPECT_TRUE( result.IsSuccess() );

    // 試しアタッチには今後も使われる可能性が低い Sata を使う
    NN_LOG("AttachDeviceAddressSpace()\n");
    result = nn::dd::AttachDeviceAddressSpace(&device, nn::dd::DeviceName_Sata);
    EXPECT_TRUE( result.IsSuccess() );

    // インクリメンタル版のマップ操作
    uint64_t address  = reinterpret_cast<uint64_t>(MappingRegion1);
    nn::dd::DeviceVirtualAddress deviceAddress1 = deviceAddressTop1;

    TestIncrementalMapDas(&device, currentProcessHandle, address, sizeof(MappingRegion1), deviceAddress1, nn::dd::MemoryPermission_ReadWrite);

    // 下位 22bit Aligned 版のマップ操作
    nn::dd::DeviceVirtualAddress deviceAddress2 = deviceAddressTop2 + (reinterpret_cast<uint64_t>(MappingRegion2) & 0x3fffff);
    NN_LOG("MapDeviceAddressSpaceAligned() cpuAdrs=0x%p\n", MappingRegion2);
    NN_LOG("                               devAdrs=0x%p\n", deviceAddress2);
    result = nn::dd::MapDeviceAddressSpaceAligned(
                                    &device,
                                    currentProcessHandle,
                                    reinterpret_cast<uint64_t>(MappingRegion2),
                                    sizeof(MappingRegion2),
                                    deviceAddress2,
                                    nn::dd::MemoryPermission_ReadWrite);
    EXPECT_TRUE( result.IsSuccess() );

    // ここからはデバイスアドレス空間にミラーリングされた状態

    // TORIAEZU:
    //  理想的には、ここで特定デバイスからデバイスアドレス空間へのアクセスを
    //  行なうテストを実装すべき。

    // ミラーリングの後処理
    NN_LOG("DetachDeviceAddressSpace()\n");
    nn::dd::DetachDeviceAddressSpace(&device, nn::dd::DeviceName_Sata);

    NN_LOG("UnmapDeviceAddressSpace()\n");
    nn::dd::UnmapDeviceAddressSpace(&device,
                                    currentProcessHandle,
                                    reinterpret_cast<uint64_t>(MappingRegion2),
                                    sizeof(MappingRegion2),
                                    deviceAddress2);

    NN_LOG("DestroyDeviceAddressSpace()\n");
    nn::dd::DestroyDeviceAddressSpace(&device);
}


//-----------------------------------------------------------------------------
// デバイスアドレス空間機能の動作テスト３
//  Create → Map → Attach → Detach → Unmap → Destroy が成功するか。
//
TEST(DeviceAddressSpace, NormalDeviceAddressSpaceTest3)
{
    NN_LOG(NN_TEXT("SEQ3> Create -> Map → Attach -> Detach -> Unmap -> Destroy のテスト\n"));

    nn::dd::DeviceAddressSpaceType   device;

    // ミラーリングの前準備
    NN_LOG("GetCurrentProcessHandle()\n");
    auto currentProcessHandle = nn::dd::GetCurrentProcessHandle();
    EXPECT_TRUE( currentProcessHandle != nn::os::InvalidNativeHandle );

    NN_LOG("CreateDeviceAddressSpace()\n");
    auto result = nn::dd::CreateDeviceAddressSpace(&device, AddressSpaceSize);
    EXPECT_TRUE( result.IsSuccess() );

    // インクリメンタル版のマップ操作
    uint64_t address  = reinterpret_cast<uint64_t>(MappingRegion1);
    nn::dd::DeviceVirtualAddress deviceAddress1 = deviceAddressTop1;

    TestIncrementalMapDas(&device, currentProcessHandle, address, sizeof(MappingRegion1), deviceAddress1, nn::dd::MemoryPermission_ReadWrite);

    // 下位 22bit Aligned 版のマップ操作
    nn::dd::DeviceVirtualAddress deviceAddress2 = deviceAddressTop2 + (reinterpret_cast<uint64_t>(MappingRegion2) & 0x3fffff);
    NN_LOG("MapDeviceAddressSpaceAligned() cpuAdrs=0x%p\n", MappingRegion2);
    NN_LOG("                               devAdrs=0x%p\n", deviceAddress2);
    result = nn::dd::MapDeviceAddressSpaceAligned(
                                    &device,
                                    currentProcessHandle,
                                    reinterpret_cast<uint64_t>(MappingRegion2),
                                    sizeof(MappingRegion2),
                                    deviceAddress2,
                                    nn::dd::MemoryPermission_ReadWrite);
    EXPECT_TRUE( result.IsSuccess() );

    // 試しアタッチには今後も使われる可能性が低い Sata を使う
    NN_LOG("AttachDeviceAddressSpace()\n");
    result = nn::dd::AttachDeviceAddressSpace(&device, nn::dd::DeviceName_Sata);
    EXPECT_TRUE( result.IsSuccess() );

    // ここからはデバイスアドレス空間にミラーリングされた状態

    // TORIAEZU:
    //  理想的には、ここで特定デバイスからデバイスアドレス空間へのアクセスを
    //  行なうテストを実装すべき。

    // ミラーリングの後処理
    NN_LOG("DetachDeviceAddressSpace()\n");
    nn::dd::DetachDeviceAddressSpace(&device, nn::dd::DeviceName_Sata);

    NN_LOG("UnmapDeviceAddressSpace()\n");
    nn::dd::UnmapDeviceAddressSpace(&device,
                                    currentProcessHandle,
                                    reinterpret_cast<uint64_t>(MappingRegion2),
                                    sizeof(MappingRegion2),
                                    deviceAddress2);

    NN_LOG("DestroyDeviceAddressSpace()\n");
    nn::dd::DestroyDeviceAddressSpace(&device);
}


//-----------------------------------------------------------------------------
// デバイスアドレス空間機能の動作テスト４
//  Create → Map → Attach → Unmap → Detach → Destroy が成功するか。
//
TEST(DeviceAddressSpace, NormalDeviceAddressSpaceTest4)
{
    NN_LOG(NN_TEXT("SEQ4> Create -> Map -> Attach -> Unamp -> Detach -> Destroy のテスト\n"));

    nn::dd::DeviceAddressSpaceType   device;

    // ミラーリングの前準備
    NN_LOG("GetCurrentProcessHandle()\n");
    auto currentProcessHandle = nn::dd::GetCurrentProcessHandle();
    EXPECT_TRUE( currentProcessHandle != nn::os::InvalidNativeHandle );

    NN_LOG("CreateDeviceAddressSpace()\n");
    auto result = nn::dd::CreateDeviceAddressSpace(&device, AddressSpaceSize);
    EXPECT_TRUE( result.IsSuccess() );

    // インクリメンタル版のマップ操作
    uint64_t address  = reinterpret_cast<uint64_t>(MappingRegion1);
    nn::dd::DeviceVirtualAddress deviceAddress1 = deviceAddressTop1;

    TestIncrementalMapDas(&device, currentProcessHandle, address, sizeof(MappingRegion1), deviceAddress1, nn::dd::MemoryPermission_ReadWrite);

    // 下位 22bit Aligned 版のマップ操作
    nn::dd::DeviceVirtualAddress deviceAddress2 = deviceAddressTop2 + (reinterpret_cast<uint64_t>(MappingRegion2) & 0x3fffff);
    NN_LOG("MapDeviceAddressSpaceAligned() cpuAdrs=0x%p\n", MappingRegion2);
    NN_LOG("                               devAdrs=0x%p\n", deviceAddress2);
    result = nn::dd::MapDeviceAddressSpaceAligned(
                                    &device,
                                    currentProcessHandle,
                                    reinterpret_cast<uint64_t>(MappingRegion2),
                                    sizeof(MappingRegion2),
                                    deviceAddress2,
                                    nn::dd::MemoryPermission_ReadWrite);
    EXPECT_TRUE( result.IsSuccess() );

    // 試しアタッチには今後も使われる可能性が低い Sata を使う
    NN_LOG("AttachDeviceAddressSpace()\n");
    result = nn::dd::AttachDeviceAddressSpace(&device, nn::dd::DeviceName_Sata);
    EXPECT_TRUE( result.IsSuccess() );

    // ここからはデバイスアドレス空間にミラーリングされた状態

    // TORIAEZU:
    //  理想的には、ここで特定デバイスからデバイスアドレス空間へのアクセスを
    //  行なうテストを実装すべき。

    // ミラーリングの後処理
    NN_LOG("UnmapDeviceAddressSpace()\n");
    nn::dd::UnmapDeviceAddressSpace(&device,
                                    currentProcessHandle,
                                    reinterpret_cast<uint64_t>(MappingRegion2),
                                    sizeof(MappingRegion2),
                                    deviceAddress2);

    NN_LOG("DetachDeviceAddressSpace()\n");
    nn::dd::DetachDeviceAddressSpace(&device, nn::dd::DeviceName_Sata);

    NN_LOG("DestroyDeviceAddressSpace()\n");
    nn::dd::DestroyDeviceAddressSpace(&device);
}


//-----------------------------------------------------------------------------
// デバイスアドレス空間機能の動作テスト1 の アドレス引数有り版
//  Create → Attach → Map → Unmap → Detach → Destroy が成功するか。
//
TEST(DeviceAddressSpaceWithAddressArg, NormalDeviceAddressSpaceTestWithAddressArg1)
{
    NN_LOG(NN_TEXT("SEQ1> Create -> Attach -> Map -> Unamp -> Detach -> Destroy のテスト\n"));

    nn::dd::DeviceAddressSpaceType  device;

    // ミラーリングの前準備
    NN_LOG("GetCurrentProcessHandle()\n");
    auto currentProcessHandle = nn::dd::GetCurrentProcessHandle();
    EXPECT_TRUE( currentProcessHandle != nn::os::InvalidNativeHandle );

    NN_LOG("CreateDeviceAddressSpace()\n");
    auto result = nn::dd::CreateDeviceAddressSpace(&device, deviceAddressTop1, AddressSpaceSize);
    EXPECT_TRUE( result.IsSuccess() );

    // 試しアタッチには今後も使われる可能性が低い Sata を使う
    NN_LOG("AttachDeviceAddressSpace()\n");
    result = nn::dd::AttachDeviceAddressSpace(&device, nn::dd::DeviceName_Sata);
    EXPECT_TRUE( result.IsSuccess() );

    // インクリメンタル版のマップ操作
    uint64_t address  = reinterpret_cast<uint64_t>(MappingRegion1);
    nn::dd::DeviceVirtualAddress deviceAddress1 = deviceAddressTop1;

    TestIncrementalMapDas(&device, currentProcessHandle, address, sizeof(MappingRegion1), deviceAddress1, nn::dd::MemoryPermission_ReadWrite);

    // 下位 22bit Aligned 版のマップ操作
    nn::dd::DeviceVirtualAddress deviceAddress2 = deviceAddressTop2 + (reinterpret_cast<uint64_t>(MappingRegion2) & 0x3fffff);
    NN_LOG("MapDeviceAddressSpaceAligned() cpuAdrs=0x%p\n", MappingRegion2);
    NN_LOG("                               devAdrs=0x%p\n", deviceAddress2);
    result = nn::dd::MapDeviceAddressSpaceAligned(
                                    &device,
                                    currentProcessHandle,
                                    reinterpret_cast<uint64_t>(MappingRegion2),
                                    sizeof(MappingRegion2),
                                    deviceAddress2,
                                    nn::dd::MemoryPermission_ReadWrite);
    EXPECT_TRUE( result.IsSuccess() );

    // ここからはデバイスアドレス空間にミラーリングされた状態

    // TORIAEZU:
    //  理想的には、ここで特定デバイスを操作して、
    //  指定したデバイスアドレス空間へのアクセスが正しく行なえることを
    //  テストすべき。汎用 DMA などのデバイスのアクセス方法が分かれば
    //  対応を検討する。

    // ミラーリングの後処理
    NN_LOG("UnmapDeviceAddressSpace()\n");
    nn::dd::UnmapDeviceAddressSpace(&device,
                                    currentProcessHandle,
                                    reinterpret_cast<uint64_t>(MappingRegion2),
                                    sizeof(MappingRegion2),
                                    deviceAddress2);

    NN_LOG("DetachDeviceAddressSpace()\n");
    nn::dd::DetachDeviceAddressSpace(&device, nn::dd::DeviceName_Sata);

    NN_LOG("DestroyDeviceAddressSpace()\n");
    nn::dd::DestroyDeviceAddressSpace(&device);
}


#if 0   // SIGLO-69191
        // とりあえず、nn::dd::DeviceName_Ppcs1 が使用されていて競合が発生。
        // 今後、F/W でどのデバイスを未使用 or 将来的に使用することになるか
        // 不明なため、このテストは無効化しておく。

//-----------------------------------------------------------------------------
// 複数のデバイスにアタッチするテスト
//
//  TORIAEZU: アタッチ可能なデバイスに複数同時にアタッチしてみる
//
const nn::dd::DeviceName targetDeviceName[] =
{
    nn::dd::DeviceName_Sata,
    nn::dd::DeviceName_Ppcs1,
};

TEST(DeviceAddressSpace, AttachDeviceAddressSpaceToMultiDevicesTest)
{
    NN_LOG(NN_TEXT("複数のデバイスにアタッチするテスト\n"));

    nn::dd::DeviceAddressSpaceType   device;

    // ミラーリングの前準備
    NN_LOG("GetCurrentProcessHandle()\n");
    auto currentProcessHandle = nn::dd::GetCurrentProcessHandle();
    EXPECT_TRUE( currentProcessHandle != nn::os::InvalidNativeHandle );

    NN_LOG("CreateDeviceAddressSpace()\n");
    auto result = nn::dd::CreateDeviceAddressSpace(&device, AddressSpaceSize);
    EXPECT_TRUE( result.IsSuccess() );

    // インクリメンタル版のマップ操作
    uint64_t address  = reinterpret_cast<uint64_t>(MappingRegion1);
    nn::dd::DeviceVirtualAddress deviceAddress1 = deviceAddressTop1;

    TestIncrementalMapDas(&device, currentProcessHandle, address, sizeof(MappingRegion1), deviceAddress1, nn::dd::MemoryPermission_ReadWrite);

    // 下位 22bit Aligned 版のマップ操作
    nn::dd::DeviceVirtualAddress deviceAddress2 = deviceAddressTop2 + (reinterpret_cast<uint64_t>(MappingRegion2) & 0x3fffff);
    NN_LOG("MapDeviceAddressSpaceAligned() cpuAdrs=0x%p\n", MappingRegion2);
    NN_LOG("                               devAdrs=0x%p\n", deviceAddress2);
    result = nn::dd::MapDeviceAddressSpaceAligned(
                                    &device,
                                    currentProcessHandle,
                                    reinterpret_cast<uint64_t>(MappingRegion2),
                                    sizeof(MappingRegion2),
                                    deviceAddress2,
                                    nn::dd::MemoryPermission_ReadWrite);
    EXPECT_TRUE( result.IsSuccess() );

    // 複数のデバイスにアタッチする
    for (auto name : targetDeviceName)
    {
        NN_LOG("AttachDeviceAddressSpace(DeviceName=%d)\n", name);
        result = nn::dd::AttachDeviceAddressSpace(&device, name);
        EXPECT_TRUE( result.IsSuccess() );
    }

    // 複数のデバイスからデタッチする
    for (auto name : targetDeviceName)
    {
        NN_LOG("DetachDeviceAddressSpace(DeviceName=%d)\n", name);
        nn::dd::DetachDeviceAddressSpace(&device, name);
    }

    // ミラーリングの後処理
    NN_LOG("UnmapDeviceAddressSpace()\n");
    nn::dd::UnmapDeviceAddressSpace(&device,
                                    currentProcessHandle,
                                    reinterpret_cast<uint64_t>(MappingRegion2),
                                    sizeof(MappingRegion2),
                                    deviceAddress2);

    NN_LOG("DestroyDeviceAddressSpace()\n");
    nn::dd::DestroyDeviceAddressSpace(&device);
}
#endif

//-----------------------------------------------------------------------------
// デバイスアドレス空間機能の動作テスト５
//  Create したデバイスアドレス空間ハンドルを GetDeviceAddressSpaceHandle し、
//  それを AttachDeviceAddressSpaceHandle() したオブジェクトを使って
//  各種テストを行なう。
//
TEST(DeviceAddressSpace, NormalDeviceAddressSpaceHandleTest)
{
    nn::dd::DeviceAddressSpaceType  device1;
    nn::dd::DeviceAddressSpaceType  device2;

    NN_LOG("GetCurrentProcessHandle()\n");
    auto currentProcessHandle = nn::dd::GetCurrentProcessHandle();
    EXPECT_TRUE( currentProcessHandle != nn::os::InvalidNativeHandle );

    NN_LOG("CreateDeviceAddressSpace()\n");
    auto result = nn::dd::CreateDeviceAddressSpace(&device1, AddressSpaceSize);
    EXPECT_TRUE( result.IsSuccess() );

    // ハンドルを取得
    NN_LOG("GetDeviceAddressSpaceHandle()\n");
    auto handle = nn::dd::GetDeviceAddressSpaceHandle(&device1);

    // 同一プロセスなのでハンドル管理フラグは false にしておく
    NN_LOG("AttachDeviceAddressSpaceHandle()\n");
    nn::dd::AttachDeviceAddressSpaceHandle(&device2, handle, false);

    // デバイス空間へのアタッチ
    NN_LOG("AttachDeviceAddressSpace()\n");
    result = nn::dd::AttachDeviceAddressSpace(&device2, nn::dd::DeviceName_Sata);
    EXPECT_TRUE( result.IsSuccess() );

    // インクリメンタル版のマップ操作
    uint64_t address  = reinterpret_cast<uint64_t>(MappingRegion1);
    nn::dd::DeviceVirtualAddress deviceAddress1 = deviceAddressTop1;

    TestIncrementalMapDas(&device1, currentProcessHandle, address, sizeof(MappingRegion1), deviceAddress1, nn::dd::MemoryPermission_ReadWrite);

    // 下位 22bit Aligned 版のマップ操作
    nn::dd::DeviceVirtualAddress deviceAddress2 = deviceAddressTop2 + (reinterpret_cast<uint64_t>(MappingRegion2) & 0x3fffff);
    NN_LOG("MapDeviceAddressSpaceAligned() cpuAdrs=0x%p\n", MappingRegion2);
    NN_LOG("                               devAdrs=0x%p\n", deviceAddress2);
    result = nn::dd::MapDeviceAddressSpaceAligned(
                                    &device2,
                                    currentProcessHandle,
                                    reinterpret_cast<uint64_t>(MappingRegion2),
                                    sizeof(MappingRegion2),
                                    deviceAddress2,
                                    nn::dd::MemoryPermission_ReadWrite);
    EXPECT_TRUE( result.IsSuccess() );

    // ここからはデバイスアドレス空間にミラーリングされた状態

    // TORIAEZU:
    //  理想的には、ここで特定デバイスを操作して、
    //  指定したデバイスアドレス空間へのアクセスが正しく行なえることを
    //  テストすべき。汎用 DMA などのデバイスのアクセス方法が分かれば
    //  対応を検討する。

    // ミラーリングの後処理
    NN_LOG("UnmapDeviceAddressSpace()\n");
    nn::dd::UnmapDeviceAddressSpace(&device2,
                                    currentProcessHandle,
                                    reinterpret_cast<uint64_t>(MappingRegion2),
                                    sizeof(MappingRegion2),
                                    deviceAddress2);

    NN_LOG("DetachDeviceAddressSpace()\n");
    nn::dd::DetachDeviceAddressSpace(&device2, nn::dd::DeviceName_Sata);

    NN_LOG("DestroyDeviceAddressSpace()\n");
    nn::dd::DestroyDeviceAddressSpace(&device2);
    nn::dd::DestroyDeviceAddressSpace(&device1);

    NN_LOG("CloseProcessHandle()\n");
    nn::dd::CloseProcessHandle( currentProcessHandle );
}


#if 0   // SIGLO-69191
        // とりあえず、nn::dd::DeviceName_Ppcs1 が使用されていて競合が発生。
        // 今後、F/W でどのデバイスを未使用 or 将来的に使用することになるか
        // 不明なため、このテストは無効化しておく。

//-----------------------------------------------------------------------------
// SIGLO-11224
// nn::dd::MapDeviceAddressSpaceAligned() で、１つのメモリ領域を、
// 異なる２つ以上の DAS にマップできるかをテストする。
//
TEST(DeviceAddressSpace, MapToTwoDeviceAddressSpaceTest)
{
    NN_LOG(NN_TEXT("１つのメモリ領域を２つの異なる DAS に MapDeviceAddressSpaceAligned() する\n"));

    uintptr_t address = reinterpret_cast<uintptr_t>(MappingRegion2);
    size_t    size    = sizeof(MappingRegion2);

    nn::dd::DeviceAddressSpaceType   das1;
    nn::dd::DeviceAddressSpaceType   das2;

    // DAS 作成
    auto result = nn::dd::CreateDeviceAddressSpace(&das1, 0x100000000ul);
    EXPECT_TRUE( result.IsSuccess() );

    result = nn::dd::CreateDeviceAddressSpace(&das2, 0x100000000ul);
    EXPECT_TRUE( result.IsSuccess() );

    // アタッチ
    result = nn::dd::AttachDeviceAddressSpace(&das1, nn::dd::DeviceName_Sata);
    EXPECT_TRUE( result.IsSuccess() );

    result = nn::dd::AttachDeviceAddressSpace(&das2, nn::dd::DeviceName_Ppcs1);
    EXPECT_TRUE( result.IsSuccess() );

    // １つの領域を複数の DAS へマップする
    auto handle = nn::dd::GetCurrentProcessHandle();
    nn::dd::DeviceVirtualAddress dasAddress1 = 0x10000000ull + (static_cast<uint64_t>(address) & 0x3fffff);
    nn::dd::DeviceVirtualAddress dasAddress2 = 0x20000000ull + (static_cast<uint64_t>(address) & 0x3fffff);

    result = nn::dd::MapDeviceAddressSpaceAligned(&das1, handle, address, size, dasAddress1, nn::dd::MemoryPermission_ReadWrite);
    EXPECT_TRUE( result.IsSuccess() );

    result = nn::dd::MapDeviceAddressSpaceAligned(&das2, handle, address, size, dasAddress2, nn::dd::MemoryPermission_ReadWrite);
    EXPECT_TRUE( result.IsSuccess() );

    NN_LOG("DAS1: address=0x%08llx, size=0x%08zx\n", dasAddress1, size);
    NN_LOG("DAS2: address=0x%08llx, size=0x%08zx\n", dasAddress2, size);

    // アンマップ
    nn::dd::UnmapDeviceAddressSpace(&das1, handle, address, size, dasAddress1);
    nn::dd::UnmapDeviceAddressSpace(&das2, handle, address, size, dasAddress2);

    // デタッチ
    nn::dd::DetachDeviceAddressSpace(&das1, nn::dd::DeviceName_Sata);
    nn::dd::DetachDeviceAddressSpace(&das2, nn::dd::DeviceName_Ppcs1);

    // DAS 削除
    nn::dd::DestroyDeviceAddressSpace(&das1);
    nn::dd::DestroyDeviceAddressSpace(&das2);
}
#endif

//-----------------------------------------------------------------------------
//  SIGLO-16318, IAAA-1792
//  GPU プロセス用に用意した nn::dd::MapDeviceAddressSpaceNotAligned() の
//  呼出＆動作テスト。
//
//  ロジック的には NormalDeviceAddressSpaceTest1 のテストと同様で、
//  呼出す API を変えただけである。
//
TEST(DeviceAddressSpace, NormalDeviceAddressSpaceTest7)
{
    NN_LOG(NN_TEXT("SEQ1> Create -> Attach -> Map -> Unamp -> Detach -> Destroy のテスト\n"));

    nn::dd::DeviceAddressSpaceType  device;

    // ミラーリングの前準備
    NN_LOG("GetCurrentProcessHandle()\n");
    auto currentProcessHandle = nn::dd::GetCurrentProcessHandle();
    EXPECT_TRUE( currentProcessHandle != nn::os::InvalidNativeHandle );

    NN_LOG("CreateDeviceAddressSpace()\n");
    auto result = nn::dd::CreateDeviceAddressSpace(&device, AddressSpaceSize);
    EXPECT_TRUE( result.IsSuccess() );

    // 試しアタッチには今後も使われる可能性が低い Sata を使う
    NN_LOG("AttachDeviceAddressSpace()\n");
    result = nn::dd::AttachDeviceAddressSpace(&device, nn::dd::DeviceName_Sata);
    EXPECT_TRUE( result.IsSuccess() );

    NN_LOG("MapDeviceAddressSpaceNotAligned()\n");
    const nn::dd::DeviceVirtualAddress deviceAddress1 = 0x40001000;
    result = nn::dd::MapDeviceAddressSpaceNotAligned(
                                    &device,
                                    currentProcessHandle,
                                    reinterpret_cast<uint64_t>(MappingRegion1),
                                    sizeof(MappingRegion1),
                                    deviceAddress1,
                                    nn::dd::MemoryPermission_ReadWrite);
    EXPECT_TRUE( result.IsSuccess() );

    NN_LOG("MapDeviceAddressSpaceNotAligned()\n");
    const nn::dd::DeviceVirtualAddress deviceAddress2 = 0x80000000;
    result = nn::dd::MapDeviceAddressSpaceNotAligned(
                                    &device,
                                    currentProcessHandle,
                                    reinterpret_cast<uint64_t>(MappingRegion2),
                                    sizeof(MappingRegion2),
                                    deviceAddress2,
                                    nn::dd::MemoryPermission_ReadWrite);
    EXPECT_TRUE( result.IsSuccess() );

    // ここからはデバイスアドレス空間にミラーリングされた状態

    // TORIAEZU:
    //  理想的には、ここで特定デバイスを操作して、
    //  指定したデバイスアドレス空間へのアクセスが正しく行なえることを
    //  テストすべき。汎用 DMA などのデバイスのアクセス方法が分かれば
    //  対応を検討する。

    // ミラーリングの後処理
    NN_LOG("UnmapDeviceAddressSpace()\n");
    nn::dd::UnmapDeviceAddressSpace(&device,
                                    currentProcessHandle,
                                    reinterpret_cast<uint64_t>(MappingRegion1),
                                    sizeof(MappingRegion1),
                                    deviceAddress1);

    NN_LOG("UnmapDeviceAddressSpace()\n");
    nn::dd::UnmapDeviceAddressSpace(&device,
                                    currentProcessHandle,
                                    reinterpret_cast<uint64_t>(MappingRegion2),
                                    sizeof(MappingRegion2),
                                    deviceAddress2);

    NN_LOG("DetachDeviceAddressSpace()\n");
    nn::dd::DetachDeviceAddressSpace(&device, nn::dd::DeviceName_Sata);

    NN_LOG("DestroyDeviceAddressSpace()\n");
    nn::dd::DestroyDeviceAddressSpace(&device);
}


//-----------------------------------------------------------------------------

}}} // namespace nnt::dd::deviceAddressSpace

#endif  // デバイスアドレス空間がサポートされた環境でのみコードを有効化

