﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dauth/detail/dauth_AuthenticationAdaptor.h"

#include <cstdlib>
#include <cstring>
#include <string>

#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/dauth/dauth_Result.h>
#include <nn/dauth/dauth_Types.h>
#include <nn/dauth/detail/dauth_Result.h>
#include <nn/http/json/http_JsonErrorMap.h>
#include <nn/http/json/http_RapidJsonApi.h>
#include <nn/http/http_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_Execution.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_TFormatString.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include "testDauth_RapidJsonInputStream.h"

using namespace nn;

namespace {

const char Challenge[] = "A7Eb5LhwwDRrZ9HDyY7NGKM92VkWTXuOSqRROBQmBfU=";

const char ChallengeData[] = "9ypzx0QdHYHgl6nlTUmNrg==";

const char ChallengeResponseBase[] =
"{"
    "\"challenge\": \"%s\","
    "\"data\": \"%s\""
"}";

const char DeviceAuthenticationToken[] = "eyJ0eXAiOiJKV1QiLCJhbGciOiJSUzI1NiIsInR"
"5cCI6IkpXVCIsImFsZyI6IlJTMjU2Iiwia2lkIjoiNGFmNDAxN2QtOTk0Yi00M2RmLWJkMzUtNjhkOTI"
"wNzMxYThlIn0.eyJzdWIiOiI2MTAwMDAwMDAwMDAwMDIyIiwiaXNzIjoiZGF1dGgtdGQxLm5kYXMuc3J"
"2Lm5pbnRlbmRvLm5ldCIsImV4cCI6MTQ1MTk4MDY3NSwiaWF0IjoxNDUxODk0Mjc1LCJqdGkiOiIyYmU"
"4Mzg3ZS0xNjg5LTRlZTUtOTQ0Mi1hNDE1YzljMDhjMDAiLCJuaW50ZW5kbyI6eyJzbiI6IlhBVjAwMDA"
"wMDAwMzQ1IiwicGMiOiJIQUMtUy1aWlpaVihVU1opIiwiZHQiOiJOWCBWRGV2IDEifX0.FcaQR0xvKAD"
"Kvslej3u15fgWmt2Ivzw1eKXV85DNgr1KB-aYZVVpXBuTlWFisR77K0sqL5mtqrqKWv6DjUobTzcxVZV"
"KfD-QK8ByzK1dwua1TIEHhIIB_QjykhFHnU1imxGwgFENyS8lVVa-qxhJRye2mAfd_Ago980ZKBQyO-1"
"WkhI3Yr2codTnqlgtlBcaAvwsM4PDHUdHQFLBRXizYYUx2zE0oyedEYjabHiiaPkmnPwCHsWOrU2sXOn"
"lz501U7Pk5-x6SbRlHs5nx1tCuMe7k9tBPsSqaB2JtuHEkkK5NecajuVjUjRKj7Q9OO4liWS41A3G8m-"
"KGS07qgRN4w";

const char DeviceAuthenticationResponseBase[] =
"{"
    "\"expires_in\":%s,"
    "\"device_auth_token\":\"%s\""
"}";

const char EdgeToken[] = "exp=1520931025~acl=%2F%2A~hmac=xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx";

const char EdgeTokenAuthenticationResponseBase[] =
"{"
    "\"expires_in\":%s,"
    "\"dtoken\":\"%s\""
"}";

const char ErrorCodeResponseBase[] =
"{"
    "\"errors\" : ["
        "{"
            "\"code\" : \"%s\","
            "\"message\" : \"%s\""
        "}"
    "]"
"}";

NN_STATIC_ASSERT(sizeof(DeviceAuthenticationToken) <= (dauth::DeviceAuthenticationTokenLengthMax + 1));

//!----------------------------------------------------------------------------
void VerifyExpectMatch(const char* pExpectValue, const char* pVerifyValue) NN_NOEXCEPT
{
    const auto sizeActual = std::strlen(pVerifyValue);
    EXPECT_EQ(std::strlen(pExpectValue), sizeActual);
    EXPECT_EQ(0, std::memcmp(pExpectValue, pVerifyValue, sizeActual));
}
void VerifyExpectMatch(const void* pExpectValue, const void* pVerifyValue, size_t size) NN_NOEXCEPT
{
    EXPECT_EQ(0, std::memcmp(pExpectValue, pVerifyValue, size));
}

//!----------------------------------------------------------------------------
Result TestDeviceAuthentication(const char* pExpectToken, const char* pSourceJson) NN_NOEXCEPT
{
    ::nnt::dauth::MemoryInputStreamForRapidJson stream;
    stream.Set(pSourceJson, std::strlen(pSourceJson));

    NN_ALIGNAS(std::alignment_of<std::max_align_t>::value) char pResponse[dauth::RequiredBufferSizeForDeviceAuthenticationToken] = {'\0'};

    dauth::detail::DeviceAuthenticationAdaptor adaptor(pResponse, sizeof(pResponse));
    nn::util::Cancelable* pCancelable = nullptr;
    NN_RESULT_DO(::nn::http::json::ImportJsonByRapidJson<::nn::http::json::DefaultJsonErrorMap>(adaptor, stream, pCancelable));
    NN_RESULT_DO(adaptor.Adapt());

    VerifyExpectMatch(pExpectToken, pResponse);
    NN_RESULT_SUCCESS;
}

//!----------------------------------------------------------------------------
Result TestEdgeTokenAuthentication(const char* pExpectToken, const char* pSourceJson) NN_NOEXCEPT
{
    ::nnt::dauth::MemoryInputStreamForRapidJson stream;
    stream.Set(pSourceJson, std::strlen(pSourceJson));

    NN_ALIGNAS(std::alignment_of<std::max_align_t>::value) char pResponse[dauth::RequiredBufferSizeForEdgeToken] = {'\0'};

    dauth::detail::EdgeTokenAuthenticationAdaptor adaptor(pResponse, sizeof(pResponse));
    nn::util::Cancelable* pCancelable = nullptr;
    NN_RESULT_DO(::nn::http::json::ImportJsonByRapidJson<::nn::http::json::DefaultJsonErrorMap>(adaptor, stream, pCancelable));
    NN_RESULT_DO(adaptor.Adapt());

    VerifyExpectMatch(pExpectToken, pResponse);
    NN_RESULT_SUCCESS;
}

//!----------------------------------------------------------------------------
Result TestChallenge(const char* pExpectChallenge, const char* pExpectData, const char* pSourceJson) NN_NOEXCEPT
{
    ::nnt::dauth::MemoryInputStreamForRapidJson stream;
    stream.Set(pSourceJson, std::strlen(pSourceJson));

    dauth::detail::ChallengeAdaptor adaptor;
    nn::util::Cancelable* pCancelable = nullptr;
    NN_RESULT_DO(::nn::http::json::ImportJsonByRapidJson<::nn::http::json::DefaultJsonErrorMap>(adaptor, stream, pCancelable));
    NN_RESULT_DO(adaptor.Adapt());

    size_t decodedLength;
    uint8_t keySource[nn::spl::KeySourceSize];
    auto b64Error = util::Base64::FromBase64String(&decodedLength, keySource, sizeof(keySource), pExpectData, util::Base64::Mode_UrlSafe);
    NN_ABORT_UNLESS(b64Error == util::Base64::Status_Success);
    NN_ABORT_UNLESS(decodedLength == sizeof(keySource));
    VerifyExpectMatch(keySource, adaptor.GetKeySourceRef().data, sizeof(keySource));
    VerifyExpectMatch(pExpectChallenge, adaptor.GetChallengeRef().data);
    NN_RESULT_SUCCESS;
}

//!----------------------------------------------------------------------------
struct ErrorContextPrivate
{
    Result  expectResult;
    const char errorCode[8];
    const char message[128];
};

const ErrorContextPrivate g_TestErrorResponseCase[] = {
    {dauth::ResultNdasStatusNo0004(), "0004", "Unauthorized device."},
    {dauth::ResultNdasStatusNo0007(), "0007", "System updated is required."},
    {dauth::ResultNdasStatusNo0008(), "0008", "Device has been banned."},
    {dauth::ResultNdasStatusNo0009(), "0009", "Internal Server Error."},
    {dauth::ResultNdasStatusNo0010(), "0010", "Under Maintenance."},
    {dauth::ResultNdasStatusNo0011(), "0011", "Invalid client."},
    {dauth::ResultNdasStatusNo0012(), "0012", "API not found."},
    {dauth::ResultNdasStatusNo0013(), "0013", "Invalid certificate."},
    {dauth::ResultNdasStatusNo0014(), "0014", "Invalid parameter in request."},
    {dauth::ResultNdasStatusNo0015(), "0015", "Invalid parameter in request."},

    // ResultInvalidServerData が返されるケース: 廃棄エラーレスポンス && HTTP 200
    {dauth::detail::ResultInvalidNdasData(), "0003", "Not Implement."},

    {ResultSuccess(), "", ""}
};

} // ::~unnamed

//!----------------------------------------------------------------------------
extern "C" void nninitStartup()
{
    NN_FUNCTION_LOCAL_STATIC(NN_ALIGNAS(os::MemoryPageSize) uint8_t, s_MallocBuffer, [16 * 1024 * 1024]);

    nn::init::InitializeAllocator(s_MallocBuffer, sizeof(s_MallocBuffer));
}

//!----------------------------------------------------------------------------
TEST(TestForAdaptor, DeviceAuthentication)
{
    char jsonSourceBuffer[4 * 1024];

    // 正常レスポンス ( DeviceAuthentication )
    ASSERT_TRUE(sizeof(jsonSourceBuffer) > util::SNPrintf(jsonSourceBuffer, sizeof(jsonSourceBuffer), DeviceAuthenticationResponseBase, "86400", DeviceAuthenticationToken));
    NNT_ASSERT_RESULT_SUCCESS(TestDeviceAuthentication(DeviceAuthenticationToken, jsonSourceBuffer));

    // エラーコードレスポンス ( DeviceAuthentication )
    for (auto* pCases = g_TestErrorResponseCase; pCases->expectResult.IsFailure(); ++pCases)
    {
        ASSERT_TRUE(sizeof(jsonSourceBuffer) > util::SNPrintf(jsonSourceBuffer, sizeof(jsonSourceBuffer), ErrorCodeResponseBase, pCases->errorCode, pCases->message));
        EXPECT_EQ(pCases->expectResult.GetInnerValueForDebug(), TestDeviceAuthentication(DeviceAuthenticationToken, jsonSourceBuffer).GetInnerValueForDebug());
    }
}

//!----------------------------------------------------------------------------
TEST(TestForAdaptor, EdgeTokenAuthentication)
{
    char jsonSourceBuffer[4 * 1024];

    // 正常レスポンス ( EdgeTokenAuthentication )
    ASSERT_TRUE(sizeof(jsonSourceBuffer) > util::SNPrintf(jsonSourceBuffer, sizeof(jsonSourceBuffer), EdgeTokenAuthenticationResponseBase, "86400", EdgeToken));
    NNT_ASSERT_RESULT_SUCCESS(TestEdgeTokenAuthentication(EdgeToken, jsonSourceBuffer));

    // エラーコードレスポンス ( EdgeTokenAuthentication )
    for (auto* pCases = g_TestErrorResponseCase; pCases->expectResult.IsFailure(); ++pCases)
    {
        ASSERT_TRUE(sizeof(jsonSourceBuffer) > util::SNPrintf(jsonSourceBuffer, sizeof(jsonSourceBuffer), ErrorCodeResponseBase, pCases->errorCode, pCases->message));
        EXPECT_EQ(pCases->expectResult.GetInnerValueForDebug(), TestEdgeTokenAuthentication(EdgeToken, jsonSourceBuffer).GetInnerValueForDebug());
    }
}

//!----------------------------------------------------------------------------
TEST(TestForAdaptor, Challenge)
{
    char jsonSourceBuffer[4 * 1024];

    // 正常レスポンス ( Challenge )
    ASSERT_TRUE(sizeof(jsonSourceBuffer) > util::SNPrintf(jsonSourceBuffer, sizeof(jsonSourceBuffer), ChallengeResponseBase, Challenge, ChallengeData));
    NNT_ASSERT_RESULT_SUCCESS(TestChallenge(Challenge, ChallengeData, jsonSourceBuffer));

    // エラーコードレスポンス ( Challenge )
    for (auto* pCases = g_TestErrorResponseCase; pCases->expectResult.IsFailure(); ++pCases)
    {
        ASSERT_TRUE(sizeof(jsonSourceBuffer) > util::SNPrintf(jsonSourceBuffer, sizeof(jsonSourceBuffer), ErrorCodeResponseBase, pCases->errorCode, pCases->message));
        EXPECT_EQ(pCases->expectResult.GetInnerValueForDebug(), TestChallenge(Challenge, ChallengeData, jsonSourceBuffer).GetInnerValueForDebug());
    }
}
