﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/crypto.h>
#include <nn/crypto/crypto_Aes128GcmEncryptor.h>
#include <nn/crypto/crypto_Aes128GcmDecryptor.h>
#include <nn/crypto/crypto_Aes128XtsEncryptor.h>
#include <nn/crypto/crypto_Aes128XtsDecryptor.h>

#include <nnt/nntest.h>

/*
  計測ケースの追加手順

  (1) AES は初期化時の引数が異なり、それをそろえるために以下のインターフェースを持つ実装クラスを用意する
      void Initialize()
      void Update(uint8_t* pOutput, size_t outputSize, const uint8_t* pInput, size_t inputSize)

  (2) TeamCity のプロット用の文字列を返すインターフェースを用意する
      const char* GetName()

  (3) パフォーマンスチェック用に以下の定数を定義（定義の実体は testCrypto_PerformanceTarget.cpp に記載）
      static const uint64_t TargetPerformance        // Out-Place 処理時のターゲット
      static const uint64_t TargetPerformanceInPlace // In-Place 処理時のターゲット

  (4) typedef ::testing::Types<*> AesCipherTypes;
      の行の <> の中に上記のクラス名を追加する
 */


// 暗号化：暗号利用モードなし
class AesEncryption
{
public:
    static const uint64_t TargetPerformance;
    static const uint64_t TargetPerformanceInPlace;

public:
    void Initialize()
    {
        uint8_t testKey[16] = {0};
        aesEnc.Initialize(testKey, sizeof(testKey));
    }

    void Update(uint8_t* pOutput, size_t outputSize, const uint8_t* pInput, size_t inputSize)
    {
        NN_UNUSED(inputSize);

        int numBlocks = static_cast<int>(outputSize / nn::crypto::AesEncryptor128::BlockSize);
        while (numBlocks--)
        {
            aesEnc.EncryptBlock(pOutput, nn::crypto::AesEncryptor128::BlockSize,
                                pInput, nn::crypto::AesEncryptor128::BlockSize);
            pOutput += nn::crypto::AesEncryptor128::BlockSize;
            pInput += nn::crypto::AesEncryptor128::BlockSize;
        }
    }

    const char* GetName()
    {
        return "AES-128-ENC";
    }

private:
    nn::crypto::AesEncryptor128 aesEnc;
};

// 復号化：暗号利用モードなし
class AesDecryption
{
public:
    static const uint64_t TargetPerformance;
    static const uint64_t TargetPerformanceInPlace;

public:
    void Initialize()
    {
        uint8_t testKey[16] = {0};
        aesDec.Initialize(testKey, sizeof(testKey));
    }

    void Update(uint8_t* pOutput, size_t outputSize, const uint8_t* pInput, size_t inputSize)
    {
        NN_UNUSED(inputSize);

        int numBlocks = static_cast<int>(outputSize / nn::crypto::AesEncryptor128::BlockSize);
        while (numBlocks--)
        {
            aesDec.DecryptBlock(pOutput, nn::crypto::AesEncryptor128::BlockSize,
                                pInput, nn::crypto::AesEncryptor128::BlockSize);
            pOutput += nn::crypto::AesEncryptor128::BlockSize;
            pInput += nn::crypto::AesEncryptor128::BlockSize;
        }
    }

    const char* GetName()
    {
        return "AES-128-DEC";
    }

private:
    nn::crypto::AesDecryptor128 aesDec;
};

// 暗号化：CBC モード
class AesCbcEncryption
{
public:
    static const uint64_t TargetPerformance;
    static const uint64_t TargetPerformanceInPlace;

public:
    void Initialize()
    {
        uint8_t testKey[16] = {0};
        uint8_t testIv[16] = {0};
        aesCbcEnc.Initialize(testKey, sizeof(testKey), testIv, sizeof(testIv));
    }

    void Update(uint8_t* pOutput, size_t outputSize, const uint8_t* pInput, size_t inputSize)
    {
        aesCbcEnc.Update(pOutput, outputSize, pInput, inputSize);
    }

    const char* GetName()
    {
        return "AES-128-CBC-ENC";
    }

private:
    nn::crypto::Aes128CbcEncryptor aesCbcEnc;
};

// 復号化：CBC モード
class AesCbcDecryption
{
public:
    static const uint64_t TargetPerformance;
    static const uint64_t TargetPerformanceInPlace;

public:
    void Initialize()
    {
        uint8_t testKey[16] = {0};
        uint8_t testIv[16] = {0};
        aesCbcDec.Initialize(testKey, sizeof(testKey), testIv, sizeof(testIv));
    }

    void Update(uint8_t* pOutput, size_t outputSize, const uint8_t* pInput, size_t inputSize)
    {
        aesCbcDec.Update(pOutput, outputSize, pInput, inputSize);
    }

    const char* GetName()
    {
        return "AES-128-CBC-DEC";
    }

private:
    nn::crypto::Aes128CbcDecryptor aesCbcDec;
};

// 暗号化：CTR モード
class AesCtrEncryption
{
public:
    static const uint64_t TargetPerformance;
    static const uint64_t TargetPerformanceInPlace;

public:
    void Initialize()
    {
        uint8_t testKey[16] = {0};
        uint8_t testIv[16] = {0};
        aesCtrEnc.Initialize(testKey, sizeof(testKey), testIv, sizeof(testIv));
    }

    void Update(uint8_t* pOutput, size_t outputSize, const uint8_t* pInput, size_t inputSize)
    {
        aesCtrEnc.Update(pOutput, outputSize, pInput, inputSize);
    }

    const char* GetName()
    {
        return "AES-128-CTR-ENC";
    }

private:
    nn::crypto::Aes128CtrEncryptor aesCtrEnc;
};

// 復号化：CTR モード
class AesCtrDecryption
{
public:
    static const uint64_t TargetPerformance;
    static const uint64_t TargetPerformanceInPlace;

public:
    void Initialize()
    {
        uint8_t testKey[16] = {0};
        uint8_t testIv[16] = {0};
        aesCtrDec.Initialize(testKey, sizeof(testKey), testIv, sizeof(testIv));
    }

    void Update(uint8_t* pOutput, size_t outputSize, const uint8_t* pInput, size_t inputSize)
    {
        aesCtrDec.Update(pOutput, outputSize, pInput, inputSize);
    }

    const char* GetName()
    {
        return "AES-128-CTR-DEC";
    }

private:
    nn::crypto::Aes128CtrDecryptor aesCtrDec;
};

//暗号化：XTS モード
class AesXtsEncryption
{
public:
    static const uint64_t TargetPerformance;
    static const uint64_t TargetPerformanceInPlace;

public:
    void Initialize()
    {
        uint8_t testKey1[16] = {0};
        uint8_t testKey2[16] = {0};
        uint8_t testTweak[16] = {0};
        aesXtsEnc.Initialize(testKey1, testKey2, sizeof(testKey1),
                             testTweak, sizeof(testTweak));
    }

    void Update(uint8_t* pOutput, size_t outputSize, const uint8_t* pInput, size_t inputSize)
    {
        aesXtsEnc.Update(pOutput, outputSize, pInput, inputSize);
    }

    const char* GetName()
    {
        return "AES-128-XTS-ENC";
    }

private:
    nn::crypto::Aes128XtsEncryptor aesXtsEnc;
};

// 復号化：XTS モード
class AesXtsDecryption
{
public:
    static const uint64_t TargetPerformance;
    static const uint64_t TargetPerformanceInPlace;

public:
    void Initialize()
    {
        uint8_t testKey1[16] = {0};
        uint8_t testKey2[16] = {0};
        uint8_t testTweak[16] = {0};
        aesXtsDec.Initialize(testKey1, testKey2, sizeof(testKey1),
                             testTweak, sizeof(testTweak));
    }

    void Update(uint8_t* pOutput, size_t outputSize, const uint8_t* pInput, size_t inputSize)
    {
        aesXtsDec.Update(pOutput, outputSize, pInput, inputSize);
    }

    const char* GetName()
    {
        return "AES-128-XTS-DEC";
    }

private:
    nn::crypto::Aes128XtsDecryptor aesXtsDec;
};

// 暗号化：GCM モード
class AesGcmEncryption
{
public:
    static const uint64_t TargetPerformance;
    static const uint64_t TargetPerformanceInPlace;

public:
    void Initialize()
    {
        uint8_t testKey[16] = {0};
        uint8_t testIv[16] = {0};
        aesGcmEnc.Initialize(testKey, sizeof(testKey), testIv, sizeof(testIv));
    }

    void Update(uint8_t* pOutput, size_t outputSize, const uint8_t* pInput, size_t inputSize)
    {
        aesGcmEnc.Update(pOutput, outputSize, pInput, inputSize);
    }

    const char* GetName()
    {
        return "AES-128-GCM-ENC";
    }

private:
    nn::crypto::Aes128GcmEncryptor aesGcmEnc;
};

// 復号化：GCM モード
class AesGcmDecryption
{
public:
    static const uint64_t TargetPerformance;
    static const uint64_t TargetPerformanceInPlace;

public:
    void Initialize()
    {
        uint8_t testKey[16] = {0};
        uint8_t testIv[16] = {0};
        aesGcmDec.Initialize(testKey, sizeof(testKey), testIv, sizeof(testIv));
    }

    void Update(uint8_t* pOutput, size_t outputSize, const uint8_t* pInput, size_t inputSize)
    {
        aesGcmDec.Update(pOutput, outputSize, pInput, inputSize);
    }

    const char* GetName()
    {
        return "AES-128-GCM-DEC";
    }

private:
    nn::crypto::Aes128GcmDecryptor aesGcmDec;
};

// 暗号化：CCM モード
class AesCcmEncryption
{
public:
    static const uint64_t TargetPerformance;
    static const uint64_t TargetPerformanceInPlace;

public:
    void Initialize()
    {
        uint8_t testKey[16] = {0};
        uint8_t testNonce[7] = {0};
        // TODO: remove hard-coded data size
        aesCcmEnc.Initialize(testKey, sizeof(testKey),
                             testNonce, sizeof(testNonce),
                             0, 256 * 1024 * 1024, 16);
    }

    void Update(uint8_t* pOutput, size_t outputSize, const uint8_t* pInput, size_t inputSize)
    {
        aesCcmEnc.Update(pOutput, outputSize, pInput, inputSize);
    }

    const char* GetName()
    {
        return "AES-128-CCM-ENC";
    }

private:
    nn::crypto::Aes128CcmEncryptor aesCcmEnc;
};

// 復号化：CCM モード
class AesCcmDecryption
{
public:
    static const uint64_t TargetPerformance;
    static const uint64_t TargetPerformanceInPlace;

public:
    void Initialize()
    {
        uint8_t testKey[16] = {0};
        uint8_t testNonce[7] = {0};
        // TODO: remove hard-coded data size
        aesCcmDec.Initialize(testKey, sizeof(testKey),
                             testNonce, sizeof(testNonce),
                             0, 256 * 1024 * 1024, 16);
    }

    void Update(uint8_t* pOutput, size_t outputSize, const uint8_t* pInput, size_t inputSize)
    {
        aesCcmDec.Update(pOutput, outputSize, pInput, inputSize);
    }

    const char* GetName()
    {
        return "AES-128-CCM-DEC";
    }

private:
    nn::crypto::Aes128CcmDecryptor aesCcmDec;
};

/**
  @brief  実装クラスを定義した型です。
 */
typedef ::testing::Types<AesEncryption, AesDecryption,
                         AesCbcEncryption, AesCbcDecryption,
                         AesCtrEncryption, AesCtrDecryption,
                         AesXtsEncryption, AesXtsDecryption,
                         AesGcmEncryption, AesGcmDecryption,
                         AesCcmEncryption, AesCcmDecryption> AesCipherTypes;
