﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/crypto/crypto_Aes128GcmDecryptor.h>
#include <nn/crypto/crypto_Aes128GcmEncryptor.h>
#include <cstring>

#include <nn/nn_Log.h>
#include <nnt/nntest.h>
#include "GcmParser.h"
#include "ProfileCheck.h"
#include "OctetString.h"


class Aes128GcmEncryptParser : public GcmEncryptParser<nn::crypto::AesEncryptor128>
{
    virtual void TestUnit() override
    {
        ASSERT_EQ(m_UnitTokens.count(Auth), 1);
        ASSERT_EQ(m_UnitTokens.count(Key), 1);
        ASSERT_EQ(m_UnitTokens.count(Plain), 1);
        ASSERT_EQ(m_UnitTokens.count(Cipher), 1);
        ASSERT_EQ(m_UnitTokens.count(Tag), 1);
        ASSERT_EQ(m_UnitTokens.count(IV), 1);

        const std::string& authStr = m_UnitTokens[Auth];
        const std::string& keyStr = m_UnitTokens[Key];
        const std::string& plainStr = m_UnitTokens[Plain];
        const std::string& cipherStr = m_UnitTokens[Cipher];
        const std::string& tagStr = m_UnitTokens[Tag];
        const std::string& ivStr = m_UnitTokens[IV];

        {
            std::string resCipher(plainStr.size(), char(0));
            std::string resTag(nn::crypto::Aes128GcmEncryptor::MacSize, char(0));

            nn::crypto::EncryptAes128Gcm(const_cast<char*>(resCipher.c_str()), resCipher.size(),
                                         const_cast<char*>(resTag.c_str()), resTag.size(),
                                         keyStr.c_str(), keyStr.size(),
                                         ivStr.c_str(), ivStr.size(),
                                         plainStr.c_str(), plainStr.size(),
                                         authStr.c_str(), authStr.size());

            // truncate the mac in order to compare by '==' operator
            resTag.resize(tagStr.size());

            ASSERT_TRUE(resCipher == cipherStr);
            ASSERT_TRUE(resTag == tagStr);
        }

        // Same as above, but perform in-place encryption
        {
            std::string msg(plainStr.begin(), plainStr.end());
            std::string resTag(nn::crypto::Aes128GcmEncryptor::MacSize, char(0));

            nn::crypto::EncryptAes128Gcm(const_cast<char*>(msg.c_str()), msg.size(),
                                         const_cast<char*>(resTag.c_str()), resTag.size(),
                                         keyStr.c_str(), keyStr.size(),
                                         ivStr.c_str(), ivStr.size(),
                                         msg.c_str(), msg.size(),
                                         authStr.c_str(), authStr.size());

            // truncate the mac in order to compare by '==' operator
            resTag.resize(tagStr.size());

            ASSERT_TRUE(msg == cipherStr);
            ASSERT_TRUE(resTag == tagStr);
        }

        // Run the test using the lower level API
        GcmEncryptParser<nn::crypto::AesEncryptor128>::TestUnit();
    }
};

class Aes128GcmDecryptParser : public GcmDecryptParser<nn::crypto::AesEncryptor128>
{
    virtual void TestUnit() override
    {
        const std::string& authStr = m_UnitTokens[Auth];
        const std::string& keyStr = m_UnitTokens[Key];
        const std::string& plainStr = m_UnitTokens[Plain];
        const std::string& cipherStr = m_UnitTokens[Cipher];
        const std::string& tagStr = m_UnitTokens[Tag];
        const std::string& ivStr = m_UnitTokens[IV];
        bool expectedOk = !Has( Fail );

        {
            std::string resPlain(cipherStr.size(), char(0));
            char tag[nn::crypto::Aes128GcmDecryptor::MacSize];

            nn::crypto::DecryptAes128Gcm(const_cast<char*>(resPlain.c_str()), resPlain.size(),
                                         tag, sizeof(tag),
                                         keyStr.c_str(), keyStr.size(),
                                         ivStr.c_str(), ivStr.size(),
                                         cipherStr.c_str(), cipherStr.size(),
                                         authStr.c_str(), authStr.size());
            if (expectedOk != (std::memcmp( tag, tagStr.c_str(), tagStr.size() ) == 0))
            {
                NN_LOG("Klen : %d\nIvlen: %d\nClen : %d\nAlen : %d\n", keyStr.size(), ivStr.size(), cipherStr.size(), authStr.size());
            }

            EXPECT_EQ( expectedOk, std::memcmp( tag, tagStr.c_str(), tagStr.size() ) == 0 );
            if ( expectedOk )
            {
                EXPECT_TRUE(resPlain == plainStr);
            }
        }


        // Same as above, by perform in-place decryption
        {
            std::string msg(cipherStr.begin(), cipherStr.end());
            char tag[nn::crypto::Aes128GcmDecryptor::MacSize];

            nn::crypto::DecryptAes128Gcm(const_cast<char*>(msg.c_str()), msg.size(),
                                         tag, sizeof(tag),
                                         keyStr.c_str(), keyStr.size(),
                                         ivStr.c_str(), ivStr.size(),
                                         cipherStr.c_str(), cipherStr.size(),
                                         authStr.c_str(), authStr.size());

            EXPECT_EQ( expectedOk, std::memcmp( tag, tagStr.c_str(), tagStr.size() ) == 0 );
            if ( expectedOk )
            {
                EXPECT_TRUE(msg == plainStr);
            }
        }

        // Run the test using the lower level API
        GcmDecryptParser<nn::crypto::AesEncryptor128>::TestUnit();
    }
};

TEST(Aes128Gcm, Decrypt128_rsp)
{
    Aes128GcmDecryptParser tester;
    tester.TestFile("aes_gcm/gcmtestvectors/gcmDecrypt128.rsp", 7875);
}

TEST(Aes128Gcm, Encrypt128_rsp)
{
    Aes128GcmEncryptParser tester;
    tester.TestFile("aes_gcm/gcmtestvectors/gcmEncryptExtIV128.rsp", 7875);
}

TEST(Aes128Gcm, 4K_Profile)
{
    OctetString key( "4a42d905e2cbb701a21e61dced668e0d" );
    OctetString iv( "e3" );
    OctetString ct(
        "d035eaf429f3d46244ec4b0edbdf7e9285068c0eaea733232b3164a9eb6075a5e037d709d54e9c33f5f0fe3dcae62769daa7fc" );
    ct.resize(4096);
    std::string out;
    out.resize( ct.size() );
    OctetString tagExpected( "da6c42d3092c7bfb8fff5a216c830e44" );

    nn::crypto::Aes128GcmDecryptor gcm;
    {
        PROFILE_CHECK_AndroidTX1("AES-128-GCM-Init");
        static const int Repeat = 10;
        for ( int i = Repeat; i--; )
        {
            gcm.Initialize( key.c_str(), key.size(), nullptr, 0 );
        }
    }
    bool tagOk = true;
    {
        static const int Repeat = 10;
        PROFILE_CHECK_AndroidTX1("AES-128-GCM-Decrypt4K",40,-1,0,0,Repeat);
        for ( int i = Repeat; i--; )
        {
            gcm.Reset( iv.c_str(), iv.size() );
            gcm.Update( (void*)out.c_str(), out.size(), ct.c_str(), ct.size() );
            char tag[gcm.MacSize];
            gcm.GetMac(tag,sizeof(tag));
            tagOk &= std::memcmp( tag, tagExpected.c_str(), tagExpected.size() ) == 0;
        }
    }
    ASSERT_TRUE( tagOk );
    OctetString pt(
        "d9cc7c65bcb64884d8156687d967d74618775b25285bb78b99d995abc0adc3b9c0279407cae09787c8218b2c144c3678fc7ac7");
    ASSERT_FALSE(memcmp( out.c_str(), pt.c_str(), pt.size()));

    pt.resize(ct.size());
    char tagOut[nn::crypto::Aes128GcmEncryptor::MacSize];
    {
        nn::crypto::Aes128GcmEncryptor gcm2;// = (nn::crypto::Aes128GcmEncryptor&)gcm;
        gcm2.Initialize( key.c_str(), key.size(), nullptr, 0 );
        static const int Repeat = 10;
        PROFILE_CHECK_AndroidTX1("AES-128-GCM-Encrypt4K",40,-1,0,0,Repeat);
        for ( int i = Repeat; i--; )
        {
            gcm2.Reset( iv.c_str(), iv.size() );
            gcm2.Update( (void*)out.c_str(), out.size(), pt.c_str(), pt.size() );
            gcm2.GetMac( tagOut, sizeof(tagOut) );
        }
    }
    ASSERT_FALSE(memcmp( tagOut,
                         OctetString("d69c0f3df24793c22773f3ac26d1cbc3"), sizeof(tagOut)));
    ASSERT_FALSE(memcmp( out.c_str(), ct.c_str(), 51));
}

TEST(Aes128Gcm, Encrypt_Hardcoded_Simple)
{
    const char* str =
        "\
Key = c939cc13397c1d37de6ae0e1cb7c423c\n\
IV = b3d8cc017cbb89b39e0f67e2\n\
PT = c3b3c41f113a31b73d9a5cd432103069\n\
AAD = 24825602bd12a984e0092d3e448eda5f\n\
CT = 93fe7d9e9bfd10348a5606e5cafa7354\n\
Tag = 0032a1dc85f1c9786925a2e71d8272dd\n\
";
    Aes128GcmEncryptParser tester;
    tester.TestString(str);
}

TEST(Aes128Gcm, Encrypt_Hardcoded_Complete)
{
    const char* str =
        "\
Key = caf492887729e5b88d87c207bffc82b1\n\
IV = c2ad4b6629d0294f1067500839366a1664344590f00e0eb1b6158b93f9f4d694cfca7c0e24d331924737fe7aab4bb69b72c6eeba3b3a2ba2d5448e2ff774495a058c50851fba51c713f82754d13b23ec7b956218e2441ffb6f419d0297ae1b93367598fe0e470cd51308e967dd222a8dbc9f96ef5235b9fa1c3750015415381e\n\
PT = 460cdc1560eb9fa5a5e5adec561f90afb6ec25527a194c3f62bb8a0d0c8d0af5df6e509240e673065583a02d06809dbb8d5111\n\
AAD = b3e7cd3a9c929539ba1839bc4c10975a338764db\n\
CT = 90ba807fbb0d32453e766f35d3591f4928709f8c9f9d644b4b9f93fb885905cae6e54f52671baee9796d20e3493c7d3dd4a896\n\
Tag = 1016be04a039829ac3ab552b07f00068\n\
";
    Aes128GcmEncryptParser tester;
    tester.TestString(str);
}
