﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/crypto/crypto_AesEncryptor.h>
#include <nn/crypto/crypto_GcmEncryptor.h>
#include <nn/crypto/crypto_GcmDecryptor.h>
#include <nn/crypto/crypto_Aes128GcmEncryptor.h>
#include <nn/crypto/crypto_Aes128GcmDecryptor.h>

#include "testCrypto_Util.h"

struct AesGcmTestVector
{
    nn::Bit8  key[32];
    size_t    keySize;
    nn::Bit8  iv[60];
    size_t    ivSize;
    nn::Bit8  aad[256];
    size_t    aadSize;
    nn::Bit8  plainText[160];
    size_t    textSize;
    nn::Bit8  cipherText[160];
    nn::Bit8  mac[16];
    size_t    macSize;
};

/* http://csrc.nist.gov/groups/ST/toolkit/BCM/documents/proposedmodes/gcm/gcm-revised-spec.pdf
   に掲載されているテストベクトルから抜粋 */
const AesGcmTestVector aesGcmTestVectors[] =
{
    // #1) Klen = 16, IVlen = 12, Alen = 0, and Plen = 0 (平文も AAD も入力なし)
    {
        {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
        16,
        {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
        12,
        {0},
        0,
        {0},
        0,
        {0},
        {0x58, 0xe2, 0xfc, 0xce, 0xfa, 0x7e, 0x30, 0x61, 0x36, 0x7f, 0x1d, 0x57, 0xa4, 0xe7, 0x45, 0x5a},
        16,
    },

    // #2) Klen = 16, IVlen = 12,  Alen = 0, and Plen = 16 (平文の入力のみ)
    {
        {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
        16,
        {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
        12,
        {0},
        0,
        {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
        16,
        {0x03, 0x88, 0xda, 0xce, 0x60, 0xb6, 0xa3, 0x92, 0xf3, 0x28, 0xc2, 0xb9, 0x71, 0xb2, 0xfe, 0x78},
        {0xab, 0x6e, 0x47, 0xd4, 0x2c, 0xec, 0x13, 0xbd, 0xf5, 0x3a, 0x67, 0xb2, 0x12, 0x57, 0xbd, 0xdf},
        16,
    },

    // #3) Klen = 16, IVlen = 12,  Alen = 0, and Plen = 64 (平文サイズがブロックサイズの整数倍)
    {
        {0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c, 0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08},
        16,
        {0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad, 0xde, 0xca, 0xf8, 0x88},
        12,
        {0},
        0,
        {0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5, 0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
         0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda, 0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
         0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53, 0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
         0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57, 0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55},
        64,
        {0x42, 0x83, 0x1e, 0xc2, 0x21, 0x77, 0x74, 0x24, 0x4b, 0x72, 0x21, 0xb7, 0x84, 0xd0, 0xd4, 0x9c,
         0xe3, 0xaa, 0x21, 0x2f, 0x2c, 0x02, 0xa4, 0xe0, 0x35, 0xc1, 0x7e, 0x23, 0x29, 0xac, 0xa1, 0x2e,
         0x21, 0xd5, 0x14, 0xb2, 0x54, 0x66, 0x93, 0x1c, 0x7d, 0x8f, 0x6a, 0x5a, 0xac, 0x84, 0xaa, 0x05,
         0x1b, 0xa3, 0x0b, 0x39, 0x6a, 0x0a, 0xac, 0x97, 0x3d, 0x58, 0xe0, 0x91, 0x47, 0x3f, 0x59, 0x85},
        {0x4d, 0x5c, 0x2a, 0xf3, 0x27, 0xcd, 0x64, 0xa6, 0x2c, 0xf3, 0x5a, 0xbd, 0x2b, 0xa6, 0xfa, 0xb4},
        16,
    },

    // #4) Klen = 16, IVlen = 12,  Alen = 20, and Plen = 60 (AAD あり、平文サイズがブロックサイズの整数倍ではない)
    {
        {0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c, 0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08},
        16,
        {0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad, 0xde, 0xca, 0xf8, 0x88},
        12,
        {0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef, 0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
         0xab, 0xad, 0xda, 0xd2},
        20,
        {0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5, 0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
         0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda, 0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
         0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53, 0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
         0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57, 0xba, 0x63, 0x7b, 0x39},
        60,
        {0x42, 0x83, 0x1e, 0xc2, 0x21, 0x77, 0x74, 0x24, 0x4b, 0x72, 0x21, 0xb7, 0x84, 0xd0, 0xd4, 0x9c,
         0xe3, 0xaa, 0x21, 0x2f, 0x2c, 0x02, 0xa4, 0xe0, 0x35, 0xc1, 0x7e, 0x23, 0x29, 0xac, 0xa1, 0x2e,
         0x21, 0xd5, 0x14, 0xb2, 0x54, 0x66, 0x93, 0x1c, 0x7d, 0x8f, 0x6a, 0x5a, 0xac, 0x84, 0xaa, 0x05,
         0x1b, 0xa3, 0x0b, 0x39, 0x6a, 0x0a, 0xac, 0x97, 0x3d, 0x58, 0xe0, 0x91},
        {0x5b, 0xc9, 0x4f, 0xbc, 0x32, 0x21, 0xa5, 0xdb, 0x94, 0xfa, 0xe9, 0x5a, 0xe7, 0x12, 0x1a, 0x47},
        16,
    },

    // #5) Klen = 16, IVlen = 8,  Alen = 20, and Plen = 60 (IV が短い)
    {
        {0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c, 0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08},
        16,
        {0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad},
        8,
        {0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef, 0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
         0xab, 0xad, 0xda, 0xd2},
        20,
        {0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5, 0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
         0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda, 0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
         0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53, 0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
         0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57, 0xba, 0x63, 0x7b, 0x39},
        60,
        {0x61, 0x35, 0x3b, 0x4c, 0x28, 0x06, 0x93, 0x4a, 0x77, 0x7f, 0xf5, 0x1f, 0xa2, 0x2a, 0x47, 0x55,
         0x69, 0x9b, 0x2a, 0x71, 0x4f, 0xcd, 0xc6, 0xf8, 0x37, 0x66, 0xe5, 0xf9, 0x7b, 0x6c, 0x74, 0x23,
         0x73, 0x80, 0x69, 0x00, 0xe4, 0x9f, 0x24, 0xb2, 0x2b, 0x09, 0x75, 0x44, 0xd4, 0x89, 0x6b, 0x42,
         0x49, 0x89, 0xb5, 0xe1, 0xeb, 0xac, 0x0f, 0x07, 0xc2, 0x3f, 0x45, 0x98},
        {0x36, 0x12, 0xd2, 0xe7, 0x9e, 0x3b, 0x07, 0x85, 0x56, 0x1b, 0xe1, 0x4a, 0xac, 0xa2, 0xfc, 0xcb},
        16,
    },

    // #6) Klen = 16, IVlen = 60,  Alen = 20, and Plen = 60 (IV が長い)
    {
        {0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c, 0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08},
        16,
        {0x93, 0x13, 0x22, 0x5d, 0xf8, 0x84, 0x06, 0xe5, 0x55, 0x90, 0x9c, 0x5a, 0xff, 0x52, 0x69, 0xaa,
         0x6a, 0x7a, 0x95, 0x38, 0x53, 0x4f, 0x7d, 0xa1, 0xe4, 0xc3, 0x03, 0xd2, 0xa3, 0x18, 0xa7, 0x28,
         0xc3, 0xc0, 0xc9, 0x51, 0x56, 0x80, 0x95, 0x39, 0xfc, 0xf0, 0xe2, 0x42, 0x9a, 0x6b, 0x52, 0x54,
         0x16, 0xae, 0xdb, 0xf5, 0xa0, 0xde, 0x6a, 0x57, 0xa6, 0x37, 0xb3, 0x9b},
        60,
        {0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef, 0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
         0xab, 0xad, 0xda, 0xd2},
        20,
        {0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5, 0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
         0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda, 0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
         0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53, 0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
         0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57, 0xba, 0x63, 0x7b, 0x39},
        60,
        {0x8c, 0xe2, 0x49, 0x98, 0x62, 0x56, 0x15, 0xb6, 0x03, 0xa0, 0x33, 0xac, 0xa1, 0x3f, 0xb8, 0x94,
         0xbe, 0x91, 0x12, 0xa5, 0xc3, 0xa2, 0x11, 0xa8, 0xba, 0x26, 0x2a, 0x3c, 0xca, 0x7e, 0x2c, 0xa7,
         0x01, 0xe4, 0xa9, 0xa4, 0xfb, 0xa4, 0x3c, 0x90, 0xcc, 0xdc, 0xb2, 0x81, 0xd4, 0x8c, 0x7c, 0x6f,
         0xd6, 0x28, 0x75, 0xd2, 0xac, 0xa4, 0x17, 0x03, 0x4c, 0x34, 0xae, 0xe5},
        {0x61, 0x9c, 0xc5, 0xae, 0xff, 0xfe, 0x0b, 0xfa, 0x46, 0x2a, 0xf4, 0x3c, 0x16, 0x99, 0xd0, 0x50},
        16,
    },

    // #7) Klen = 16, IVlen = 12,  Alen = 20, and Plen = 0 (AAD の入力のみ)
    {
        {0x2f, 0xb4, 0x5e, 0x5b, 0x8f, 0x99, 0x3a, 0x2b, 0xfe, 0xbc, 0x4b, 0x15, 0xb5, 0x33, 0xe0, 0xb4},
        16,
        {0x5b, 0x05, 0x75, 0x5f, 0x98, 0x4d, 0x2b, 0x90, 0xf9, 0x4b, 0x80, 0x27},
        12,
        {0xe8, 0x54, 0x91, 0xb2, 0x20, 0x2c, 0xaf, 0x1d, 0x7d, 0xce, 0x03, 0xb9, 0x7e, 0x09, 0x33, 0x1c,
         0x32, 0x47, 0x39, 0x41},
        20,
        {0},
        0,
        {0},
        {0xc7, 0x5b, 0x78, 0x32, 0xb2, 0xa2, 0xd9, 0xbd, 0x82, 0x74, 0x12, 0xb6, 0xef, 0x57, 0x69, 0xdb},
        16,
    },

    // #8) Klen = 24, IVlen = 12,  Alen = 20, and Plen = 60 (192bit 鍵)
    {
        {0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c, 0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08,
         0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c},
        24,
        {0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad, 0xde, 0xca, 0xf8, 0x88},
        12,
        {0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef, 0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
         0xab, 0xad, 0xda, 0xd2},
        20,
        {0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5, 0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
         0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda, 0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
         0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53, 0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
         0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57, 0xba, 0x63, 0x7b, 0x39},
        60,
        {0x39, 0x80, 0xca, 0x0b, 0x3c, 0x00, 0xe8, 0x41, 0xeb, 0x06, 0xfa, 0xc4, 0x87, 0x2a, 0x27, 0x57,
         0x85, 0x9e, 0x1c, 0xea, 0xa6, 0xef, 0xd9, 0x84, 0x62, 0x85, 0x93, 0xb4, 0x0c, 0xa1, 0xe1, 0x9c,
         0x7d, 0x77, 0x3d, 0x00, 0xc1, 0x44, 0xc5, 0x25, 0xac, 0x61, 0x9d, 0x18, 0xc8, 0x4a, 0x3f, 0x47,
         0x18, 0xe2, 0x44, 0x8b, 0x2f, 0xe3, 0x24, 0xd9, 0xcc, 0xda, 0x27, 0x10},
        {0x25, 0x19, 0x49, 0x8e, 0x80, 0xf1, 0x47, 0x8f, 0x37, 0xba, 0x55, 0xbd, 0x6d, 0x27, 0x61, 0x8c},
        16,
    },

    // #9) Klen = 32, IVlen = 12,  Alen = 20, and Plen = 60 (256bit 鍵)
    {
        {0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c, 0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08,
         0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c, 0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08},
        32,
        {0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad, 0xde, 0xca, 0xf8, 0x88},
        12,
        {0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef, 0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
         0xab, 0xad, 0xda, 0xd2},
        20,
        {0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5, 0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
         0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda, 0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
         0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53, 0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
         0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57, 0xba, 0x63, 0x7b, 0x39},
        60,
        {0x52, 0x2d, 0xc1, 0xf0, 0x99, 0x56, 0x7d, 0x07, 0xf4, 0x7f, 0x37, 0xa3, 0x2a, 0x84, 0x42, 0x7d,
         0x64, 0x3a, 0x8c, 0xdc, 0xbf, 0xe5, 0xc0, 0xc9, 0x75, 0x98, 0xa2, 0xbd, 0x25, 0x55, 0xd1, 0xaa,
         0x8c, 0xb0, 0x8e, 0x48, 0x59, 0x0d, 0xbb, 0x3d, 0xa7, 0xb0, 0x8b, 0x10, 0x56, 0x82, 0x88, 0x38,
         0xc5, 0xf6, 0x1e, 0x63, 0x93, 0xba, 0x7a, 0x0a, 0xbc, 0xc9, 0xf6, 0x62},
        {0x76, 0xfc, 0x6e, 0xce, 0x0f, 0x4e, 0x17, 0x68, 0xcd, 0xdf, 0x88, 0x53, 0xbb, 0x2d, 0x55, 0x1b},
        16,
    },
};

const int TestVectorCount = sizeof(aesGcmTestVectors) / sizeof(aesGcmTestVectors[0]);

template <typename T>
void AuthenticatedEncryptionTest(const AesGcmTestVector& testVector, bool isInPlace)
{
    T                           aes;
    nn::crypto::GcmEncryptor<T> aesGcm;
    nn::Bit8                    output[160];
    nn::Bit8                    mac[nn::crypto::GcmEncryptor<T>::MacSize];

    EXPECT_GE(sizeof(output), nn::crypto::GcmEncryptor<T>::BlockSize);

    aes.Initialize(testVector.key, testVector.keySize);
    aesGcm.Initialize(&aes, testVector.iv, testVector.ivSize);

    if (testVector.aadSize > 0)
    {
        // データが小さい場合はそのまま、大きい場合は分割して入力
        size_t remaining = testVector.aadSize;
        while (remaining > 0)
        {
            size_t inputSize = std::min(sizeof(testVector.aad), remaining);
            aesGcm.UpdateAad(testVector.aad, inputSize);
            remaining -= inputSize;
        }
    }

    if (testVector.textSize > 0)
    {
        if (isInPlace)
        {
            std::memcpy(output, testVector.plainText, testVector.textSize);
            aesGcm.Update(output, sizeof(output), output, testVector.textSize);
        }
        else
        {
            aesGcm.Update(output, sizeof(output), testVector.plainText, testVector.textSize);
        }
    }

    aesGcm.GetMac(mac, testVector.macSize);

    // 暗号化の結果が正しいことの確認
    EXPECT_ARRAY_EQ(output, testVector.cipherText, testVector.textSize);

    // MAC の値が正しいことの確認
    EXPECT_ARRAY_EQ(mac, testVector.mac, testVector.macSize);
}

template <typename T>
void AuthenticatedDecryptionTest(const AesGcmTestVector& testVector, bool isInPlace)
{
    T                           aes;
    nn::crypto::GcmDecryptor<T> aesGcm;
    nn::Bit8                    output[160];
    nn::Bit8                    mac[nn::crypto::GcmDecryptor<T>::MacSize];

    EXPECT_GE(sizeof(output), nn::crypto::GcmDecryptor<T>::BlockSize);

    aes.Initialize(testVector.key, testVector.keySize);
    aesGcm.Initialize(&aes, testVector.iv, testVector.ivSize);

    if (testVector.aadSize > 0)
    {
        // データが小さい場合はそのまま、大きい場合は分割して入力
        size_t remaining = testVector.aadSize;
        while (remaining > 0)
        {
            size_t inputSize = std::min(sizeof(testVector.aad), remaining);
            aesGcm.UpdateAad(testVector.aad, inputSize);
            remaining -= inputSize;
        }
    }

    if (testVector.textSize > 0)
    {
        if (isInPlace)
        {
            std::memcpy(output, testVector.cipherText, testVector.textSize);
            aesGcm.Update(output, sizeof(output), output, testVector.textSize);
        }
        else
        {
            aesGcm.Update(output, sizeof(output), testVector.cipherText, testVector.textSize);
        }
    }

    aesGcm.GetMac(mac, testVector.macSize);

    // 復号化の結果が正しいことの確認
    EXPECT_ARRAY_EQ(output, testVector.plainText, testVector.textSize);

    // MAC の値が正しいことの確認
    EXPECT_ARRAY_EQ(mac, testVector.mac, testVector.macSize);
}

template <typename T>
void AuthenticatedEncryptDataBufferingTest(const AesGcmTestVector& testVector)
{
    T                           aes;
    nn::crypto::GcmEncryptor<T> aesGcm;
    nn::Bit8                    output[160];
    nn::Bit8                    mac[nn::crypto::GcmDecryptor<T>::MacSize];
    const size_t                MaxInputSize = std::min((T::BlockSize * 4), testVector.textSize);

    aes.Initialize(testVector.key, testVector.keySize);

    // 複数ブロック＋端数の処理までテストするため、適当なブロックサイズ分までの各入力サイズについてテストする
    for (size_t inputSize = 1; inputSize < MaxInputSize; inputSize++)
    {
        aesGcm.Initialize(&aes, testVector.iv, testVector.ivSize);

        // AAD を分割入力
        const nn::Bit8* pInput = testVector.aad;
        size_t remaining = testVector.aadSize;
        while (remaining)
        {
            size_t processSize = std::min(inputSize, remaining);
            aesGcm.UpdateAad(pInput, processSize);
            pInput += processSize;
            remaining -= processSize;
        }

        // 平文を分割入力
        pInput = testVector.plainText;
        nn::Bit8*       pOutput = output;
        size_t          outBufSize = sizeof(output);
        remaining = testVector.textSize;
        while (remaining)
        {
            size_t processSize = std::min(inputSize, remaining);
            size_t ret = aesGcm.Update(pOutput, outBufSize, pInput, processSize);
            pInput += processSize;
            remaining -= processSize;
            // 処理された分だけ出力バッファを進める
            pOutput += ret;
            outBufSize -= ret;
        }

        // 暗号化の結果が正しいことの確認
        EXPECT_ARRAY_EQ(output, testVector.cipherText, testVector.textSize);

        aesGcm.GetMac(mac, testVector.macSize);

        // MAC の値が正しいことの確認
        EXPECT_ARRAY_EQ(mac, testVector.mac, testVector.macSize);
    }
}

template <typename T>
void AuthenticatedDecryptDataBufferingTest(const AesGcmTestVector& testVector)
{
    T                           aes;
    nn::crypto::GcmDecryptor<T> aesGcm;
    nn::Bit8                    output[160];
    nn::Bit8                    mac[nn::crypto::GcmDecryptor<T>::MacSize];
    const size_t                MaxInputSize = std::min((T::BlockSize * 4), testVector.textSize);

    aes.Initialize(testVector.key, testVector.keySize);

    // 複数ブロック＋端数の処理までテストするため、適当なブロックサイズ分までの各入力サイズについてテストする
    for (size_t inputSize = 1; inputSize < MaxInputSize; inputSize++)
    {
        aesGcm.Initialize(&aes, testVector.iv, testVector.ivSize);

        // AAD を分割入力
        const nn::Bit8* pInput = testVector.aad;
        size_t remaining = testVector.aadSize;
        while (remaining)
        {
            size_t processSize = std::min(inputSize, remaining);
            aesGcm.UpdateAad(pInput, processSize);
            pInput += processSize;
            remaining -= processSize;
        }

        // 暗号文を分割入力
        pInput = testVector.cipherText;
        nn::Bit8*       pOutput = output;
        size_t          outBufSize = sizeof(output);
        remaining = testVector.textSize;
        while (remaining)
        {
            size_t processSize = std::min(inputSize, remaining);
            size_t ret = aesGcm.Update(pOutput, outBufSize, pInput, processSize);
            pInput += processSize;
            remaining -= processSize;
            // 処理された分だけ出力バッファを進める
            pOutput += ret;
            outBufSize -= ret;
        }

        // 復号化の結果が正しいことの確認
        EXPECT_ARRAY_EQ(output, testVector.plainText, testVector.textSize);

        aesGcm.GetMac(mac, testVector.macSize);

        // MAC の値が正しいことの確認
        EXPECT_ARRAY_EQ(mac, testVector.mac, testVector.macSize);
    }
}

/**
  @brief   AES-GCM による認証付き暗号化をテストします。

  @details
  テストベクトルを用いて正しく暗号化と MAC 計算ができることをテストします。
 */
TEST(AesGcmTest, AuthenticatedEncryption)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesGcmTestVectors[i].keySize)
        {
        case 16:
            AuthenticatedEncryptionTest<nn::crypto::AesEncryptor128>(aesGcmTestVectors[i], false);
            break;
        case 24:
            AuthenticatedEncryptionTest<nn::crypto::AesEncryptor192>(aesGcmTestVectors[i], false);
            break;
        case 32:
            AuthenticatedEncryptionTest<nn::crypto::AesEncryptor256>(aesGcmTestVectors[i], false);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   AES-GCM による認証付き復号化をテストします。

  @details
  テストベクトルを用いて正しく復号化と MAC 計算ができることをテストします。
 */
TEST(AesGcmTest, AuthenticatedDecryption)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesGcmTestVectors[i].keySize)
        {
        case 16:
            AuthenticatedDecryptionTest<nn::crypto::AesEncryptor128>(aesGcmTestVectors[i], false);
            break;
        case 24:
            AuthenticatedDecryptionTest<nn::crypto::AesEncryptor192>(aesGcmTestVectors[i], false);
            break;
        case 32:
            AuthenticatedDecryptionTest<nn::crypto::AesEncryptor256>(aesGcmTestVectors[i], false);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   AES-CCM による同一入出力バッファ上での認証付き暗号化をテストします。

  @details
  テストベクトルを用いて正しく暗号化と MAC 計算ができることをテストします。
 */
TEST(AesGcmTest, InPlaceAuthenticatedEncryption)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesGcmTestVectors[i].keySize)
        {
        case 16:
            AuthenticatedEncryptionTest<nn::crypto::AesEncryptor128>(aesGcmTestVectors[i], true);
            break;
        case 24:
            AuthenticatedEncryptionTest<nn::crypto::AesEncryptor192>(aesGcmTestVectors[i], true);
            break;
        case 32:
            AuthenticatedEncryptionTest<nn::crypto::AesEncryptor256>(aesGcmTestVectors[i], true);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   AES-CCM による同一入出力バッファ上での認証付き復号化をテストします。

  @details
  テストベクトルを用いて正しく復号化と MAC 計算ができることをテストします。
 */
TEST(AesGcmTest, InPlaceAuthenticatedDecryption)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesGcmTestVectors[i].keySize)
        {
        case 16:
            AuthenticatedDecryptionTest<nn::crypto::AesEncryptor128>(aesGcmTestVectors[i], true);
            break;
        case 24:
            AuthenticatedDecryptionTest<nn::crypto::AesEncryptor192>(aesGcmTestVectors[i], true);
            break;
        case 32:
            AuthenticatedDecryptionTest<nn::crypto::AesEncryptor256>(aesGcmTestVectors[i], true);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   AES-GCM による暗号化を入力サイズを変えてテストします。

  @details
  適当な大きさまで入力サイズを変えて、正しく暗号化できることをテストします。
 */
TEST(AesGcmTest, AuthenticatedEncryptDataBuffering)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesGcmTestVectors[i].keySize)
        {
        case 16:
            AuthenticatedEncryptDataBufferingTest<nn::crypto::AesEncryptor128>(aesGcmTestVectors[i]);
            break;
        case 24:
            AuthenticatedEncryptDataBufferingTest<nn::crypto::AesEncryptor192>(aesGcmTestVectors[i]);
            break;
        case 32:
            AuthenticatedEncryptDataBufferingTest<nn::crypto::AesEncryptor256>(aesGcmTestVectors[i]);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   AES-GCM による復号化を入力サイズを変えてテストします。

  @details
  適当な大きさまで入力サイズを変えて、正しく復号化できることをテストします。
 */
TEST(AesGcmTest, AuthenticatedDecryptDataBuffering)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesGcmTestVectors[i].keySize)
        {
        case 16:
            AuthenticatedDecryptDataBufferingTest<nn::crypto::AesEncryptor128>(aesGcmTestVectors[i]);
            break;
        case 24:
            AuthenticatedDecryptDataBufferingTest<nn::crypto::AesEncryptor192>(aesGcmTestVectors[i]);
            break;
        case 32:
            AuthenticatedDecryptDataBufferingTest<nn::crypto::AesEncryptor256>(aesGcmTestVectors[i]);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   通常の入力パターンでの GcmEncryptor クラスの状態遷移をテストします。
 */
TEST(AesGcmTest, StateTransition)
{
    nn::crypto::AesEncryptor128                           aes;
    nn::crypto::GcmEncryptor<nn::crypto::AesEncryptor128> aesGcm;
    nn::Bit8                                              output[60];
    nn::Bit8                                              mac[16];
    nn::Bit8                                              mac2[16];

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // 初期化前は Initialize 以外の処理は呼べない
    EXPECT_DEATH_IF_SUPPORTED(aesGcm.Reset(aesGcmTestVectors[4].iv, aesGcmTestVectors[4].ivSize), "");

    EXPECT_DEATH_IF_SUPPORTED(aesGcm.UpdateAad(aesGcmTestVectors[4].aad, aesGcmTestVectors[4].aadSize), "");

    EXPECT_DEATH_IF_SUPPORTED(aesGcm.Update(output, sizeof(output),
                                            aesGcmTestVectors[4].plainText ,aesGcmTestVectors[4].textSize), "");

    EXPECT_DEATH_IF_SUPPORTED(aesGcm.GetMac(mac, sizeof(mac)), "");
#endif

    // 初期化
    aes.Initialize(aesGcmTestVectors[4].key, aesGcmTestVectors[4].keySize);
    aesGcm.Initialize(&aes, aesGcmTestVectors[4].iv, aesGcmTestVectors[4].ivSize);

    // AAD を入力
    aesGcm.UpdateAad(aesGcmTestVectors[4].aad, aesGcmTestVectors[4].aadSize);

    // 平文を入力
    aesGcm.Update(output, sizeof(output),
                  aesGcmTestVectors[4].plainText ,aesGcmTestVectors[4].textSize);

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // 平文の入力後の AAD の入力は NG
    EXPECT_DEATH_IF_SUPPORTED(aesGcm.UpdateAad(aesGcmTestVectors[4].aad, aesGcmTestVectors[4].aadSize), "");
#endif

    // AAD、平文入力後の MAC の取得は可能
    aesGcm.GetMac(mac, sizeof(mac));

    // MAC の取得は連続で呼んでも大丈夫で、同じ値が出力されるはず
    aesGcm.GetMac(mac2, sizeof(mac2));

    EXPECT_ARRAY_EQ(output, aesGcmTestVectors[4].cipherText, aesGcmTestVectors[4].textSize);
    EXPECT_ARRAY_EQ(mac, aesGcmTestVectors[4].mac, aesGcmTestVectors[4].macSize);
    EXPECT_ARRAY_EQ(mac, mac2, aesGcmTestVectors[4].macSize);

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // GetMac の後は UpdateAad も Update も呼べない
    EXPECT_DEATH_IF_SUPPORTED(aesGcm.UpdateAad(aesGcmTestVectors[4].aad, aesGcmTestVectors[4].aadSize), "");

    EXPECT_DEATH_IF_SUPPORTED(aesGcm.Update(output, sizeof(output),
                                            aesGcmTestVectors[4].plainText ,aesGcmTestVectors[4].textSize), "");
#endif

    // Reset からの計算の再開は可能
    aesGcm.Reset(aesGcmTestVectors[4].iv, aesGcmTestVectors[4].ivSize);

    aesGcm.UpdateAad(aesGcmTestVectors[4].aad, aesGcmTestVectors[4].aadSize);
    aesGcm.Update(output, sizeof(output),
                  aesGcmTestVectors[4].plainText ,aesGcmTestVectors[4].textSize);
    aesGcm.GetMac(mac, sizeof(mac));

    EXPECT_ARRAY_EQ(mac, aesGcmTestVectors[4].mac, aesGcmTestVectors[4].macSize);
}

/*
  @brief   Aes128GcmEncryptor/Decryptor による暗号化・復号化をテストします。
 */
TEST(AesGcmTest, UtilityClassInterface)
{
    nn::Bit8 output[160];
    nn::Bit8 mac[nn::crypto::Aes128GcmEncryptor::MacSize];

    // 暗号化のテスト
    for (int i = 0; i < TestVectorCount; i++)
    {
        // 128bit 鍵のみテスト
        if (aesGcmTestVectors[i].keySize != 16)
        {
            continue;
        }

        nn::crypto::Aes128GcmEncryptor aes128GcmEnc;

        aes128GcmEnc.Initialize(aesGcmTestVectors[i].key, aesGcmTestVectors[i].keySize,
                                aesGcmTestVectors[i].iv, aesGcmTestVectors[i].ivSize);

        if (aesGcmTestVectors[i].aadSize > 0)
        {
            aes128GcmEnc.UpdateAad(aesGcmTestVectors[i].aad, aesGcmTestVectors[i].aadSize);
        }

        if (aesGcmTestVectors[i].textSize > 0)
        {
            aes128GcmEnc.Update(output, sizeof(output),
                                aesGcmTestVectors[i].plainText, aesGcmTestVectors[i].textSize);
        }

        aes128GcmEnc.GetMac(mac, sizeof(mac));

        // 暗号化の結果を確認
        EXPECT_ARRAY_EQ(output, aesGcmTestVectors[i].cipherText, aesGcmTestVectors[i].textSize);

        // MAC 計算の結果を確認
        EXPECT_ARRAY_EQ(mac, aesGcmTestVectors[i].mac, aesGcmTestVectors[i].macSize);
    }

    // 復号化のテスト
    for (int i = 0; i < TestVectorCount; i++)
    {
        // 128bit 鍵のみテスト
        if (aesGcmTestVectors[i].keySize != 16)
        {
            continue;
        }

        nn::crypto::Aes128GcmDecryptor aes128GcmDec;

        aes128GcmDec.Initialize(aesGcmTestVectors[i].key, aesGcmTestVectors[i].keySize,
                                aesGcmTestVectors[i].iv, aesGcmTestVectors[i].ivSize);

        if (aesGcmTestVectors[i].aadSize > 0)
        {
            aes128GcmDec.UpdateAad(aesGcmTestVectors[i].aad, aesGcmTestVectors[i].aadSize);
        }

        if (aesGcmTestVectors[i].textSize > 0)
        {
            aes128GcmDec.Update(output, sizeof(output),
                                aesGcmTestVectors[i].cipherText, aesGcmTestVectors[i].textSize);
        }

        aes128GcmDec.GetMac(mac, sizeof(mac));

        // 復号化の結果を確認
        EXPECT_ARRAY_EQ(output, aesGcmTestVectors[i].plainText, aesGcmTestVectors[i].textSize);

        // MAC 計算の結果を確認
        EXPECT_ARRAY_EQ(mac, aesGcmTestVectors[i].mac, aesGcmTestVectors[i].macSize);
    }
}

/**
  @brief   ユーティリティ関数による暗号化・復号化をテストします。
 */
TEST(AesGcmTest, UtilityFunctionInterface)
{
    nn::Bit8 output[160];
    nn::Bit8 mac[nn::crypto::Aes128GcmEncryptor::MacSize];

    // 暗号化のテスト
    for (int i = 0; i < TestVectorCount; i++)
    {
        // 128bit 鍵のみテスト
        if (aesGcmTestVectors[i].keySize != 16)
        {
            continue;
        }

        nn::crypto::EncryptAes128Gcm(output, sizeof(output),
                                     mac, sizeof(mac),
                                     aesGcmTestVectors[i].key, aesGcmTestVectors[i].keySize,
                                     aesGcmTestVectors[i].iv, aesGcmTestVectors[i].ivSize,
                                     aesGcmTestVectors[i].plainText, aesGcmTestVectors[i].textSize,
                                     aesGcmTestVectors[i].aad, aesGcmTestVectors[i].aadSize);

        // 暗号化の結果を確認
        EXPECT_ARRAY_EQ(output, aesGcmTestVectors[i].cipherText, aesGcmTestVectors[i].textSize);

        // MAC 計算の結果を確認
        EXPECT_ARRAY_EQ(mac, aesGcmTestVectors[i].mac, aesGcmTestVectors[i].macSize);
    }

    // 復号化のテスト
    for (int i = 0; i < TestVectorCount; i++)
    {
        // 128bit 鍵のみテスト
        if (aesGcmTestVectors[i].keySize != 16)
        {
            continue;
        }

        nn::crypto::DecryptAes128Gcm(output, sizeof(output),
                                     mac, sizeof(mac),
                                     aesGcmTestVectors[i].key, aesGcmTestVectors[i].keySize,
                                     aesGcmTestVectors[i].iv, aesGcmTestVectors[i].ivSize,
                                     aesGcmTestVectors[i].cipherText, aesGcmTestVectors[i].textSize,
                                     aesGcmTestVectors[i].aad, aesGcmTestVectors[i].aadSize);

        // 復号化の結果を確認
        EXPECT_ARRAY_EQ(output, aesGcmTestVectors[i].plainText, aesGcmTestVectors[i].textSize);

        // MAC 計算の結果を確認
        EXPECT_ARRAY_EQ(mac, aesGcmTestVectors[i].mac, aesGcmTestVectors[i].macSize);
    }
}

/**
  @brief   デストラクタで内部データがクリアされることをテストします。
 */
template <size_t KeySize>
void GcmDestructorTest()
{
    // Dummy key and iv
    const uint8_t key[32] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    };
    const uint8_t iv[16] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
    };

    // Encryptor test
    {
        nn::crypto::AesEncryptor<KeySize> aes;
        aes.Initialize(key, KeySize);

        nn::crypto::GcmEncryptor<nn::crypto::AesEncryptor<KeySize>> mode;
        mode.Initialize(&aes, iv, 12);

        // 明示的にデストラクタを呼んで呼び出し前後でのメモリクリアを確認する
        EXPECT_ARRAY_NONZERO(&mode, sizeof(mode));
        mode.~GcmEncryptor<nn::crypto::AesEncryptor<KeySize>>();
        EXPECT_ARRAY_ZERO(&mode, sizeof(mode));

        // AES 本体は暗号利用モードのほうだけクリアされてもクリアされていない
        EXPECT_ARRAY_NONZERO(&aes, sizeof(aes));
    }

    // Decryptor test
    {
        nn::crypto::AesEncryptor<KeySize> aes;
        aes.Initialize(key, KeySize);

        nn::crypto::GcmDecryptor<nn::crypto::AesEncryptor<KeySize>> mode;
        mode.Initialize(&aes, iv, 12);

        // 明示的にデストラクタを呼んで呼び出し前後でのメモリクリアを確認する
        EXPECT_ARRAY_NONZERO(&mode, sizeof(mode));
        mode.~GcmDecryptor<nn::crypto::AesEncryptor<KeySize>>();
        EXPECT_ARRAY_ZERO(&mode, sizeof(mode));

        // AES 本体は暗号利用モードがクリアされてもクリアされていない
        EXPECT_ARRAY_NONZERO(&aes, sizeof(aes));
    }
}

TEST(AesGcmTest, Destructor)
{
    GcmDestructorTest<16>();
    GcmDestructorTest<24>();
    GcmDestructorTest<32>();
}

/**
  @brief   Update に 0 バイトが渡されても正常に計算できることをチェックします。
 */
TEST(AesGcmTest, ZeroByteEncryption)
{
    nn::crypto::AesEncryptor128                           aes;
    nn::crypto::GcmEncryptor<nn::crypto::AesEncryptor128> aesGcm;
    nn::Bit8                                              output[32];
    nn::Bit8                                              mac[16];

    // 0 to AAD
    {
        // 初期化
        aes.Initialize(aesGcmTestVectors[0].key, aesGcmTestVectors[0].keySize);
        aesGcm.Initialize(&aes, aesGcmTestVectors[0].iv, aesGcmTestVectors[0].ivSize);

        // AAD の入力前に 0 バイトの Update を呼ぶ
        aesGcm.UpdateAad(nullptr, 0);
        aesGcm.UpdateAad(aesGcmTestVectors[0].aad, aesGcmTestVectors[0].aadSize);

        // 平文の入力
        aesGcm.Update(output, sizeof(output),
                      aesGcmTestVectors[0].plainText ,aesGcmTestVectors[0].textSize);

        // MAC の取得
        aesGcm.GetMac(mac, sizeof(mac));

        // 出力結果と MAC が正しいことを確認
        EXPECT_ARRAY_EQ(output, aesGcmTestVectors[0].cipherText, aesGcmTestVectors[0].textSize);
        EXPECT_ARRAY_EQ(mac, aesGcmTestVectors[0].mac, aesGcmTestVectors[0].macSize);
    }

    // 0 to Encryption
    {
        // 初期化
        aes.Initialize(aesGcmTestVectors[0].key, aesGcmTestVectors[0].keySize);
        aesGcm.Initialize(&aes, aesGcmTestVectors[0].iv, aesGcmTestVectors[0].ivSize);

        // AAD の入力
        aesGcm.UpdateAad(aesGcmTestVectors[0].aad, aesGcmTestVectors[0].aadSize);

        // 平文の入力前に 0 バイトの Update を呼ぶ
        EXPECT_EQ(aesGcm.Update(output, sizeof(output), nullptr, 0), 0);
        aesGcm.Update(output, sizeof(output),
                      aesGcmTestVectors[0].plainText ,aesGcmTestVectors[0].textSize);

        // MAC の取得
        aesGcm.GetMac(mac, sizeof(mac));

        // 出力結果と MAC が正しいことを確認
        EXPECT_ARRAY_EQ(output, aesGcmTestVectors[0].cipherText, aesGcmTestVectors[0].textSize);
        EXPECT_ARRAY_EQ(mac, aesGcmTestVectors[0].mac, aesGcmTestVectors[0].macSize);
    }
}

/**
  @brief   Update に 0 バイトが渡されても正常に計算できることをチェックします。
 */
TEST(AesGcmTest, ZeroByteDecryption)
{
    nn::crypto::AesEncryptor128                           aes;
    nn::crypto::GcmDecryptor<nn::crypto::AesEncryptor128> aesGcm;
    nn::Bit8                                              output[32];
    nn::Bit8                                              mac[16];

    // 0 to AAD
    {
        // 初期化
        aes.Initialize(aesGcmTestVectors[0].key, aesGcmTestVectors[0].keySize);
        aesGcm.Initialize(&aes, aesGcmTestVectors[0].iv, aesGcmTestVectors[0].ivSize);

        // AAD の入力前に 0 バイトの Update を呼ぶ
        aesGcm.UpdateAad(nullptr, 0);
        aesGcm.UpdateAad(aesGcmTestVectors[0].aad, aesGcmTestVectors[0].aadSize);

        // 平文の入力
        aesGcm.Update(output, sizeof(output),
                      aesGcmTestVectors[0].cipherText, aesGcmTestVectors[0].textSize);

        // MAC の取得
        aesGcm.GetMac(mac, sizeof(mac));

        // 出力結果と MAC が正しいことを確認
        EXPECT_ARRAY_EQ(output, aesGcmTestVectors[0].plainText, aesGcmTestVectors[0].textSize);
        EXPECT_ARRAY_EQ(mac, aesGcmTestVectors[0].mac, aesGcmTestVectors[0].macSize);
    }

    // 0 to Decryption
    {
        // 初期化
        aes.Initialize(aesGcmTestVectors[0].key, aesGcmTestVectors[0].keySize);
        aesGcm.Initialize(&aes, aesGcmTestVectors[0].iv, aesGcmTestVectors[0].ivSize);

        // AAD の入力
        aesGcm.UpdateAad(aesGcmTestVectors[0].aad, aesGcmTestVectors[0].aadSize);

        // 平文の入力前に 0 バイトの Update を呼ぶ
        EXPECT_EQ(aesGcm.Update(output, sizeof(output), nullptr, 0), 0);
        aesGcm.Update(output, sizeof(output),
                      aesGcmTestVectors[0].cipherText, aesGcmTestVectors[0].textSize);

        // MAC の取得
        aesGcm.GetMac(mac, sizeof(mac));

        // 出力結果と MAC が正しいことを確認
        EXPECT_ARRAY_EQ(output, aesGcmTestVectors[0].plainText, aesGcmTestVectors[0].textSize);
        EXPECT_ARRAY_EQ(mac, aesGcmTestVectors[0].mac, aesGcmTestVectors[0].macSize);
    }
}
