﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/crypto/crypto_AesEncryptor.h>
#include <nn/crypto/crypto_CtrEncryptor.h>
#include <nn/crypto/crypto_CtrDecryptor.h>
#include <nn/crypto/crypto_Aes128CtrEncryptor.h>
#include <nn/crypto/crypto_Aes128CtrDecryptor.h>

#include "testCrypto_Util.h"

struct AesCtrTestVector
{
    nn::Bit8  key[32];
    size_t    keySize;
    nn::Bit8  iv[16];
    size_t    ivSize;
    nn::Bit8  plainText[160];
    size_t    textSize;
    nn::Bit8  cipherText[160];
};

/* CTR モード検証用のテストベクトル。
   NIST は多くを提供していないので出典を明記する。 */
const AesCtrTestVector aesCtrTestVectors[] =
{
    // #1) 鍵サイズ = 128bit、平文 = 16 バイト (1 ブロックちょうどのメッセージ) (RFC3686)
    {
        {0xae, 0x68, 0x52, 0xf8, 0x12, 0x10, 0x67, 0xcc, 0x4b, 0xf7, 0xa5, 0x76, 0x55, 0x77, 0xf3, 0x9e},
        16,
        {0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01},
        16,
        {0x53, 0x69, 0x6e, 0x67, 0x6c, 0x65, 0x20, 0x62, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x6d, 0x73, 0x67},
        16,
        {0xe4, 0x09, 0x5d, 0x4f, 0xb7, 0xa7, 0xb3, 0x79, 0x2d, 0x61, 0x75, 0xa3, 0x26, 0x13, 0x11, 0xb8},
    },

    // #2) 鍵サイズ = 128bit、平文 = 36 バイト (ブロックサイズではないメッセージ) (RFC3686)
    {
        {0x76, 0x91, 0xbe, 0x03, 0x5e, 0x50, 0x20, 0xa8, 0xac, 0x6e, 0x61, 0x85, 0x29, 0xf9, 0xa0, 0xdc},
        16,
        {0x00, 0xe0, 0x01, 0x7b, 0x27, 0x77, 0x7f, 0x3f, 0x4a, 0x17, 0x86, 0xf0, 0x00, 0x00, 0x00, 0x01},
        16,
        {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
         0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
         0x20, 0x21, 0x22, 0x23},
        36,
        {0xc1, 0xcf, 0x48, 0xa8, 0x9f, 0x2f, 0xfd, 0xd9, 0xcf, 0x46, 0x52, 0xe9, 0xef, 0xdb, 0x72, 0xd7,
         0x45, 0x40, 0xa4, 0x2b, 0xde, 0x6d, 0x78, 0x36, 0xd5, 0x9a, 0x5c, 0xea, 0xae, 0xf3, 0x10, 0x53,
         0x25, 0xb2, 0x07, 0x2f},
    },

    // #3) 鍵サイズ = 128bit、平文 = 64 バイト (複数ブロック。単一桁カウンタ桁上り) (NIST SP800-38A)
    {
        {0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c},
        16,
        {0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff},
        16,
        {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
         0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
         0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
         0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10},
        64,
        {0x87, 0x4d, 0x61, 0x91, 0xb6, 0x20, 0xe3, 0x26, 0x1b, 0xef, 0x68, 0x64, 0x99, 0x0d, 0xb6, 0xce,
         0x98, 0x06, 0xf6, 0x6b, 0x79, 0x70, 0xfd, 0xff, 0x86, 0x17, 0x18, 0x7b, 0xb9, 0xff, 0xfd, 0xff,
         0x5a, 0xe4, 0xdf, 0x3e, 0xdb, 0xd5, 0xd3, 0x5e, 0x5b, 0x4f, 0x09, 0x02, 0x0d, 0xb0, 0x3e, 0xab,
         0x1e, 0x03, 0x1d, 0xda, 0x2f, 0xbe, 0x03, 0xd1, 0x79, 0x21, 0x70, 0xa0, 0xf3, 0x00, 0x9c, 0xee},
    },

    // #4) 鍵サイズ = 128bit、平文 = 64 バイト (複数桁カウンタ桁上がり) (独自)
    {
        {0x56, 0xe4, 0x7a, 0x38, 0xc5, 0x59, 0x89, 0x74, 0xbc, 0x46, 0x90, 0x3d, 0xba, 0x29, 0x03, 0x49},
        16,
        {0x00, 0x00, 0x00, 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe},
        16,
        {0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
         0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
         0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
         0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf},
        64,
        {0x78, 0xc6, 0xb0, 0x44, 0xeb, 0x21, 0x94, 0x6a, 0xa0, 0x88, 0xe0, 0xa8, 0xaf, 0x69, 0x1f, 0xcf,
         0x6c, 0x6f, 0xb4, 0xe0, 0xa7, 0x55, 0xc5, 0x5a, 0xf8, 0x8a, 0x39, 0x25, 0xf8, 0x61, 0x2c, 0x3f,
         0xf3, 0x9d, 0x79, 0x05, 0x37, 0x21, 0xb7, 0x29, 0xd5, 0xd5, 0x3d, 0xec, 0x3d, 0xc1, 0xea, 0x2d,
         0x92, 0x97, 0x46, 0x25, 0xc0, 0xdb, 0xaa, 0xb6, 0x7a, 0x70, 0x16, 0xea, 0xad, 0xe3, 0x77, 0x7f},
    },

    // #5) 鍵サイズ = 128bit、平文 = 64 バイト (最上位桁までカウンタ桁上がり) (独自)
    {
        {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f},
        16,
        {0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe},
        16,
        {0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
         0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
         0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
         0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf},
        64,
        {0x78, 0xaf, 0x02, 0x3f, 0x9a, 0x3b, 0xbe, 0xc8, 0xd4, 0xf4, 0xef, 0x74, 0x0f, 0xe8, 0x9c, 0x52,
         0x19, 0xae, 0xa6, 0xbd, 0x92, 0xff, 0xda, 0x1f, 0x4c, 0xa9, 0xa3, 0x3b, 0xdf, 0x95, 0xa0, 0x8b,
         0x23, 0xbd, 0x11, 0xa0, 0x19, 0xb9, 0x41, 0x67, 0x52, 0x36, 0xc4, 0xf5, 0xac, 0x2d, 0x52, 0x4d,
         0xaf, 0x37, 0x35, 0x29, 0xbf, 0xd2, 0x29, 0xce, 0xd5, 0x78, 0xae, 0x1c, 0x0b, 0x14, 0x2d, 0xbd},
    },

    // #6) 鍵サイズ = 128bit、平文 = 64 バイト (カウンタ桁あふれ) (独自)
    {
        {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f},
        16,
        {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe},
        16,
        {0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
         0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
         0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
         0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf},
        64,
        {0x16, 0x14, 0x60, 0x7b, 0x89, 0x2e, 0x72, 0xa8, 0x67, 0xe7, 0x72, 0x5f, 0x02, 0xc3, 0x39, 0x41,
         0x8c, 0xf5, 0xad, 0x81, 0x7a, 0xb2, 0x34, 0x94, 0xdc, 0x6e, 0x18, 0x22, 0xb2, 0xed, 0x05, 0xac,
         0x06, 0x60, 0xf9, 0xf4, 0x43, 0x4a, 0x9d, 0x45, 0xa7, 0x86, 0x4b, 0xa9, 0x6d, 0x05, 0x16, 0xb6,
         0xa3, 0x97, 0xc1, 0x46, 0x41, 0x15, 0x62, 0xc9, 0x91, 0xa2, 0x67, 0x38, 0xb9, 0x29, 0xf3, 0xd5},
    },

    // #7) 鍵サイズ = 128bit、平文 = 160 バイト (少し長めのメッセージ) (独自)
    {
        {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f},
        16,
        {0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff},
        16,
        {0x22, 0x57, 0x2D, 0x22, 0xBE, 0x17, 0x48, 0xA0, 0x26, 0xEB, 0x36, 0x38, 0xA6, 0x55, 0x53, 0xB6,
         0xdc, 0x52, 0x1d, 0x6d, 0x88, 0x0d, 0xa9, 0x82, 0x26, 0xd2, 0x7b, 0xd7, 0x96, 0x5d, 0x00, 0x72,
         0x3f, 0xa7, 0x40, 0x09, 0x8d, 0x52, 0x32, 0x78, 0x1a, 0xb1, 0x79, 0xc0, 0xd0, 0x25, 0x23, 0x78,
         0x7f, 0xbb, 0x29, 0x4e, 0xad, 0x46, 0xe8, 0xa6, 0xb9, 0x3e, 0x46, 0x47, 0x4e, 0xfa, 0x2d, 0x34,
         0xe1, 0x78, 0xcd, 0x83, 0x6d, 0xed, 0x51, 0x30, 0x87, 0x14, 0x2a, 0x31, 0xe9, 0xb1, 0xdd, 0xbf,
         0x42, 0x55, 0x2b, 0x95, 0xbd, 0xb3, 0xbf, 0xda, 0xc4, 0xa9, 0x6b, 0xf9, 0xbc, 0x4b, 0x4b, 0xe2,
         0xe1, 0x67, 0xd5, 0x5c, 0x7b, 0x5f, 0x6c, 0xb8, 0x6a, 0x22, 0x86, 0xd5, 0xd5, 0xa8, 0x66, 0x63,
         0x14, 0xa7, 0x7c, 0x6f, 0x86, 0xc4, 0xfb, 0xb4, 0x69, 0xa4, 0x6d, 0xde, 0x51, 0xf1, 0x25, 0x4b,
         0x28, 0xb0, 0xa3, 0xb9, 0x42, 0x71, 0x61, 0x1a, 0x62, 0x8c, 0x1b, 0x21, 0x18, 0xd8, 0x69, 0x49,
         0xe0, 0x90, 0x28, 0xcc, 0x4b, 0x54, 0xdd, 0x0a, 0x76, 0x3e, 0x45, 0x8d, 0xf4, 0x5d, 0x44, 0xff},
        160,
        {0x44, 0xf0, 0xea, 0xca, 0x8a, 0x45, 0x79, 0xe8, 0xb1, 0xba, 0xe8, 0x3f, 0x95, 0x43, 0xfe, 0x1b,
         0x6e, 0xd3, 0xca, 0x6d, 0x3f, 0x93, 0x95, 0x2f, 0x82, 0x7f, 0x08, 0x6c, 0xf8, 0xc1, 0x1f, 0x98,
         0xed, 0xd6, 0xd2, 0x5f, 0xf1, 0x09, 0xd9, 0xe5, 0xe1, 0x30, 0xf2, 0x99, 0x9f, 0xb7, 0x76, 0x09,
         0x0f, 0x63, 0x4f, 0x14, 0x91, 0xf9, 0xb0, 0xe1, 0xee, 0x53, 0x45, 0x5f, 0x01, 0x9a, 0x1a, 0xae,
         0x8a, 0x79, 0x6f, 0xab, 0x40, 0x7b, 0x02, 0x52, 0x03, 0x96, 0xf3, 0x23, 0x16, 0x4f, 0x5c, 0x82,
         0xab, 0x61, 0x0a, 0x65, 0xd8, 0x67, 0x86, 0x84, 0x6c, 0x24, 0x4e, 0x87, 0xfc, 0x04, 0x73, 0x61,
         0x05, 0x61, 0x8f, 0x69, 0x20, 0x82, 0x79, 0x64, 0xf1, 0x9d, 0xf6, 0x0a, 0xad, 0x7f, 0x58, 0x5a,
         0xf2, 0x83, 0xd1, 0xba, 0x85, 0x4f, 0x3d, 0x7f, 0x2f, 0x51, 0xea, 0x4c, 0xcc, 0x32, 0x84, 0xb1,
         0x1b, 0xc5, 0x44, 0xc7, 0x7d, 0x52, 0x70, 0xd4, 0xe6, 0xc5, 0x47, 0x4e, 0xae, 0xed, 0x50, 0x1b,
         0x90, 0x4d, 0x94, 0xa2, 0x67, 0x38, 0xe9, 0x23, 0xd1, 0xe6, 0x57, 0xbd, 0x2e, 0x39, 0xc7, 0x47},
    },

    // #8) 鍵サイズ = 192bit、平文 = 64 バイト (192bit 鍵) (NIST SP800-38A)
    {
        {0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
         0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b},
        24,
        {0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff},
        16,
        {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
         0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
         0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
         0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10},
        64,
        {0x1a, 0xbc, 0x93, 0x24, 0x17, 0x52, 0x1c, 0xa2, 0x4f, 0x2b, 0x04, 0x59, 0xfe, 0x7e, 0x6e, 0x0b,
         0x09, 0x03, 0x39, 0xec, 0x0a, 0xa6, 0xfa, 0xef, 0xd5, 0xcc, 0xc2, 0xc6, 0xf4, 0xce, 0x8e, 0x94,
         0x1e, 0x36, 0xb2, 0x6b, 0xd1, 0xeb, 0xc6, 0x70, 0xd1, 0xbd, 0x1d, 0x66, 0x56, 0x20, 0xab, 0xf7,
         0x4f, 0x78, 0xa7, 0xf6, 0xd2, 0x98, 0x09, 0x58, 0x5a, 0x97, 0xda, 0xec, 0x58, 0xc6, 0xb0, 0x50},
    },

    // #9) 鍵サイズ = 256bit、平文 = 64 バイト (256bit 鍵) (NIST SP800-38A)
    {
        {0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
         0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4},
        32,
        {0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff},
        16,
        {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
         0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
         0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
         0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10},
        64,
        {0x60, 0x1e, 0xc3, 0x13, 0x77, 0x57, 0x89, 0xa5, 0xb7, 0xa7, 0xf5, 0x04, 0xbb, 0xf3, 0xd2, 0x28,
         0xf4, 0x43, 0xe3, 0xca, 0x4d, 0x62, 0xb5, 0x9a, 0xca, 0x84, 0xe9, 0x90, 0xca, 0xca, 0xf5, 0xc5,
         0x2b, 0x09, 0x30, 0xda, 0xa2, 0x3d, 0xe9, 0x4c, 0xe8, 0x70, 0x17, 0xba, 0x2d, 0x84, 0x98, 0x8d,
         0xdf, 0xc9, 0xc5, 0x8d, 0xb6, 0x7a, 0xad, 0xa6, 0x13, 0xc2, 0xdd, 0x08, 0x45, 0x79, 0x41, 0xa6},
    },
};

const int TestVectorCount = sizeof(aesCtrTestVectors) / sizeof(aesCtrTestVectors[0]);

template <typename T>
void EncryptionTest(const AesCtrTestVector& testVector, bool isInPlace)
{
    T                           aes;
    nn::crypto::CtrEncryptor<T> aesCtr;
    nn::Bit8                    output[160];

    EXPECT_GE(sizeof(output), nn::crypto::CtrEncryptor<T>::BlockSize);

    aes.Initialize(testVector.key, testVector.keySize);
    aesCtr.Initialize(&aes, testVector.iv, testVector.ivSize);
    if (isInPlace)
    {
        std::memcpy(output, testVector.plainText, testVector.textSize);
        aesCtr.Update(output, sizeof(output), output, testVector.textSize);
    }
    else
    {
        aesCtr.Update(output, sizeof(output), testVector.plainText, testVector.textSize);
    }
    EXPECT_ARRAY_EQ(output, testVector.cipherText, testVector.textSize);
}

template <typename T>
void DecryptionTest(const AesCtrTestVector& testVector, bool isInPlace)
{
    T                           aes;
    nn::crypto::CtrDecryptor<T> aesCtr;
    nn::Bit8                    output[160];

    EXPECT_GE(sizeof(output), nn::crypto::CtrDecryptor<T>::BlockSize);

    aes.Initialize(testVector.key, testVector.keySize);
    aesCtr.Initialize(&aes, testVector.iv, testVector.ivSize);
    if (isInPlace)
    {
        std::memcpy(output, testVector.cipherText, testVector.textSize);
        aesCtr.Update(output, sizeof(output), output, testVector.textSize);
    }
    else
    {
        aesCtr.Update(output, sizeof(output), testVector.cipherText, testVector.textSize);
    }
    EXPECT_ARRAY_EQ(output, testVector.plainText, testVector.textSize);
}

template <typename T>
void EncryptDataBufferingTest(const AesCtrTestVector& testVector)
{
    T                           aes;
    nn::crypto::CtrEncryptor<T> aesCtr;
    nn::Bit8                    output[160];
    const size_t                MaxInputSize = std::min((T::BlockSize * 4), testVector.textSize);

    aes.Initialize(testVector.key, testVector.keySize);

    // 複数ブロック＋端数の処理までテストするため、適当なブロックサイズ分までの各入力サイズについてテストする
    for (size_t inputSize = 1; inputSize < MaxInputSize; inputSize++)
    {
        aesCtr.Initialize(&aes, testVector.iv, testVector.ivSize);

        const nn::Bit8* pInput = testVector.plainText;
        nn::Bit8*       pOutput = output;
        size_t          outBufSize = sizeof(output);
        size_t          remaining = testVector.textSize;
        while (remaining)
        {
            size_t processSize = std::min(inputSize, remaining);
            size_t ret = aesCtr.Update(pOutput, outBufSize, pInput, processSize);
            pInput += processSize;
            remaining -= processSize;
            // 処理された分だけ出力バッファを進める
            pOutput += ret;
            outBufSize -= ret;
        }

        EXPECT_ARRAY_EQ(output, testVector.cipherText, testVector.textSize);
    }
}

template <typename T>
void DecryptDataBufferingTest(const AesCtrTestVector& testVector)
{
    T                           aes;
    nn::crypto::CtrDecryptor<T> aesCtr;
    nn::Bit8                    output[160];
    const size_t                MaxInputSize = std::min((T::BlockSize * 4), testVector.textSize);

    aes.Initialize(testVector.key, testVector.keySize);

    // 複数ブロック＋端数の処理までテストするため、適当なブロックサイズ分までの各入力サイズについてテストする
    for (size_t inputSize = 1; inputSize < MaxInputSize; inputSize++)
    {
        aesCtr.Initialize(&aes, testVector.iv, testVector.ivSize);

        const nn::Bit8* pInput = testVector.cipherText;
        nn::Bit8*       pOutput = output;
        size_t          outBufSize = sizeof(output);
        size_t          remaining = testVector.textSize;
        while (remaining)
        {
            size_t processSize = std::min(inputSize, remaining);
            size_t ret = aesCtr.Update(pOutput, outBufSize, pInput, processSize);
            pInput += processSize;
            remaining -= processSize;
            // 処理された分だけ出力バッファを進める
            pOutput += ret;
            outBufSize -= ret;
        }

        EXPECT_ARRAY_EQ(output, testVector.plainText, testVector.textSize);
    }
}

/**
  @brief   AES-CTR による暗号化をテストします。

  @details
  テストベクトルを用いて 128bit、192bit、256bit
  のそれぞれの鍵サイズで正しく暗号化できることをテストします。
 */
TEST(AesCtrTest, Encryption)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesCtrTestVectors[i].keySize)
        {
        case 16:
            EncryptionTest<nn::crypto::AesEncryptor128>(aesCtrTestVectors[i], false);
            break;
        case 24:
            EncryptionTest<nn::crypto::AesEncryptor192>(aesCtrTestVectors[i], false);
            break;
        case 32:
            EncryptionTest<nn::crypto::AesEncryptor256>(aesCtrTestVectors[i], false);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   AES-CTR による復号化をテストします。

  @details
  テストベクトルを用いて 128bit、192bit、256bit
  のそれぞれの鍵サイズで正しく暗号化できることをテストします。
 */
TEST(AesCtrTest, Decryption)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesCtrTestVectors[i].keySize)
        {
        case 16:
            // CTR モードでは復号化時に指定するのも Encryptor
            DecryptionTest<nn::crypto::AesEncryptor128>(aesCtrTestVectors[i], false);
            break;
        case 24:
            DecryptionTest<nn::crypto::AesEncryptor192>(aesCtrTestVectors[i], false);
            break;
        case 32:
            DecryptionTest<nn::crypto::AesEncryptor256>(aesCtrTestVectors[i], false);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   AES-CTR による同一入出力バッファ上での暗号化をテストします。

  @details
  テストベクトルを用いて 128bit、192bit、256bit
  のそれぞれの鍵サイズで正しく暗号化できることをテストします。
 */
TEST(AesCtrTest, InPlaceEncryption)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesCtrTestVectors[i].keySize)
        {
        case 16:
            EncryptionTest<nn::crypto::AesEncryptor128>(aesCtrTestVectors[i], true);
            break;
        case 24:
            EncryptionTest<nn::crypto::AesEncryptor192>(aesCtrTestVectors[i], true);
            break;
        case 32:
            EncryptionTest<nn::crypto::AesEncryptor256>(aesCtrTestVectors[i], true);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   AES-CTR による同一入出力バッファ上での復号化をテストします。

  @details
  テストベクトルを用いて 128bit、192bit、256bit
  のそれぞれの鍵サイズで正しく暗号化できることをテストします。
 */
TEST(AesCtrTest, InPlaceDecryption)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesCtrTestVectors[i].keySize)
        {
        case 16:
            // CTR モードでは復号化時に指定するのも Encryptor
            DecryptionTest<nn::crypto::AesEncryptor128>(aesCtrTestVectors[i], true);
            break;
        case 24:
            DecryptionTest<nn::crypto::AesEncryptor192>(aesCtrTestVectors[i], true);
            break;
        case 32:
            DecryptionTest<nn::crypto::AesEncryptor256>(aesCtrTestVectors[i], true);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   AES-CTR による暗号化を入力サイズを変えてテストします。

  @details
  適当な大きさまで入力サイズを変えて、正しく暗号化できることをテストします。
 */
TEST(AesCtrTest, EncryptDataBuffering)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesCtrTestVectors[i].keySize)
        {
        case 16:
            EncryptDataBufferingTest<nn::crypto::AesEncryptor128>(aesCtrTestVectors[i]);
            break;
        case 24:
            EncryptDataBufferingTest<nn::crypto::AesEncryptor192>(aesCtrTestVectors[i]);
            break;
        case 32:
            EncryptDataBufferingTest<nn::crypto::AesEncryptor256>(aesCtrTestVectors[i]);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   AES-CTR による復号化を入力サイズを変えてテストします。

  @details
  適当な大きさまで入力サイズを変えて、正しく復号化できることをテストします。
 */
TEST(AesCtrTest, DecryptDataBuffering)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesCtrTestVectors[i].keySize)
        {
        case 16:
            // CTR モードでは復号化時に指定するのも Encryptor
            DecryptDataBufferingTest<nn::crypto::AesEncryptor128>(aesCtrTestVectors[i]);
            break;
        case 24:
            DecryptDataBufferingTest<nn::crypto::AesEncryptor192>(aesCtrTestVectors[i]);
            break;
        case 32:
            DecryptDataBufferingTest<nn::crypto::AesEncryptor256>(aesCtrTestVectors[i]);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   Aes128CtrEncryptor/Decryptor による暗号化・復号化をテストします。
 */
TEST(AesCtrTest, UtilityClassInterface)
{
    nn::Bit8 output[160];

    // 暗号化のテスト
    for (int i = 0; i < TestVectorCount; i++)
    {
        // 128bit 鍵のみテスト
        if (aesCtrTestVectors[i].keySize != 16)
        {
            continue;
        }

        nn::crypto::Aes128CtrEncryptor aes128CtrEnc;

        aes128CtrEnc.Initialize(aesCtrTestVectors[i].key, aesCtrTestVectors[i].keySize,
                                aesCtrTestVectors[i].iv, aesCtrTestVectors[i].ivSize);
        aes128CtrEnc.Update(output, sizeof(output),
                            aesCtrTestVectors[i].plainText, aesCtrTestVectors[i].textSize);

        EXPECT_ARRAY_EQ(output, aesCtrTestVectors[i].cipherText, aesCtrTestVectors[i].textSize);
    }

    // 復号化のテスト
    for (int i = 0; i < TestVectorCount; i++)
    {
        // 128bit 鍵のみテスト
        if (aesCtrTestVectors[i].keySize != 16)
        {
            continue;
        }

        nn::crypto::Aes128CtrDecryptor aes128CtrDec;

        aes128CtrDec.Initialize(aesCtrTestVectors[i].key, aesCtrTestVectors[i].keySize,
                                aesCtrTestVectors[i].iv, aesCtrTestVectors[i].ivSize);
        aes128CtrDec.Update(output, sizeof(output),
                            aesCtrTestVectors[i].cipherText, aesCtrTestVectors[i].textSize);

        EXPECT_ARRAY_EQ(output, aesCtrTestVectors[i].plainText, aesCtrTestVectors[i].textSize);
    }
}

/**
  @brief   ユーティリティ関数による暗号化・復号化をテストします。
 */
TEST(AesCtrTest, UtilityFunctionInterface)
{
    nn::Bit8 output[160];

    // 暗号化のテスト
    for (int i = 0; i < TestVectorCount; i++)
    {
        // 128bit 鍵のみテスト
        if (aesCtrTestVectors[i].keySize != 16)
        {
            continue;
        }

        nn::crypto::EncryptAes128Ctr(output, sizeof(output),
                                     aesCtrTestVectors[i].key, aesCtrTestVectors[i].keySize,
                                     aesCtrTestVectors[i].iv, aesCtrTestVectors[i].ivSize,
                                     aesCtrTestVectors[i].plainText, aesCtrTestVectors[i].textSize);

        EXPECT_ARRAY_EQ(output, aesCtrTestVectors[i].cipherText, aesCtrTestVectors[i].textSize);
    }

    // 復号化のテスト
    for (int i = 0; i < TestVectorCount; i++)
    {
        // 128bit 鍵のみテスト
        if (aesCtrTestVectors[i].keySize != 16)
        {
            continue;
        }

        nn::crypto::DecryptAes128Ctr(output, sizeof(output),
                                     aesCtrTestVectors[i].key, aesCtrTestVectors[i].keySize,
                                     aesCtrTestVectors[i].iv, aesCtrTestVectors[i].ivSize,
                                     aesCtrTestVectors[i].cipherText, aesCtrTestVectors[i].textSize);

        EXPECT_ARRAY_EQ(output, aesCtrTestVectors[i].plainText, aesCtrTestVectors[i].textSize);
    }
}

/**
  @brief   デストラクタで内部データがクリアされることをテストします。
 */
template <size_t KeySize>
void CtrDestructorTest()
{
    // Dummy key and iv
    const uint8_t key[32] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    };
    const uint8_t iv[16] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
    };

    // Encryptor test
    {
        nn::crypto::AesEncryptor<KeySize> aes;
        aes.Initialize(key, KeySize);

        nn::crypto::CtrEncryptor<nn::crypto::AesEncryptor<KeySize>> mode;
        mode.Initialize(&aes, iv, mode.IvSize);

        // 明示的にデストラクタを呼んで呼び出し前後でのメモリクリアを確認する
        EXPECT_ARRAY_NONZERO(&mode, sizeof(mode));
        mode.~CtrEncryptor<nn::crypto::AesEncryptor<KeySize>>();
        EXPECT_ARRAY_ZERO(&mode, sizeof(mode));

        // AES 本体は暗号利用モードのほうだけクリアされてもクリアされていない
        EXPECT_ARRAY_NONZERO(&aes, sizeof(aes));
    }

    // Decryptor test
    {
        nn::crypto::AesEncryptor<KeySize> aes;
        aes.Initialize(key, KeySize);

        nn::crypto::CtrDecryptor<nn::crypto::AesEncryptor<KeySize>> mode;
        mode.Initialize(&aes, iv, mode.IvSize);

        // 明示的にデストラクタを呼んで呼び出し前後でのメモリクリアを確認する
        EXPECT_ARRAY_NONZERO(&mode, sizeof(mode));
        mode.~CtrDecryptor<nn::crypto::AesEncryptor<KeySize>>();
        EXPECT_ARRAY_ZERO(&mode, sizeof(mode));

        // AES 本体は暗号利用モードがクリアされてもクリアされていない
        EXPECT_ARRAY_NONZERO(&aes, sizeof(aes));
    }
}

TEST(AesCtrTest, Destructor)
{
    CtrDestructorTest<16>();
    CtrDestructorTest<24>();
    CtrDestructorTest<32>();
}

/**
  @brief   Update に 0 バイトが渡されても正常に計算できることをチェックします。
 */
TEST(AesCtrTest, ZeroByteEncryption)
{
    nn::crypto::AesEncryptor128                           aes;
    nn::crypto::CtrEncryptor<nn::crypto::AesEncryptor128> aesCtr;
    nn::Bit8                                              output[32];

    // 初期化
    aes.Initialize(aesCtrTestVectors[0].key, aesCtrTestVectors[0].keySize);
    aesCtr.Initialize(&aes, aesCtrTestVectors[0].iv, aesCtrTestVectors[0].ivSize);

    // 平文の入力前に 0 バイトの Update を呼ぶ
    EXPECT_EQ(aesCtr.Update(output, sizeof(output), nullptr, 0), 0);

    aesCtr.Update(output, sizeof(output),
                  aesCtrTestVectors[0].plainText, aesCtrTestVectors[0].textSize);

    // 出力結果が正しいことを確認
    EXPECT_ARRAY_EQ(output, aesCtrTestVectors[0].cipherText, aesCtrTestVectors[0].textSize);
}

TEST(AesCtrTest, ZeroByteDecryption)
{
    nn::crypto::AesEncryptor128                           aes;
    nn::crypto::CtrDecryptor<nn::crypto::AesEncryptor128> aesCtr;
    nn::Bit8                                              output[32];

    // 初期化
    aes.Initialize(aesCtrTestVectors[0].key, aesCtrTestVectors[0].keySize);
    aesCtr.Initialize(&aes, aesCtrTestVectors[0].iv, aesCtrTestVectors[0].ivSize);

    // 平文の入力前に 0 バイトの Update を呼ぶ
    EXPECT_EQ(aesCtr.Update(output, sizeof(output), nullptr, 0), 0);

    aesCtr.Update(output, sizeof(output),
                  aesCtrTestVectors[0].cipherText, aesCtrTestVectors[0].textSize);

    // 出力結果が正しいことを確認
    EXPECT_ARRAY_EQ(output, aesCtrTestVectors[0].plainText, aesCtrTestVectors[0].textSize);
}
