﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/crypto/crypto_AesEncryptor.h>
#include <nn/crypto/crypto_AesDecryptor.h>

#include "testCrypto_Util.h"

struct TestVector
{
    nn::Bit8  pKey[32];
    int       keySize;
    nn::Bit8  pPlainText[16];
    int       textSize;
    nn::Bit8  pCipherText[16];
};

/* NIST FIPS 197 Appendix C に掲載されているテストベクトル */
const TestVector aesTestVector128 =
{
    {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f},
    16,
    {0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff},
    16,
    {0x69, 0xc4, 0xe0, 0xd8, 0x6a, 0x7b, 0x04, 0x30, 0xd8, 0xcd, 0xb7, 0x80, 0x70, 0xb4, 0xc5, 0x5a},
};

const TestVector aesTestVector192 =
{
    {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
     0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17},
    24,
    {0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff},
    16,
    {0xdd, 0xa9, 0x7c, 0xa4, 0x86, 0x4c, 0xdf, 0xe0, 0x6e, 0xaf, 0x70, 0xa0, 0xec, 0x0d, 0x71, 0x91},
};

const TestVector aesTestVector256 =
{
    {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
     0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f},
    32,
    {0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff},
    16,
    {0x8e, 0xa2, 0xb7, 0xca, 0x51, 0x67, 0x45, 0xbf, 0xea, 0xfc, 0x49, 0x90, 0x4b, 0x49, 0x60, 0x89},
};

template <typename T>
void EncryptionTest(const TestVector& testVector, bool isInPlace)
{
    T        aes;
    nn::Bit8 output[aes.BlockSize];

    EXPECT_GE(sizeof(output), T::BlockSize);

    aes.Initialize(testVector.pKey, testVector.keySize);
    if (isInPlace)
    {
        std::memcpy(output, testVector.pPlainText, testVector.textSize);
        aes.EncryptBlock(output, sizeof(output), output, testVector.textSize);
    }
    else
    {
        aes.EncryptBlock(output, sizeof(output), testVector.pPlainText, testVector.textSize);
    }
    EXPECT_ARRAY_EQ(output, testVector.pCipherText, aes.BlockSize);
}

template <typename T>
void DecryptionTest(const TestVector& testVector, bool isInPlace)
{
    T        aes;
    nn::Bit8 output[aes.BlockSize];

    EXPECT_GE(sizeof(output), T::BlockSize);

    aes.Initialize(testVector.pKey, testVector.keySize);
    if (isInPlace)
    {
        std::memcpy(output, testVector.pCipherText, testVector.textSize);
        aes.DecryptBlock(output, sizeof(output), output, testVector.textSize);
    }
    else
    {
        aes.DecryptBlock(output, sizeof(output), testVector.pCipherText, testVector.textSize);
    }
    EXPECT_ARRAY_EQ(output, testVector.pPlainText, aes.BlockSize);
}

template <typename T>
void DestructorTest(const TestVector& testVector)
{
    T aes;

    // 初期化
    aes.Initialize(testVector.pKey, testVector.keySize);

    // インスタンスの破棄前に明示的にデストラクタを呼んでメモリクリアを確認する
    EXPECT_ARRAY_NONZERO(&aes, sizeof(T));
    aes.~T();
    EXPECT_ARRAY_ZERO(&aes, sizeof(T));
}

/**
  @brief   AES の単一ブロックの暗号化をテストします。

  @details
  NIST FIPS 197 Appendix C に掲載されているテストベクトルを用いて 128bit、192bit、256bit
  のそれぞれの鍵サイズで正しくブロックが暗号化できることをテストします。
 */
TEST(AesTest, Encryption)
{
    EncryptionTest<nn::crypto::AesEncryptor128>(aesTestVector128, false);
    EncryptionTest<nn::crypto::AesEncryptor192>(aesTestVector192, false);
    EncryptionTest<nn::crypto::AesEncryptor256>(aesTestVector256, false);
}

/**
  @brief   AES の単一ブロックの復号化をテストします。

  @details
  NIST FIPS 197 Appendix C に掲載されているテストベクトルを用いて 128bit、192bit、256bit
  のそれぞれの鍵サイズで正しくブロックが復号化できることをテストします。
 */
TEST(AesTest, Decryption)
{
    DecryptionTest<nn::crypto::AesDecryptor128>(aesTestVector128, false);
    DecryptionTest<nn::crypto::AesDecryptor192>(aesTestVector192, false);
    DecryptionTest<nn::crypto::AesDecryptor256>(aesTestVector256, false);
}

/**
  @brief   AES の単一ブロックの同一入出力バッファ上での暗号化をテストします。
 */
TEST(AesTest, InPlaceEncryption)
{
    EncryptionTest<nn::crypto::AesEncryptor128>(aesTestVector128, true);
    EncryptionTest<nn::crypto::AesEncryptor192>(aesTestVector192, true);
    EncryptionTest<nn::crypto::AesEncryptor256>(aesTestVector256, true);
}

/**
  @brief   AES の単一ブロックの同一入出力バッファ上での復号化をテストします。
 */
TEST(AesTest, InPlaceDecryption)
{
    DecryptionTest<nn::crypto::AesDecryptor128>(aesTestVector128, true);
    DecryptionTest<nn::crypto::AesDecryptor192>(aesTestVector192, true);
    DecryptionTest<nn::crypto::AesDecryptor256>(aesTestVector256, true);
}

/**
  @brief   デストラクタで内部データがクリアされることをテストします。
 */
TEST(AesTest, Destructor)
{
    DestructorTest<nn::crypto::AesEncryptor128>(aesTestVector128);
    DestructorTest<nn::crypto::AesEncryptor192>(aesTestVector192);
    DestructorTest<nn::crypto::AesEncryptor256>(aesTestVector256);

    DestructorTest<nn::crypto::AesDecryptor128>(aesTestVector128);
    DestructorTest<nn::crypto::AesDecryptor192>(aesTestVector192);
    DestructorTest<nn::crypto::AesDecryptor256>(aesTestVector256);
}
