﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using System.Diagnostics;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace TestUtility
{
    public delegate void StreamWriteDelegate(Stream output);

    /// <summary>
    /// 複数の一時ファイルや一時ディレクトリのパスを生成し、保持します。
    /// using と共に使うことを想定しており、作成したファイルやディレクトリをスコープから抜ける際にまとめて削除します。
    /// </summary>
    public class ScopedTemporaryFileHolder : IDisposable
    {
        /// <summary>
        /// TestContext を利用し、UnitTest の OutputPath を一時ファイルの配置場所にします。
        /// </summary>
        public ScopedTemporaryFileHolder(TestContext testContext, bool afterDelete = true)
        {
            TemporaryRoot = new TestPath(testContext).GetOutputRoot();
            AfterDelete = afterDelete;
        }

        /// <summary>
        /// 一時ファイルの配置場所を指定して作成します。
        /// </summary>
        public ScopedTemporaryFileHolder(string temporaryRoot, bool afterDelete = true)
        {
            TemporaryRoot = temporaryRoot;
            AfterDelete = afterDelete;
        }

        /// <summary>
        /// 一時ファイルのパスを作成します。
        /// ファイルは作成しません。
        /// 作成したファイルはこのインスタンスにより管理され、Dispose 時に削除されます。
        /// </summary>
        public FileInfo CreateTemporaryFilePath(string prefix = "", string extension = "")
        {
            var tempFile = new FileInfo(Path.Combine(TemporaryRoot, prefix + Path.GetRandomFileName() + extension));

            if (tempFile.Exists)
            {
                throw new System.IO.IOException(string.Format("Temporary file already exists: {0}", tempFile.FullName));
            }

            AddTemporaryFile(tempFile);
            return tempFile;
        }

        /// <summary>
        /// 一時ディレクトリのパスを作成します。
        /// ディレクトリは作成しません。
        /// 作成したディレクトリはこのインスタンスにより管理され、Dispose 時に削除されます。
        /// </summary>
        public DirectoryInfo CreateTemporaryDirectoryPath(string prefix = "")
        {
            var tempDirectory = new DirectoryInfo(Path.Combine(TemporaryRoot, prefix + Path.GetRandomFileName()));

            if (tempDirectory.Exists)
            {
                throw new System.IO.IOException(string.Format("Temporary directory already exists: {0}", tempDirectory.FullName));
            }

            AddTemporaryDirectory(tempDirectory);
            return tempDirectory;
        }

        /// <summary>
        /// StreamWriteDelegateを呼び出し一時ファイルを作成します。
        /// 作成したファイルはこのインスタンスにより管理され、Dispose 時に削除されます。
        /// </summary>
        public FileInfo CreateTemporaryFile(StreamWriteDelegate writer, string prefix = "", string extension = ".tmp")
        {
            FileInfo tempFile = CreateTemporaryFilePath(prefix, extension);

            using (var outputStream = tempFile.Create())
            {
                writer(outputStream);
            }

            return tempFile;
        }

        /// <summary>
        /// テキストとファイル名を指定して一時ファイルを作成します。
        /// 作成したファイルはこのインスタンスにより管理され、Dispose 時に削除されます。
        /// </summary>
        public FileInfo CreateTemporaryFile(FileInfo tempFile, string data)
        {
            using (var outputStream = tempFile.CreateText())
            {
                outputStream.Write(data);
            }

            this.AddTemporaryFile(tempFile);

            return tempFile;
        }

        /// <summary>
        /// ディレクトリとファイル名とテキストを指定して一時ファイルを作成します。
        /// 作成したファイルはこのインスタンスにより管理され、Dispose 時に削除されます。
        /// </summary>
        public FileInfo CreateTemporaryFile(DirectoryInfo directory, string filename, string data)
        {
            FileInfo tempFile = new FileInfo(Path.Combine(directory.FullName, filename));

            using (var outputStream = tempFile.CreateText())
            {
                outputStream.Write(data);
            }

            this.AddTemporaryFile(tempFile);

            return tempFile;
        }

        /// <summary>
        /// バイト列を指定して一時ファイルを作成します。
        /// 作成したファイルはこのインスタンスにより管理され、Dispose 時に削除されます。
        /// </summary>
        public FileInfo CreateTemporaryFile(byte[] data, string prefix = "", string extension = ".tmp")
        {
            FileInfo tempFile = CreateTemporaryFilePath(prefix, extension);

            using (var outputStream = tempFile.Create())
            {
                outputStream.Write(data, 0, data.Length);
            }

            return tempFile;
        }

        /// <summary>
        /// 文字列を指定して一時ファイルを作成します。
        /// 作成したファイルはこのインスタンスにより管理され、Dispose 時に削除されます。
        /// </summary>
        public FileInfo CreateTemporaryFile(string text, string prefix = "", string extension = ".tmp")
        {
            FileInfo tempFile = CreateTemporaryFilePath(prefix, extension);

            using (var outputStream = tempFile.CreateText())
            {
                outputStream.Write(text);
            }

            return tempFile;
        }

        /// <summary>
        /// 一時ディレクトリを作成します。
        /// 作成したディレクトリはこのインスタンスにより管理され、Dispose 時に削除されます。
        /// </summary>
        public DirectoryInfo CreateTemporaryDirectory(string prefix = "")
        {
            var temporaryDirectory = CreateTemporaryDirectoryPath(prefix);
            temporaryDirectory.Create();
            return temporaryDirectory;
        }

        /// <summary>
        /// 一時ファイルを追加します。
        /// 別の場所で作成した一時ファイルを削除したい場合に使用します。
        /// </summary>
        public void AddTemporaryFile(FileInfo fileInfo)
        {
            files.Add(fileInfo);
        }

        /// <summary>
        /// 一時ディレクトリを追加します。
        /// 別の場所で作成した一時ディレクトリを削除したい場合に使用します。
        /// </summary>
        public void AddTemporaryDirectory(DirectoryInfo directoryInfo)
        {
            directories.Add(directoryInfo);
        }

        /// <summary>
        /// 一時ファイルや一時ディレクトリを全て削除します。
        /// Dispose 時に呼ばれるものなので、通常は明示的に呼ぶ必要はありません。
        /// </summary>
        public void DeleteAll()
        {
            foreach (var file in files)
            {
                if (File.Exists(file.FullName))
                {
                    File.Delete(file.FullName);
                }
            }

            foreach (var directory in directories)
            {
                if (Directory.Exists(directory.FullName))
                {
                    Directory.Delete(directory.FullName, true);
                }
            }
        }

        public void Dispose()
        {
            if (AfterDelete)
            {
                DeleteAll();
            }
        }

        public string TemporaryRoot { get; private set; }
        public bool AfterDelete { get; private set; }
        private List<FileInfo> files = new List<FileInfo>();
        private List<DirectoryInfo> directories = new List<DirectoryInfo>();
    }
}
