﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>
#include <string>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <direct.h>  // _getcwd
#endif  // defined(NN_BUILD_CONFIG_OS_WIN)

#include <nnt.h>
#include <nnt/codecUtil/testCodec_Util.h>
#include <nnt/codecUtil/testCodec_FixtureBase.h>

#include <nn/codec.h>
#include <nn/codec/codec_OpusEncoder.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/crypto.h> // nn::crypto::Sha1Generator

// #define ENABLE_SAVE_DECODE_DATA

namespace {

// マルチインスタンステストでの実行スレッド数 (実行コア数とは別)
const int ThreadCount = 4;

// テスト対象ファイルの拡張子
const std::string targetFileExtension = "opus";
const std::string outputFileExtension = "raw";

const std::string GetTestDataRootDirectoryPath() NN_NOEXCEPT
{
    return nnt::codec::util::GetCodecTestBinariesPath()  + "\\testCodec_OpusCodec";
}

std::string RemoveFileExtention(const std::string& path)
{
    std::string::size_type position = path.find_last_of(".");
    return (position == std::string::npos) ? path : path.substr(0, position);
}

nn::codec::OpusCodingMode GetCodingModeFromString(const char* filename)
{
    if (nullptr != std::strstr(filename, "Silk"))
    {
        return nn::codec::OpusCodingMode_Silk;
    }
    return nn::codec::OpusCodingMode_Celt;
}

int GetPreSkipSampleCountFromEncoder(
    int sampleRate,
    int channelCount,
    nn::codec::OpusCodingMode codingMode
)
{
    nn::codec::OpusEncoder encoder;
    const std::size_t size = encoder.GetWorkBufferSize(sampleRate, channelCount);
    NN_ASSERT(size > 0);
    void* buffer = std::malloc(size);
    NN_ASSERT(nullptr != buffer);
    encoder.Initialize(sampleRate, channelCount, buffer, size);
    encoder.BindCodingMode(codingMode);
    const auto preSkipSampleCount = encoder.GetPreSkipSampleCount();
    encoder.Finalize();
    std::free(buffer);
    return preSkipSampleCount;
}

template <typename DecoderType>
void Tester(void* arg)
{
    const auto entry = *reinterpret_cast<nn::fs::DirectoryEntry*>(arg);

    auto opusData = new uint8_t[static_cast<int32_t>(entry.fileSize)];
    ASSERT_NE(nullptr, opusData);
    nn::fs::FileHandle srcHandle;
    std::string srcPath = nnt::codec::util::GetMountPointRom() + ":/Inputs/" + entry.name;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&srcHandle, srcPath.c_str(), nn::fs::OpenMode_Read));
    const std::string dstFileName = RemoveFileExtention(entry.name) + "." + outputFileExtension;
#if defined(ENABLE_SAVE_DECODE_DATA)
    nn::fs::FileHandle dstHandle;
    std::string dstPath = nnt::codec::util::GetMountPointHost() + ":/Outputs/" + dstFileName;
    nn::fs::DeleteFile(dstPath.c_str());
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(dstPath.c_str(), 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&dstHandle, dstPath.c_str(), nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
#endif // defined(ENABLE_SAVE_DECODE_DATA)

    // read input file
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(srcHandle, 0, opusData, static_cast<int32_t>(entry.fileSize)));

    // read header info
    nn::codec::OpusBasicInfo opusBasicInfo;
    std::memcpy(&opusBasicInfo, opusData, sizeof(opusBasicInfo));
    EXPECT_EQ(nn::codec::OpusInfoType_BasicInfo, opusBasicInfo.header.type);
    const auto sampleRate = opusBasicInfo.sampleRate;
    const auto channelCount = opusBasicInfo.channelCount;
    const auto preSkipSampleCount = opusBasicInfo.preSkipSampleCount;
    const auto codingMode = GetCodingModeFromString(entry.name);
    EXPECT_EQ(GetPreSkipSampleCountFromEncoder(sampleRate, channelCount, codingMode), preSkipSampleCount);

    // initialize OpusDecoder
    DecoderType* decoder = nnt::codec::CreateOpusDecoder<DecoderType>(0);
    EXPECT_FALSE(decoder->IsInitialized());
    void* workBuffer = std::malloc(decoder->GetWorkBufferSize(sampleRate, channelCount));
    EXPECT_TRUE(workBuffer != nullptr);
    EXPECT_EQ(nn::codec::OpusResult_Success, decoder->Initialize(sampleRate, channelCount, workBuffer, decoder->GetWorkBufferSize(sampleRate, channelCount)));
    EXPECT_TRUE(decoder->IsInitialized());
    EXPECT_EQ(sampleRate, decoder->GetSampleRate());
    EXPECT_EQ(channelCount, decoder->GetChannelCount());

    // get the head of opus data
    unsigned char* p = reinterpret_cast<unsigned char*>(opusData + opusBasicInfo.dataInfoOffset);
    nn::codec::OpusInfoHeader dataInfoHeader;
    std::memcpy(&dataInfoHeader, p, sizeof(dataInfoHeader));
    NN_ABORT_UNLESS(dataInfoHeader.type == nn::codec::OpusInfoType_DataInfo);
    std::size_t remainedCount = dataInfoHeader.size;
    p += sizeof(dataInfoHeader);

    // calculate sha1 value for verifying the encoded data.
    nn::crypto::Sha1Generator sha1;
    sha1.Initialize();

    // prepare output pcm buffer
    auto pcmData = new int16_t[960 * channelCount];
    ASSERT_NE(nullptr, pcmData);
#if defined(ENABLE_SAVE_DECODE_DATA)
    int64_t writeOffset = 0;
#endif // defined(ENABLE_SAVE_DECODE_DATA)

    // dummy decord to test Reset()
    if(remainedCount > 0u)
    {
        std::size_t consumed;
        int sampleCount;
        EXPECT_EQ(nn::codec::OpusResult_Success, decoder->DecodeInterleaved(&consumed, &sampleCount, pcmData, 960 * channelCount * sizeof(int16_t), p, remainedCount));
    }
    decoder->Reset();

    // decode loop
    while (remainedCount > 0)
    {
        std::size_t consumed;
        int sampleCount;
        EXPECT_EQ(nn::codec::OpusResult_Success, decoder->DecodeInterleaved(&consumed, &sampleCount, pcmData, 960 * channelCount * sizeof(int16_t), p, remainedCount));
        const std::size_t decodedSampleSize = sampleCount * channelCount * sizeof(int16_t);
        sha1.Update(pcmData, decodedSampleSize);
        p += consumed;
        remainedCount -= consumed;

#if defined(ENABLE_SAVE_DECODE_DATA)
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::WriteFile(dstHandle, writeOffset, pcmData, decodedSampleSize, nn::fs::WriteOption()));
        writeOffset += decodedSampleSize;
#endif // defined(ENABLE_SAVE_DECODE_DATA)

    }

    // finalization
    nn::fs::CloseFile(srcHandle);
#if defined(ENABLE_SAVE_DECODE_DATA)
    nn::fs::FlushFile(dstHandle);
    nn::fs::CloseFile(dstHandle);
#endif // defined(ENABLE_SAVE_DECODE_DATA)

    decoder->Finalize();
    EXPECT_FALSE(decoder->IsInitialized());
    nnt::codec::DeleteOpusDecoder<DecoderType>(decoder);
    std::free(workBuffer);

    delete[] opusData;
    delete[] pcmData;

    // ----------------
    // 結果検証
    // ----------------
    const size_t hashSize = nn::crypto::Sha1Generator::HashSize;
    uint8_t hash[2][hashSize];

    // 結果のハッシュ値
    sha1.GetHash(hash[0], hashSize);

    // opus_demo で同じ処理をさせた結果のハッシュ値
    const std::string refPath = nnt::codec::util::GetReferenceDirectoryPath() + "/" + dstFileName + ".sig";
    ASSERT_TRUE(nnt::codec::util::GetHashValueFromFile(hash[1], hashSize, refPath));
    // 照合
    EXPECT_EQ(0, std::memcmp(hash[0], hash[1], hashSize));
}

// Fixture 定義 (型をテストケースで使うためだけの定義)
template <class T>
class OpusDecoderDecodeTest : public ::testing::Test
{
protected:
    typedef T DecoderType;
};

}  // namespace

/**
 * @brief       インスタンス化する型を列挙します。
 */
typedef ::testing::Types<nn::codec::OpusDecoder, nn::codec::HardwareOpusDecoder> Implementations;

TYPED_TEST_CASE(OpusDecoderDecodeTest, Implementations);

/**
 * @brief       単体デコーダによるデコードテストです。
 * @details
 * MultiInstanceDecodeInterleaved がパスするなら不要なので、"DISABLED_" を付しておく。
 */
TYPED_TEST(OpusDecoderDecodeTest, DISABLED_DecodeInterleaved)
{
    size_t mountRomCacheBufferSize = 4 * 1024;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheBufferSize));
    char* mountRomCacheBuffer = new char[mountRomCacheBufferSize];
    NN_UTIL_SCOPE_EXIT
    {
        delete[] mountRomCacheBuffer;
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountRom(nnt::codec::util::GetMountPointRom().c_str(), mountRomCacheBuffer, mountRomCacheBufferSize));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost(nnt::codec::util::GetMountPointHost().c_str(), GetTestDataRootDirectoryPath().c_str()));

    nn::fs::DirectoryHandle directoryHandle;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(&directoryHandle, (nnt::codec::util::GetMountPointRom() + ":/Inputs/").c_str(), nn::fs::OpenDirectoryMode_File));
    int64_t entryCount;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetDirectoryEntryCount(&entryCount, directoryHandle));

    auto entries = new nn::fs::DirectoryEntry[static_cast<int32_t>(entryCount)];
    ASSERT_NE(nullptr, entries);

    {
        int64_t tmp;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadDirectory(&tmp, entries, directoryHandle, entryCount));
        EXPECT_EQ(tmp, entryCount);
    }
    for (int i = 0; i < entryCount; ++i)
    {
        if (nnt::codec::util::GetFileExtension(entries[i].name, 1) != targetFileExtension)
        {
            continue;
        }
        SCOPED_TRACE(entries[i].name);
        ASSERT_EQ(entries[i].directoryEntryType, nn::fs::DirectoryEntryType_File);
        Tester<typename TestFixture::DecoderType>(&entries[i]);
    }

    nn::fs::CloseDirectory(directoryHandle);

    nn::fs::Unmount(nnt::codec::util::GetMountPointRom().c_str());
    nn::fs::Unmount(nnt::codec::util::GetMountPointHost().c_str());
}

/**
 * @brief       複数のデコーダインスタンスにより、デコード処理を並走させます。
 */
TYPED_TEST(OpusDecoderDecodeTest, MultiInstanceDecodeInterleaved)
{
    size_t mountRomCacheBufferSize = 4 * 1024;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheBufferSize));
    char* mountRomCacheBuffer = new char[mountRomCacheBufferSize];
    NN_UTIL_SCOPE_EXIT
    {
        delete[] mountRomCacheBuffer;
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountRom(nnt::codec::util::GetMountPointRom().c_str(), mountRomCacheBuffer, mountRomCacheBufferSize));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost(nnt::codec::util::GetMountPointHost().c_str(), GetTestDataRootDirectoryPath().c_str()));

    nn::fs::DirectoryHandle directoryHandle;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(&directoryHandle, (nnt::codec::util::GetMountPointRom() + ":/Inputs/").c_str(), nn::fs::OpenDirectoryMode_File));
    int64_t entryCount;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetDirectoryEntryCount(&entryCount, directoryHandle));

    auto entries = new nn::fs::DirectoryEntry[static_cast<int32_t>(entryCount)];
    ASSERT_NE(nullptr, entries);

    {
        int64_t tmp;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadDirectory(&tmp, entries, directoryHandle, entryCount));
        EXPECT_EQ(tmp, entryCount);
    }
    // テストスレッド関連の初期化
    nnt::codec::util::TesterManager manager;
    manager.Initialize(ThreadCount, std::malloc);

    // スレッドを実行するコアを管理するモジュールの初期化＆確認
    nnt::codec::util::CoreSelector coreSelector;
    ASSERT_TRUE(coreSelector.IsInitialized());
    ASSERT_TRUE(coreSelector.GetCoreCount() > 0);

    for (int i = 0; i < entryCount; ++i)
    {
        if (nnt::codec::util::GetFileExtension(entries[i].name, 1) != targetFileExtension)
        {
            continue;
        }
        SCOPED_TRACE(entries[i].name);
        ASSERT_EQ(entries[i].directoryEntryType, nn::fs::DirectoryEntryType_File);
        ASSERT_TRUE(manager.Run(entries[i], Tester<typename TestFixture::DecoderType>, coreSelector.PostIncrement()));
    }
    manager.Finalize(std::free);

    nn::fs::CloseDirectory(directoryHandle);

    nn::fs::Unmount(nnt::codec::util::GetMountPointRom().c_str());
    nn::fs::Unmount(nnt::codec::util::GetMountPointHost().c_str());
}
