﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Assert.h>

#include <nn/audio.h>
#include <nnt/audioUtil/testAudio_Util.h>
#include <nnt/codecUtil/testCodec_Util.h>
#include <nnt/audioUtil/testAudio_Constants.h>
#include <nn/codec/codec_AdpcmDecoder.h>
#include <nn/util/util_BytePtr.h>
#include <memory>

namespace
{
const std::string g_AdpcmSourceTestFileName = "/shutter.adpcm";
const std::string g_AdpcmConvertedTestFileName = "/AdpcmDecoder1ch.wav";

NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[14 * 1024 * 1024];
nn::mem::StandardAllocator g_WaveBufferAllocator;

std::size_t ReadAdpcmFile(nn::audio::AdpcmHeaderInfo* header, void** adpcmData, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t size;
    uint8_t adpcmheader[nn::audio::AdpcmHeaderSize];

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    *adpcmData = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size) - sizeof(adpcmheader), nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*adpcmData);

    result = nn::fs::ReadFile(handle, 0, adpcmheader, sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    result = nn::fs::ReadFile(handle, sizeof(adpcmheader), *adpcmData, static_cast<size_t>(size) - sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(handle);

    nn::audio::ParseAdpcmHeader(header, adpcmheader, sizeof(adpcmheader));

    return static_cast<std::size_t>(size) - sizeof(adpcmheader);
}

void DecodeAdpcmToWav(nnt::audio::util::WaveFile* waveFile)
{
    nn::audio::AdpcmHeaderInfo* header;
    void* dataSe;

    const std::string SourceFile = g_SourceDataFolder + g_AdpcmSourceTestFileName;
    header = reinterpret_cast<nn::audio::AdpcmHeaderInfo*>(g_WaveBufferAllocator.Allocate(sizeof(nn::audio::AdpcmHeaderInfo), NN_ALIGNOF(nn::audio::AdpcmHeaderInfo)));
    std::size_t dataSeSize = ReadAdpcmFile(header, &dataSe, SourceFile.c_str());
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN const size_t arraySize = header->sampleCount * sizeof(int16_t);

    nn::codec::AdpcmParameter parameter;
    nn::codec::AdpcmContext context = { 0 };

    auto pcm = std::unique_ptr<int16_t[]>(new int16_t[arraySize]);

    context._predScale = header->loopContext.predScale;
    for (int i = 0; i < 16; i++)
    {
        parameter._coefficients[i] = header->parameter.coefficients[i];
    }

    int sampleCount = header->sampleCount;
    int frame = 0;
    int currentSample = 0;
    while (currentSample + nn::codec::AdpcmFrameSampleCount <= sampleCount)
    {
        nn::codec::DecodeAdpcm(
            nn::util::BytePtr(pcm.get(), currentSample * sizeof(int16_t)).Get<int16_t>(),
            &context,
            &parameter,
            nn::util::ConstBytePtr(dataSe, frame * nn::codec::AdpcmFrameSize).Get(),
            nn::codec::AdpcmFrameSize,
            nn::codec::AdpcmFrameSampleCount);

        frame++;
        currentSample += nn::codec::AdpcmFrameSampleCount;
    }

    const int8_t* buf = reinterpret_cast<int8_t*>(pcm.get());
    waveFile->Write(buf, dataSeSize);
}

void RunWaveComparisonTestAdpcm(int channelCount, std::string audioTestFileName) NN_NOEXCEPT
{
    std::string mountPath = nnt::codec::util::GetCodecTestBinariesPath() + "\\testCodec_AdpcmDecoder";

    // Test environment
    auto params = nnt::audio::util::GetAudioRendererParameterForWaveComparison();

    const std::string TargetFile = nnt::audio::util::GetResultDataDirectory() + audioTestFileName;
    const std::string GoldenFile = g_RefernerceDataFolder + audioTestFileName;
    const nn::TimeSpan CheckDuration = nn::TimeSpan::FromSeconds(1);
    const int32_t BitRate = 16;
    const int ChannelCountMax = 6;

    NN_ASSERT_LESS_EQUAL(channelCount, ChannelCountMax);

    // Prepare Test Components
    /////////////////////////////////////////////////////////////////////////////
    nnt::audio::util::SimpleFsUtility fsUtil;
    nnt::audio::util::WaveFile testWav;

    ASSERT_TRUE(fsUtil.InitializeFileSystem(g_MountName.c_str(), mountPath.c_str()).IsSuccess());

    int64_t estimatedFileSize = nnt::audio::util::WaveFile::GetRequiredFileSize(params.sampleRate, channelCount, BitRate, CheckDuration);
    nn::Result rt = fsUtil.PrepareNewFile(TargetFile.c_str(), estimatedFileSize);
    ASSERT_TRUE(rt.IsSuccess()) << "Failed to create new file.";
    testWav.Open(TargetFile.c_str(), params.sampleRate, static_cast<uint16_t>(channelCount), BitRate);

    g_WaveBufferAllocator.Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));

    // Write Adpcm Data to Wav File
    //////////////////////////////////////////////////////////////////////////
    DecodeAdpcmToWav(&testWav);
    testWav.Close();
    nnt::audio::util::SimpleVerify(TargetFile, GoldenFile);
}

} // namespace


/**
 * @brief       Normal case tests
 */
TEST(AdpcmDecoder, NormalCase)
{
    nn::codec::AdpcmParameter parameter;
    nn::codec::AdpcmContext context;

    std::memset(&parameter, 0, sizeof(parameter));
    std::memset(&context, 0, sizeof(context));

    unsigned char adpcm[nn::codec::AdpcmFrameSize];
    std::memset(adpcm, 0, sizeof(adpcm));

    int16_t pcm[nn::codec::AdpcmFrameSampleCount];
    const int16_t PcmInitialValue = -1;

    for (int i = 0; i < nn::codec::AdpcmFrameSampleCount; ++i)
    {
        pcm[i] = PcmInitialValue;
    }
    nn::codec::DecodeAdpcm(pcm, &context, &parameter, adpcm, nn::codec::AdpcmFrameSize, nn::codec::AdpcmFrameSampleCount);
    for (int i = 0; i < nn::codec::AdpcmFrameSampleCount; ++i)
    {
        EXPECT_EQ(pcm[i], 0);
    }

    for (int i = 0; i < nn::codec::AdpcmFrameSampleCount; ++i)
    {
        pcm[i] = PcmInitialValue;
    }
    nn::codec::DecodeAdpcm(pcm, &context, &parameter, adpcm, nn::codec::AdpcmFrameSize, nn::codec::AdpcmFrameSampleCount / 2);
    for (int i = 0; i < nn::codec::AdpcmFrameSampleCount / 2; ++i)
    {
        EXPECT_EQ(pcm[i], 0);
    }
    for (int i = nn::codec::AdpcmFrameSampleCount / 2; i < nn::codec::AdpcmFrameSampleCount; ++i)
    {
        EXPECT_EQ(pcm[i], PcmInitialValue);
    }

    nn::codec::GetAdpcmLoopContext(&context, &parameter, adpcm, nn::codec::AdpcmFrameSize, nn::codec::AdpcmFrameSampleCount);
    EXPECT_EQ(context._predScale, 0);
    EXPECT_EQ(context._history[0], 0);
    EXPECT_EQ(context._history[1], 0);
}

TEST(AdpcmDecoder, WaveComparison1ch)
{
    RunWaveComparisonTestAdpcm(1, g_AdpcmConvertedTestFileName);
}
