﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstring>
#include <cstdlib>
#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nnt.h>

#include <nn/cdmsc.h>

const uint32_t DataSeed = 0xdeadbeef;
const uint32_t DataSize = 1024 * 1024 * 8;
const uint64_t StartLba = 0;

NN_ALIGNAS(4096) uint8_t pattern[DataSize];

void Fx3MakeGaloisPattern(uint8_t* pData, uint32_t size, uint32_t seed, uint32_t *newSeed)
{
    uint32_t processed;
    uint32_t lfsr = seed;

    for (processed = 0; processed < size; processed += 4)
    {
        uint32_t remaining = size - processed;
        /* taps: 32 31 29 1; characteristic polynomial: x^32 + x^31 + x^29 + x + 1 */
        lfsr = (lfsr >> 1) ^ (uint32_t)((0 - (lfsr & 1u)) & 0xd0000001u);
        /* we cannot always have 32-bit alignment in buffer, so it's best
           to memcpy */
        memcpy(pData + processed, &lfsr, (remaining >= 4) ? 4 : remaining);
    }

    if (newSeed != nullptr)
    {
        *newSeed = lfsr;
    }
}

bool Fx3CheckGaloisPattern(uint8_t* pData, uint32_t size, uint32_t seed, uint32_t *newSeed)
{
    uint32_t processed;
    uint32_t lfsr = seed;

    for (processed = 0; processed < size; processed += 4)
    {
        uint32_t remaining = size - processed;
        /* taps: 32 31 29 1; characteristic polynomial: x^32 + x^31 + x^29 + x + 1 */
        lfsr = (lfsr >> 1) ^ (uint32_t)((0 - (lfsr & 1u)) & 0xd0000001u);

        /* we cannot always have 32-bit alignment in buffer, so it's best
           to memcmp */
        if (memcmp(pData + processed, &lfsr, (remaining >= 4) ? 4 : remaining) != 0)
        {
            NN_LOG("FAIL: Data varification error!\n");
            return false;
        }
    }

    if (newSeed != nullptr)
    {
        *newSeed = lfsr;
    }

    return true;
}

TEST(SmokeTest, SmokeTest)
{
    nn::os::EventType       deviceAvailableEvent;
    nn::os::EventType       deviceDetachEvent;
    nn::cdmsc::UnitProfile  profile;

    uint32_t seed;

    nn::os::Tick tick0;
    nn::os::Tick tick1;
    nn::os::Tick ticks;

    uint64_t lba;
    uint32_t block;
    uint64_t us;
    uint32_t totalBytes;

    nn::os::InitializeEvent(&deviceAvailableEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&deviceDetachEvent,    false, nn::os::EventClearMode_AutoClear);

    NNT_EXPECT_RESULT_SUCCESS(
        nn::cdmsc::Initialize(
            &deviceAvailableEvent,
            std::aligned_alloc,
            [](void *p, size_t size) {
                NN_UNUSED(size);
                free(p);
            }
        )
    );

    NN_LOG("Waiting for device attach...\n");

    nn::os::WaitEvent(&deviceAvailableEvent);

    NNT_EXPECT_RESULT_SUCCESS(nn::cdmsc::Probe(&deviceDetachEvent, &profile));

    NN_LOG("Device attached!\n");

    block = DataSize / profile.blockSize;

    tick0 = nn::os::GetSystemTick();
    NN_LOG("W:");
    lba  = StartLba;
    seed = DataSeed;
    for (int i = 0; i < 32; i++)
    {
        Fx3MakeGaloisPattern(pattern, DataSize, seed, &seed);

        NNT_EXPECT_RESULT_SUCCESS(
            nn::cdmsc::Write(pattern, profile.handle, lba, block)
        );

        lba += block;

        if (i % 8 == 0)
        {
            NN_LOG(" ");
        }
        NN_LOG(".");
    }
    NN_LOG("\n");
    tick1 = nn::os::GetSystemTick();

    ticks = tick1 - tick0;
    us = ticks.ToTimeSpan().GetMicroSeconds();
    totalBytes = 32 * DataSize;
    NN_LOG("Write 0x%x bytes in %dus, %fMB/s\n", totalBytes, us,
           (float)totalBytes / ((float)us / (float)1000000) / (1024 * 1024));

    tick0 = nn::os::GetSystemTick();
    NN_LOG("R:");
    lba  = StartLba;
    seed = DataSeed;
    for (int i = 0; i < 32; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(
            nn::cdmsc::Read(pattern, profile.handle, lba, block)
        );

        ASSERT_TRUE(
            Fx3CheckGaloisPattern(pattern, DataSize, seed, &seed)
        );

        lba += block;

        if (i % 8 == 0)
        {
            NN_LOG(" ");
        }
        NN_LOG(".");
    }
    NN_LOG("\n");
    tick1 = nn::os::GetSystemTick();

    ticks = tick1 - tick0;
    us = ticks.ToTimeSpan().GetMicroSeconds();
    totalBytes = 32 * DataSize;
    NN_LOG("Read 0x%x bytes in %dus, %fMB/s\n", totalBytes, us,
           (float)totalBytes / ((float)us / (float)1000000) / (1024 * 1024));

    NN_LOG("Waiting for device detach...\n");

    nn::os::WaitEvent(&deviceDetachEvent);

    NN_LOG("Device Detached!\n");

    NNT_EXPECT_RESULT_SUCCESS(nn::cdmsc::Finalize());

    nn::os::FinalizeEvent(&deviceAvailableEvent);
    nn::os::FinalizeEvent(&deviceDetachEvent);
}

