﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/gtest/gtest.h>
#include <nn/init/init_Malloc.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/nn_Assert.h>
#include <nnt/teamcity/testTeamcity_Logger.h>

#include "../Fx3Control/Fx3Control.h"
#include "CdmscFx3MessageLogger.h"

namespace nnt {
namespace usb {
namespace cdmsc {

const size_t g_HeapSize = 16 * 1024 * 1024;
bool g_DownloadFirmware = true;

class Fx3MscTestEnvironment : public ::testing::Environment
{
public:
    virtual void SetUp() override
    {
        Fx3Initialize(WAIT_SECONDS_FOR_ATTACH, g_DownloadFirmware);
        FX3_LOG("Initialized Fx3MscTestEnvironment.\n");
        // if (IsFx3Initialized())
        // {
        //     //Fx3CheckFirmwareVersion();
        // }
    }

    virtual void TearDown() override
    {
        if (IsFx3Initialized())
        {
            Fx3Finalize();
            FX3_LOG("Finalized Fx3MscTestEnvironment.\n");
        }
        else
        {
            FX3_LOG("Fx3 Failed to Finalize. Never Initialized.\n");
        }
    }
};

extern "C" void nninitStartup()
{
    nn::Result result;

    /* set heap size */
    result = nn::os::SetMemoryHeapSize(g_HeapSize);
    if (!result.IsSuccess()) {
        FX3_LOG("Failed SetMemoryHeapSize\n");
        return;
    }
    FX3_LOG("SetMemoryHeapSize 0x%x OK\n", g_HeapSize);

    uintptr_t address;
    result = nn::os::AllocateMemoryBlock(&address, g_HeapSize / 2);
    NN_ASSERT(result.IsSuccess());

    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), g_HeapSize / 2);

    ::testing::AddGlobalTestEnvironment(new Fx3MscTestEnvironment);
}

void ParseArgument(int argc, char** argv)
{
    static const char TestArgFixedFw[] = "--fixedfw";

    while (argc--)
    {
        if (strncmp(argv[0], TestArgFixedFw, sizeof(TestArgFixedFw) - 1) == 0)
        {
            g_DownloadFirmware = false;
        }
        argv++;
    }
}

extern "C" void nnMain()
{
    int    argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    // TeamCity の表示を適切にするため、イベントリスナの登録を一旦すべて解除し、
    // ServiceMessageLogger -> デフォルトのイベントリスナ の順で登録し直す。
    ::testing::TestEventListeners& listeners = ::testing::UnitTest::GetInstance()->listeners();
    ::testing::TestEventListener* defaultResultPrinter = listeners.Release(listeners.default_result_printer());
    listeners.Append(new CdmscFx3MessageLogger());
    listeners.Append(new nnt::teamcity::ServiceMessageLogger());
    listeners.Append(defaultResultPrinter);

    ParseArgument(argc, argv);

    const int exitCode = RUN_ALL_TESTS();

    ::nnt::Exit(exitCode);
}

} // hs
} // usb
} // nnt
