﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Log.h>

namespace nnt{ namespace capsrv{

    enum MakerNoteVersion
    {
        MakerNoteVersion_Version0 = 0,
        MakerNoteVersion_Version1,
        MakerNoteVersion_Version2,
        MakerNoteVersion_Version3,
        MakerNoteVersion_End,
    };

    enum MakerNoteEntryTag
    {
        //MakerNoteEntryTag_Signature              = 0x0000,

        MakerNoteEntryTag_Integer0               = 0x0001,
        MakerNoteEntryTag_Bytes0                 = 0x0002,
        MakerNoteEntryTag_Integer1               = 0x0003,
        MakerNoteEntryTag_Bytes1                 = 0x0004,

        MakerNoteEntryTag_Integer2Req            = 0x0005,
        MakerNoteEntryTag_Integer2Opt            = 0x0006,
        MakerNoteEntryTag_Integer3Req            = 0x0007,
        MakerNoteEntryTag_Integer3Opt            = 0x0008,

        MakerNoteEntryTag_Unknown                = 0x1000,

        MakerNoteEntryTag_Max,
    };

    template<int TLength>
    struct ArrayValueImpl
    {
        static const int Length = TLength;
        char value[Length];

        bool operator==(const ArrayValueImpl<TLength>& v) const NN_NOEXCEPT
        {
            return (std::memcmp(this->value, v.value, sizeof(*this)) == 0);
        }
    };

    typedef ArrayValueImpl<16> ArrayValue;

    struct MakerNoteInfo
    {
    public:
        static const uint32_t Integer0Default =  1;
        static const uint32_t Integer1Default = 11;
        static const uint32_t Integer2ReqDefault = 21;
        static const uint32_t Integer2OptDefault = 22;
        static const uint32_t Integer3ReqDefault = 31;
        static const uint32_t Integer3OptDefault = 32;
    public:
        uint32_t version;
        uint32_t integer0;
        uint32_t integer1;
        ArrayValue bytes0;
        ArrayValue bytes1;

        uint32_t integer2req;
        uint32_t integer2opt;
        uint32_t integer3req;
        uint32_t integer3opt;

    public:
        void Fill(int v) NN_NOEXCEPT
        {
            std::memset(this, v, sizeof(*this));
        }

        void Dump(int indent = 0) NN_NOEXCEPT
        {
            std::string indentString;
            for(int i = 0; i < indent; i++)
            {
                indentString += " ";
            }
            NN_LOG("%sversion    : %u\n", indentString.c_str(), this->version);
            NN_LOG("%sinteger0   : %u\n", indentString.c_str(), this->integer0);
            NN_LOG("%sinteger1   : %u\n", indentString.c_str(), this->integer1);
            NN_LOG("%sinteger2req: %u\n", indentString.c_str(), this->integer2req);
            NN_LOG("%sinteger2opt: %u\n", indentString.c_str(), this->integer2opt);
            NN_LOG("%sinteger3req: %u\n", indentString.c_str(), this->integer3req);
            NN_LOG("%sinteger3opt: %u\n", indentString.c_str(), this->integer3opt);

            ArrayValue v = {};
            v = bytes0;
            v.value[15] = 0;
            NN_LOG("%sbytes0     : %s\n", indentString.c_str(), &v);
            v = bytes1;
            v.value[15] = 0;
            NN_LOG("$sbytes1     : %s\n", indentString.c_str(), &v);
        }
    };

    nn::Result ParseMakerNoteForTestFoundation(
        MakerNoteInfo* pOutValue,
        const void* pMakerNoteData,
        size_t makerNoteDataSize
        ) NN_NOEXCEPT;

    size_t GenerateMakerNoteForTestFoundation(
        void* buffer,
        size_t size,
        const MakerNoteInfo& info
        ) NN_NOEXCEPT;

    int GetEntryMinCountForTestFoundataion(uint32_t version, uint16_t tag) NN_NOEXCEPT;
    int GetEntryMaxCountForTestFoundataion(uint32_t version, uint16_t tag) NN_NOEXCEPT;

}}
