﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <vector>
#include <list>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt.h>

#include <nn/capsrv/capsrv_AlbumFileSizeLimit.h>
#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nn/capsrv/capsrv_AlbumControl.h>
#include <nn/capsrv/capsrv_AlbumTesting.h>
#include <nn/capsrv/movie/capsrv_MovieWriterFileSystem.h>
#include <nn/capsrv/movie/capsrv_MovieMetaData.h>
#include "../../Common/testCapsrv_Macro.h"
#include "../../Common/testCapsrv_DirectAlbumAccessor.h"
#include "../../Common/testCapsrv_FileInfo.h"
#include "../../Common/testCapsrv_AlbumEntryUtility.h"
#include "../../Common/testCapsrv_MovieFileCreator.h"
#include "testCapsrv_StartupTestCase.h"
#include "detail/testCapsrv_TestMovieStreamWrapperBase.h"
#include "detail/testCapsrv_TestMovieStreamRandomAccessImpl.h"

//static int RepeatCount = 100;

#define NNT_CAPSRV_IO_LOG(...) NN_LOG(__VA_ARGS__)

#ifndef NNT_CAPSRV_IO_LOG
#define NNT_CAPSRV_IO_LOG(...)
#endif

#define NNT_CAPSRV_FILELIST(varList, varCount)  \
    const nnt::capsrv::FileInfo varList[] = {  \
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 00, 0x0123456789ABCDEF, .mp4, 10000000, 12345),   \
        NNT_CAPSRV_FILEINFO(NA, 2016, 04, 02, 20, 50, 59, 01, 0xAAAAAAAAAAAAAAAA, .mp4, 10000000, 12346),   \
    };  \
    static const int varCount = sizeof(varList) / sizeof(varList[0]);   \
    NN_UNUSED(varCount);    \
    NN_UNUSED(varList);

namespace {
    class TestFile
        : public nnt::capsrv::detail::TestMovieStreamWrapperBase
    {
    public:
        void Initialize(nn::fs::FileHandle h) NN_NOEXCEPT
        {
            m_FileHandle = h;
        }

        void Finalize() NN_NOEXCEPT
        {
            m_FileHandle = {};
        }

    protected:
        nn::Result WriteImpl(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::WriteFile(m_FileHandle, offset, buffer, size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
        }

        nn::Result ReadImpl(size_t* pOutReadSize, void* buffer, size_t size, int64_t offset) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::ReadFile(pOutReadSize, m_FileHandle, offset, buffer, size);
        }

        nn::Result GetSizeImpl(int64_t* pOutValue) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::GetFileSize(pOutValue, m_FileHandle);
        }

        nn::Result SetSizeImpl(int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::SetFileSize(m_FileHandle, size);
        }

        nn::Result FlushImpl() NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::FlushFile(m_FileHandle);
        }

        const uint8_t* GetInternalStorageImpl(size_t* pOutSize) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(pOutSize);
            return nullptr;
        }

    private:
        nn::fs::FileHandle m_FileHandle;
    };
}

namespace {

    void TestMovieWriterFileSystemImpl(
        size_t sizeToWriteMax,
        size_t sizeToReadMax,
        int64_t chunkSize,
        int64_t cacheCount
    )
    {
        NN_LOG("sizeToWriteMax  = %llu\n", sizeToWriteMax);
        NN_LOG("sizeToReadMax   = %llu\n", sizeToReadMax);
        NN_LOG("cacheChunkSize  = %llu\n", chunkSize);
        NN_LOG("cacheChunkCount = %llu\n", cacheCount);

        // ワークメモリ
        size_t mwfsSize = nn::capsrv::movie::MovieWriterFileSystem::GetRequiredMemorySizeForFile(chunkSize, cacheCount);
        size_t mwfsAlignment = nn::capsrv::movie::MovieWriterFileSystem::GetRequiredMemoryAlignmentForFile();
        NN_LOG("memorySize  = %llu\n", mwfsSize);
        NN_LOG("memoryAlign = %llu\n", mwfsAlignment);
        std::vector<char> mem(mwfsSize);

        auto pMeta = new nn::capsrv::movie::MovieMetaData();
        NN_UTIL_SCOPE_EXIT{ delete pMeta; };

        // ファイルシステム
        nn::capsrv::movie::MovieWriterFileSystem mwfs;
        mwfs.Initialize("Movie");
        NN_UTIL_SCOPE_EXIT{ mwfs.Finalize(); };

        std::mt19937 rand;

        NNT_CAPSRV_FILELIST(fileList, fileCount);
        for(int i = 0; i < fileCount; i++)
        {
            auto fileId = fileList[i].GetAlbumFileId();
            if(!nn::capsrv::IsAlbumMounted(fileId.storage))
            {
                continue;
            }
            auto fileData = fileList[i].GetData();

            //nn::capsrv::AlbumFileId fileId = {};
            //NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GenerateCurrentAlbumFileId(&fileId, {0x123456}, nn::capsrv::AlbumFileContents_Movie));


            // 動画書込みストリームを作成
            nn::capsrv::AlbumMovieWriteStreamHandle h = {};
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieWriteStream(&h, fileId));

            // データ部分の書込み
            int64_t dataSize = 0;
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));
            {
                // ファイルシステムにアタッチ
                nn::fs::FileHandle handle = {};
                // 失敗するのはアタッチ可能な数の上限に到達したときだけ。
                NN_ABORT_UNLESS_RESULT_SUCCESS(mwfs.AttachMovieWriteStreamDataSection(&handle, h, chunkSize, cacheCount, mem.data(), mem.size()));
                NN_UTIL_SCOPE_EXIT{ mwfs.DetachMovieWriteStreamDataSection(handle); };

                TestFile tf;
                tf.Initialize(handle);
                {
                    ptrdiff_t dataSize = fileData.size();
                    ptrdiff_t remain = dataSize;
                    ptrdiff_t pos = 0;
                    while(remain > 0)
                    {
                        size_t sizeToWrite = std::min(static_cast<size_t>(remain), sizeToWriteMax);
                        NNT_EXPECT_RESULT_SUCCESS(tf.Write(pos, fileData.data() + pos, sizeToWrite));
                        remain -= sizeToWrite;
                        pos += sizeToWrite;
                    }
                    NNT_EXPECT_RESULT_SUCCESS(tf.Flush());
                    EXPECT_TRUE(tf.CheckStorage());
                }
                NNT_EXPECT_RESULT_SUCCESS(tf.GetSize(&dataSize));
                tf.Finalize();
            }
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::EndAlbumMovieWriteStreamDataSection(h));

            // メタ部分の書込み
            auto meta = nnt::capsrv::MovieFileCreator::CreateMetaWithRandomImageData(fileId, dataSize, rand, nnt::capsrv::MovieMeta::Flag_ZeroSignature);
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamMetaSection(h));
            {
                NNT_EXPECT_RESULT_SUCCESS(
                    nn::capsrv::WriteMetaToAlbumMovieWriteStream(h, meta.value.data(), meta.value.size(), meta.makerNoteVersion, meta.makerNoteOffset, meta.makerNoteSize)
                );
            }
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::EndAlbumMovieWriteStreamMetaSection(h));

            // ファイルへの書込みを完了
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::FinishAlbumMovieWriteStream(h));
            // ファイルを他のプロセスから見えるようにして完了。
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::CommitAlbumMovieWriteStream(h));

            //TODO:
            //// 書き込んだ内容を検証
            //size_t writtenDataSize = 0;
            //size_t requiredBufferSize = fileData.size() + static_cast<size_t>(chunkSize);
            //std::vector<uint8_t> writtenData(requiredBufferSize);
            //NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotFile(&writtenDataSize, writtenData.data(), writtenData.size(), &fileId));
            //fileList[i].CheckData(writtenData.data(), static_cast<int64_t>(fileData.size()));
        }
    }

}



TEST(AlbumAccessApi, MovieWriterFileSystem_Write)
{
    const size_t SizeToWriteMax = 4 * 1024;
    const size_t SizeToReadMax = 64 * 1024;
    const int64_t ChunkSize = nn::capsrv::AlbumMovieDataUnitSize;
    const int64_t CacheCount = 4;

    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControlExtension());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    TestMovieWriterFileSystemImpl(SizeToWriteMax, SizeToReadMax, ChunkSize, CacheCount);

    TestMovieWriterFileSystemImpl(       200, SizeToReadMax, ChunkSize, CacheCount);
    TestMovieWriterFileSystemImpl(250 * 1024, SizeToReadMax, ChunkSize, CacheCount);

    TestMovieWriterFileSystemImpl(SizeToWriteMax,        200, ChunkSize, CacheCount);
    TestMovieWriterFileSystemImpl(SizeToWriteMax, 250 * 1024, ChunkSize, CacheCount);

    TestMovieWriterFileSystemImpl(SizeToWriteMax, SizeToReadMax, 2 * ChunkSize, CacheCount);
    TestMovieWriterFileSystemImpl(SizeToWriteMax, SizeToReadMax, 3 * ChunkSize, CacheCount);

    TestMovieWriterFileSystemImpl(SizeToWriteMax, SizeToReadMax, ChunkSize, 1);
    TestMovieWriterFileSystemImpl(SizeToWriteMax, SizeToReadMax, ChunkSize, 2);
}



TEST(AlbumAccessApi, MovieWriterFileSystem_Random)
{
    const int64_t ChunkSize = nn::capsrv::AlbumMovieDataUnitSize;
    const int64_t CacheCount = 4;
    std::mt19937 rand(0x24572);

    nnt::capsrv::detail::TestMovieStreamRandomAccess::Config config = {};
    config.loopCount = 1000;
    config.chunkSize = ChunkSize;
    config.offsetMax = 10 * 1024 * 1024;
    config.sizeMax   = 10 * 1024;
    config.offsetCutoffThreshold   = 256 * 1024;
    config.sizeCutoffThreshold     = 256;
    config.probabilityWrite  = 0.60f;
    config.probabilityRead   = 0.15f;
    config.probabilityResize = 0.20f;
    config.probabilityFlush  = 0.05f;

    nnt::capsrv::StartupTestCase();
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControlExtension());
    NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

    // ワークメモリ
    size_t mwfsSize = nn::capsrv::movie::MovieWriterFileSystem::GetRequiredMemorySizeForFile(ChunkSize, CacheCount);
    size_t mwfsAlignment = nn::capsrv::movie::MovieWriterFileSystem::GetRequiredMemoryAlignmentForFile();
    NN_LOG("memorySize  = %llu\n", mwfsSize);
    NN_LOG("memoryAlign = %llu\n", mwfsAlignment);
    std::vector<char> mem(mwfsSize);

    // ファイルシステム
    nn::capsrv::movie::MovieWriterFileSystem mwfs;
    mwfs.Initialize("Movie");
    NN_UTIL_SCOPE_EXIT{ mwfs.Finalize(); };

    NNT_CAPSRV_FILELIST(fileList, fileCount);
    for(int i = 0; i < fileCount; i++)
    {
        NN_LOG("file: %d\n", i);
        auto fileId = fileList[i].GetAlbumFileId();
        if(!nn::capsrv::IsAlbumMounted(fileId.storage))
        {
            continue;
        }

        nn::capsrv::AlbumMovieWriteStreamHandle h = {};
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieWriteStream(&h, fileId));
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::StartAlbumMovieWriteStreamDataSection(h));

        nn::fs::FileHandle handle = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(mwfs.AttachMovieWriteStreamDataSection(&handle, h, ChunkSize, CacheCount, mem.data(), mem.size()));
        TestFile tf;
        tf.Initialize(handle);

        nnt::capsrv::detail::TestMovieStreamRandomAccess::Test(tf, config, rand);

        NNT_EXPECT_RESULT_SUCCESS(tf.Flush());
        EXPECT_TRUE(tf.CheckStorage());
        tf.Finalize();
        mwfs.DetachMovieWriteStreamDataSection(handle);
        NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::DiscardAlbumMovieWriteStreamImpl(h));
    }
}// NOLINT(impl/function_size)
