﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


// 以下の API のテストを行います。
// - GetAlbumFileList
// - GetAlbumFileSize
// - GetAlbumFileSizeLimit

#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>

#include <nn/fs.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/result/result_HandlingUtility.h>

#include <nnt.h>

#include <nn/capsrv/capsrv_Result.h>
#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nn/capsrv/capsrv_AlbumFileSizeLimit.h>
#include "../../Common/testCapsrv_DirectAlbumAccessor.h"
#include "../../Common/testCapsrv_FileInfo.h"
#include "../../Common/testCapsrv_Macro.h"
#include "testCapsrv_StartupTestCase.h"
#include "../../../Programs/Iris/Sources/Libraries/capsrv/server/album/capsrvServer_AlbumPathUtility.h"

TEST(AlbumAccessApi, GetAlbumFileList_Movie)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();

    // 既にあるファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    // 作成するファイルのリスト。
    // 時刻はユニークにしておくこと。
    const nnt::capsrv::FileInfo FileList[] = {
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 00, 0x0123456789ABCDEF, .mp4, 1024, 12345),
        NNT_CAPSRV_FILEINFO(NA, 2016, 04, 02, 20, 50, 59, 01, 0xAAAAAAAAAAAAAAAA, .mp4, 2048, 12346),
        NNT_CAPSRV_FILEINFO(NA, 2016, 05, 03, 18, 31, 58, 02, 0xBBBBBBBBBBBBBBBB, .mp4, 512, 12347),
        NNT_CAPSRV_FILEINFO(SD, 2016, 07, 04, 15, 20, 57, 03, 0xCCCCCCCCCCCCCCCC, .mp4, 128, 12348),
        NNT_CAPSRV_FILEINFO(SD, 2016, 08, 06, 14, 10, 56, 04, 0xDDDDDDDDDDDDDDDD, .mp4, 4832, 12349),
    };
    static const int TotalFileCount = sizeof(FileList) / sizeof(FileList[0]);

    // ファイル数が 0 になっていることを確認
    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE(s)
    {
        int count = 0;
        nn::Result result = nn::capsrv::GetAlbumFileCount(&count, s);
        EXPECT_TRUE(result.IsSuccess());
        EXPECT_EQ(0, count);
    }

    // ファイル作成
    int createdFileCount = 0;
    for(int i = 0; i < TotalFileCount; i++)
    {
        if(FileList[i].Create())
        {
            createdFileCount++;
        }
    }

    // ファイル数が作ったファイルの数になっていることを確認
    {
        int countSum = 0;
        NNT_CAPSRV_FOREACH_MOUNTED_STORAGE(s)
        {
            int count = 0;
            nn::Result result = nn::capsrv::GetAlbumFileCount(&count, s);
            EXPECT_TRUE(result.IsSuccess());
            countSum += count;
        }
        EXPECT_EQ(createdFileCount, countSum);
    }

    // ファイルサイズの上限を確認
    EXPECT_EQ(nn::capsrv::AlbumFileSizeLimit_Movie, nn::capsrv::GetAlbumFileSizeLimit(nn::capsrv::AlbumFileContents_Movie));

    // ファイルリスト取得関数のテスト。
    nn::capsrv::AlbumEntry albumEntries[TotalFileCount];
    NNT_CAPSRV_FOREACH_DIRECT_MOUNTED_STORAGE(s)
    {
        // チェックリスト。全部 1 になったら OK。
        char checkList[TotalFileCount] = {};
        int expectedCount = 0;
        // 調べているストレージにあるはずのファイル数を事前にカウント
        // 調べているストレージと異なる場所のファイルはチェックリストに 1 をいれておく
        for(int i = 0; i < TotalFileCount; i++)
        {
            if(FileList[i].storage == s)
            {
                expectedCount++;
            }
            else
            {
                checkList[i] = 1;
            }
        }

        // ファイルリストを取得
        int count = 0;
        nn::Result albumResult = nn::capsrv::GetAlbumFileList(&count, albumEntries, TotalFileCount, s);
        EXPECT_TRUE(albumResult.IsSuccess());
        EXPECT_EQ(expectedCount, count);

        // 時刻が一致するものをチェック。
        for(int k = 0; k < count; k++)
        {
            const nn::capsrv::AlbumEntry& e = albumEntries[k];
            for(int i = 0; i < TotalFileCount; i++)
            {
                const nnt::capsrv::FileInfo& fileinfo = FileList[i];
                if(e.fileId.storage != fileinfo.storage)
                {
                    continue;
                }
                if(fileinfo.IsMatchTime(e))
                {
                    EXPECT_EQ(0, checkList[i]); // 2 回一致するものがあったらおかしい
                    checkList[i] = 1;

                    // ついでにファイルサイズを確認
                    size_t fileSize = 0;
                    EXPECT_TRUE(nn::capsrv::GetAlbumFileSize(&fileSize, &e.fileId).IsSuccess());
                    EXPECT_EQ(fileinfo.filesize, static_cast<int64_t>(fileSize));

                    break;
                }
            }
        }

        // チェックリストが全部 1 になっていることを確認
        int checkNgCount = 0;
        for(int i = 0; i < TotalFileCount; i++)
        {
            if(checkList[i] == 0)
            {
                checkNgCount++;
            }
        }
        EXPECT_EQ(0, checkNgCount);
    }

    // 作ったファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    nn::capsrv::FinalizeAlbumAccess();
    SUCCEED();
}// NOLINT(impl/function_size)

TEST(AlbumAccessApi, GetAlbumFileList_Movie_ManyFiles)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();

    // 既にあるファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());


    static const int FileCount = 200;
    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE_I(s, storage)
    {
        // Create many files to stress the end of the entry buffer
        for (int fidx = 0; fidx < FileCount; fidx++)
        {
            nnt::capsrv::FileInfo finfo =
                NNT_CAPSRV_FILEINFO(NA, 2031, 12, 31, 00, 00, 00, 00, 0x0123456789ABCDEF, .mp4, 1024, 12345);

            // Override file info members with what we need
            finfo.storage = storage;
            finfo.time.id = static_cast<uint8_t> (fidx % 100);
            finfo.time.minute = static_cast<uint8_t> ((fidx / 100) % 60);
            EXPECT_TRUE(finfo.Create());
        }
    }

    if (nn::capsrv::IsAlbumMounted(nn::capsrv::AlbumStorage_Sd))
    {
        // Create many folders to stress the beginning of the entry buffer
        for (int year = 2020; year < 2032; year++)
        {
            char path[256];
            int yearLength = 0;
            yearLength = nn::util::SNPrintf(path, sizeof(path), "%s%04d", nnt::capsrv::DirectAlbumAccessor::GetRootPath(NNT_CAPSRV_STORAGE_VALUES_SD), year);
            nn::Result result = nn::fs::CreateDirectory(path);
            NN_ABORT_UNLESS(result.IsSuccess() || nn::fs::ResultPathAlreadyExists::Includes(result));

            for (int month = 0; month < 14; month++)
            {
                int monthLength = yearLength + nn::util::SNPrintf(path + yearLength, sizeof(path) - static_cast<size_t> (yearLength), "/%02d", month);
                result = nn::fs::CreateDirectory(path);
                NN_ABORT_UNLESS(result.IsSuccess() || nn::fs::ResultPathAlreadyExists::Includes(result));
                for (int day = 0; day < 35; day++)
                {
                    // Only handle corner cases (else, the test would take too long)
                    if (!(
                        year == 2020 || year == 2031 ||
                        month == 0 || month == 13 ||
                        day == 0 || day == 34
                        ))
                    {
                        continue;
                    }

                    nn::util::SNPrintf(path + monthLength, sizeof(path) - static_cast<size_t> (monthLength), "/%02d", day);
                    result = nn::fs::CreateDirectory(path);
                    NN_ABORT_UNLESS(result.IsSuccess() || nn::fs::ResultPathAlreadyExists::Includes(result));
                }
            }
        }

        const char BadFolder1[] = "2016/01/bad/test";
        const char BadFolder2[] = "2016/bad/01/test";
        const char BadFolder3[] = "bad/01/01/test";
        const char* (BadFolders[]) = { BadFolder1, BadFolder2, BadFolder3 };
        for (int badFolderIdx = 0; badFolderIdx < 3; badFolderIdx++)
        {
            char path[256];
            nn::util::SNPrintf(path, sizeof(path), "%s%s/test", nnt::capsrv::DirectAlbumAccessor::GetRootPath(NNT_CAPSRV_STORAGE_VALUES_SD), BadFolders[badFolderIdx]);
            nnt::capsrv::DirectAlbumAccessor::CreateParentDirectories(path);
        }
    }

    // ファイル数が作ったファイルの数になっていることを確認
    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE_I(s, storage)
    {
        int count = 0;
        nn::Result result = nn::capsrv::GetAlbumFileCount(&count, storage);
        EXPECT_TRUE(result.IsSuccess());
        EXPECT_EQ(FileCount, count);
    }

    // ファイルリスト取得関数のテスト。
    nn::capsrv::AlbumEntry albumEntries[FileCount];
    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE_I(s, storage)
    {
        int count = 0;
        nn::Result albumResult = nn::capsrv::GetAlbumFileList(&count, albumEntries, FileCount, storage);
        EXPECT_TRUE(albumResult.IsSuccess());
        EXPECT_EQ(FileCount, count);
    }

    // 作ったファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    nn::capsrv::FinalizeAlbumAccess();
    SUCCEED();
}

// NOTE:
// テストで 2GB メモリが確保できないのでサイズ上限ぎりぎりのファイルの I/O のテストができない

//TEST(AlbumAccessApi, GetAlbumFileList_Movie_FileSizeLimit)
//{
//    nnt::capsrv::StartupTestCase();
//    nn::capsrv::InitializeAlbumAccess();
//
//    // 既にあるファイルをクリア
//    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());
//
//    // 作成するファイルのリスト。
//    // 時刻はユニークにしておくこと。
//    const nnt::capsrv::FileInfo FileList[] = {
//        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 00, 0x0123456789ABCDEF, .jpg, nn::capsrv::AlbumFileSizeLimit_Movie    , 12345),
//        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 01, 0x0123456789ABCDEF, .jpg, nn::capsrv::AlbumFileSizeLimit_Movie + 1ll, 12345),
//    };
//    static const int TotalFileCount = sizeof(FileList) / sizeof(FileList[0]);
//
//    // ファイル作成
//    int createdFileCount = 0;
//    for(int i = 0; i < TotalFileCount; i++)
//    {
//        if(FileList[i].Create())
//        {
//            createdFileCount++;
//        }
//    }
//
//    // ファイルリストを取得
//    NN_LOG("Check file size limitation for GetAlbumFileList()\n");
//    {
//        nn::capsrv::AlbumEntry albumEntryList[TotalFileCount];
//        int count = 0;
//        EXPECT_TRUE(nn::capsrv::GetAlbumFileList(&count, albumEntryList, TotalFileCount, nn::capsrv::AlbumStorage_Nand).IsSuccess());
//
//        EXPECT_EQ(1, count);
//        EXPECT_TRUE(FileList[0].IsMatchTime(albumEntryList[0]));
//    }
//
//    NN_LOG("Check file size limitation for GetAlbumFileSize()\n");
//    {
//        {
//            auto& fileInfo = FileList[0];
//            nn::capsrv::AlbumFileId fileId = {};
//            fileId.applicationId = fileInfo.appId;
//            fileId.contents = nn::capsrv::AlbumFileContents_Movie;
//            fileId.storage = fileInfo.storage;
//            fileId.time = fileInfo.time;
//
//            size_t size;
//            EXPECT_TRUE(nn::capsrv::GetAlbumFileSize(&size, &fileId).IsSuccess());
//            EXPECT_EQ(nn::capsrv::AlbumFileSizeLimit_ScreenShot, size);
//        }
//        {
//            auto& fileInfo = FileList[1];
//            nn::capsrv::AlbumFileId fileId = {};
//            fileId.applicationId = fileInfo.appId;
//            fileId.contents = nn::capsrv::AlbumFileContents_Movie;
//            fileId.storage = fileInfo.storage;
//            fileId.time = fileInfo.time;
//
//            size_t size;
//            auto result = nn::capsrv::GetAlbumFileSize(&size, &fileId);
//            EXPECT_TRUE(nn::capsrv::ResultAlbumFileNotFound::Includes(result));
//        }
//    }
//
//
//    // 作ったファイルをクリア
//    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());
//
//    nn::capsrv::FinalizeAlbumAccess();
//    SUCCEED();
//}// NOLINT(impl/function_size)
