﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>

#include <nn/fs.h>
#include <nn/util/util_StringUtil.h>

#include <nnt.h>

#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nn/capsrv/capsrv_AlbumTesting.h>
#include "../../Common/testCapsrv_Macro.h"
#include "../../Common/testCapsrv_DirectAlbumAccessor.h"
#include "../../Common/testCapsrv_FileInfo.h"
#include "../../Common/testCapsrv_AlbumEntryUtility.h"
#include "testCapsrv_StartupTestCase.h"

namespace {
    // 作成するファイルのリスト。
    // 時刻はユニークにしておくこと。
    const nnt::capsrv::FileInfo FileList[] = {
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 00, 0x0123456789ABCDEF, .jpg, 1024, 12345),
        NNT_CAPSRV_FILEINFO(NA, 2016, 04, 02, 20, 50, 59, 01, 0xAAAAAAAAAAAAAAAA, .jpg, 1024, 12346),
        NNT_CAPSRV_FILEINFO(NA, 2016, 05, 03, 18, 31, 58, 02, 0xBBBBBBBBBBBBBBBB, .jpg, 1024, 12347),
        NNT_CAPSRV_FILEINFO(SD, 2016, 07, 04, 15, 20, 57, 03, 0xCCCCCCCCCCCCCCCC, .jpg, 1024, 12348),
        NNT_CAPSRV_FILEINFO(SD, 2016, 08, 06, 14, 10, 56, 04, 0xDDDDDDDDDDDDDDDD, .jpg, 1024, 12349),
    };
    static const int TotalFileCount = sizeof(FileList) / sizeof(FileList[0]);

    const nn::capsrv::AlbumStorageType StorageList[] = { nn::capsrv::AlbumStorage_Nand, nn::capsrv::AlbumStorage_Sd };
    static const int StorageCount = sizeof(StorageList) / sizeof(StorageList[0]);

    void CheckDeleteAlbumFile(int targetStorage, int targetFileIndex)
    {
        // 既にあるファイルをクリア
        EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());


        // ファイル作成
        for(int i = 0; i < TotalFileCount; i++)
        {
            FileList[i].Create();
        }

        // ファイルリスト取得
        std::vector<nn::capsrv::AlbumEntry> albumEntriyLists[StorageCount];
        for(int s = 0; s < StorageCount; s++)
        {
            auto& list = albumEntriyLists[s];
            list = nnt::capsrv::GetAlbumEntryList(StorageList[s]);
            nnt::capsrv::SortAlbumEntryListStorageTime(list);
        }

        if(albumEntriyLists[targetStorage].size() <= static_cast<size_t>(targetFileIndex))
        {
            NN_LOG("Check deleting file skipped: requested index %d, exists %d files.\n", targetFileIndex, albumEntriyLists[targetStorage].size());
            return;
        }

        // 適当に一つ削除
        NN_LOG("Check deleting file: storage %d, index %d\n", targetStorage, targetFileIndex);
        if(nn::capsrv::IsAlbumMounted(StorageList[targetStorage]))
        {
            nn::Result deleteResult = nn::capsrv::DeleteAlbumFile(&albumEntriyLists[targetStorage][targetFileIndex].fileId);
            EXPECT_TRUE(deleteResult.IsSuccess());

            // 削除後のファイルリスト取得
            std::vector<nn::capsrv::AlbumEntry> albumEntriyLists2[StorageCount];
            for(int s = 0; s < StorageCount; s++)
            {
                auto& list = albumEntriyLists2[s];
                list = nnt::capsrv::GetAlbumEntryList(StorageList[s]);
                nnt::capsrv::SortAlbumEntryListStorageTime(list);
            }

            // チェック
            {
                std::vector<nn::capsrv::AlbumEntry> entryListPre;
                std::vector<nn::capsrv::AlbumEntry> entryListPost;
                for(int s = 0; s < StorageCount; s++)
                {
                    entryListPre.insert(entryListPre.end(), albumEntriyLists[s].begin(), albumEntriyLists[s].end());
                    entryListPost.insert(entryListPost.end(), albumEntriyLists2[s].begin(), albumEntriyLists2[s].end());
                }

                // 1 つ減っているはず
                EXPECT_EQ(entryListPre.size() - 1, entryListPost.size());

                // 狙ったものが消えていることを確認
                nnt::capsrv::AlbumEntryListDifference difference;
                nnt::capsrv::SubtractAlbumEntryList(&difference, entryListPre, entryListPost);
                EXPECT_EQ(difference.positive.size(), 1);
                EXPECT_EQ(difference.negative.size(), 0);
                if(difference.positive.size() > 0)
                {
                    EXPECT_EQ(nnt::capsrv::CompareTime(albumEntriyLists[targetStorage][targetFileIndex].fileId.time, difference.positive[0].fileId.time), 0);
                }
            }
        }
        else
        {
            NN_LOG("storage %d is not mounted\n", targetStorage);
        }
    }

}// anonymous namespace


TEST(AlbumAccessApi, DeleteAlbumFile)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();

    for(int s = 0; s < StorageCount; s++)
    {
        for(int i = 0; i < TotalFileCount; i++)
        {
            CheckDeleteAlbumFile(s, i);
        }
    }

    // 作ったファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    nn::capsrv::FinalizeAlbumAccess();
    SUCCEED();
}

// 大きすぎるファイルサイズのテスト
TEST(AlbumAccessApi, DeleteAlbumFile_FileSizeLimit)
{
    nnt::capsrv::StartupTestCase();
    nn::capsrv::InitializeAlbumAccess();

    // 既にあるファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    // 作成するファイルのリスト。
    // 時刻はユニークにしておくこと。
    const nnt::capsrv::FileInfo TestFileList[] = {
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 00, 0x0123456789ABCDEF, .jpg, nn::capsrv::AlbumFileSizeLimit_ScreenShot    , 12345),
        NNT_CAPSRV_FILEINFO(NA, 2016, 01, 01, 00, 00, 00, 01, 0x0123456789ABCDEF, .jpg, nn::capsrv::AlbumFileSizeLimit_ScreenShot + 1, 12345),
    };
    static const int TestFileCount = sizeof(TestFileList) / sizeof(TestFileList[0]);

    // ファイル作成
    int createdFileCount = 0;
    for(int i = 0; i < TestFileCount; i++)
    {
        if(TestFileList[i].Create())
        {
            createdFileCount++;
        }
    }

    NN_LOG("Check file size limitation for DeleteAlbumFile()\n");
    {
        {
            auto& fileInfo = TestFileList[0];
            nn::capsrv::AlbumFileId fileId = {};
            fileId.applicationId = fileInfo.appId;
            fileId.contents = nn::capsrv::AlbumFileContents_ScreenShot;
            fileId.storage = fileInfo.storage;
            fileId.time = fileInfo.time;

            auto result = nn::capsrv::DeleteAlbumFile(&fileId);
            EXPECT_TRUE(result.IsSuccess());
        }
        {
            auto& fileInfo = TestFileList[1];
            nn::capsrv::AlbumFileId fileId = {};
            fileId.applicationId = fileInfo.appId;
            fileId.contents = nn::capsrv::AlbumFileContents_ScreenShot;
            fileId.storage = fileInfo.storage;
            fileId.time = fileInfo.time;

            auto result = nn::capsrv::DeleteAlbumFile(&fileId);
#if 1
            // SIGLO-79412
            //  アルバムファイル削除時のサイズチェックを無効にしたため、
            //  6.0NUP 以降では、こういうファイルも削除に成功する。
            EXPECT_TRUE(result.IsSuccess());
#else
            EXPECT_TRUE(nn::capsrv::ResultAlbumFileNotFound::Includes(result));
#endif
        }
    }


    // 作ったファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    nn::capsrv::FinalizeAlbumAccess();
    SUCCEED();
}// NOLINT(impl/function_size)


// アルバムの使用制限のテスト
TEST(AlbumAccessApi, DeleteAlbumFile_AlbumLimitation)
{
    nnt::capsrv::StartupTestCase();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());

    // 既にあるファイルをクリア
    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    // ファイル作成
    for(int i = 0; i < TotalFileCount; i++)
    {
        FileList[i].Create();
    }

    // キャッシュ更新
    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE(storage)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::RefreshAlbumCache(storage));
    }

    NNT_CAPSRV_FOREACH_MOUNTED_STORAGE(storage)
    {
        // 削除前のキャッシュを取得
        nn::capsrv::AlbumCacheData cacheData0 = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetAlbumCache(&cacheData0, storage, nn::capsrv::AlbumFileContents_ScreenShot));

        // 1 ファイル削除
        int entryCount = 0;
        nn::capsrv::AlbumEntry entryList[TotalFileCount];
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetAlbumFileList(&entryCount, entryList, TotalFileCount, storage));

        if(entryCount == 0)
        {
            NN_LOG("No file exists on storage %d\n", static_cast<int>(storage));
            continue;
        }

        auto& entry = entryList[0];
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::DeleteAlbumFile(&entry.fileId));

        // キャッシュが正しく更新されていることを確認
        nn::capsrv::AlbumCacheData cacheData1 = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetAlbumCache(&cacheData1, storage, nn::capsrv::AlbumFileContents_ScreenShot));

        EXPECT_EQ(cacheData0.fileCount - 1, cacheData1.fileCount);
    }

    EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

    nn::capsrv::FinalizeAlbumAccess();
    SUCCEED();
}// NOLINT(impl/function_size)
