﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <vector>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt.h>

#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nn/capsrv/capsrv_AlbumControl.h>
#include <nn/capsrv/capsrv_AlbumTesting.h>
#include <nn/capsrv/movie/capsrv_MovieMetaData.h>
#include <nn/capsrv/movie/capsrv_MovieMetaDataBuilder.h>
#include "../../Common/testCapsrv_Macro.h"
#include "../../Common/testCapsrv_DirectAlbumAccessor.h"
#include "../../Common/testCapsrv_FileInfo.h"
#include "../../Common/testCapsrv_AlbumEntryUtility.h"
#include "../../Common/testCapsrv_MovieFileCreator.h"
#include "testCapsrv_StartupTestCase.h"

namespace {
    enum SourceImageFormat
    {
        SourceImageFormat_Rgba,
        SourceImageFormat_Nv12,
    };

    void TestCreateProtoMovieMetaDataImpl(SourceImageFormat sourceImageFormat) NN_NOEXCEPT
    {
        nnt::capsrv::StartupTestCase();
        EXPECT_TRUE(nnt::capsrv::DirectAlbumAccessor::CleanupAllAlbums());

        NN_LOG("checking sourceImageFormat = %d\n", sourceImageFormat);

        const int Width = 1280;
        const int Height = 720;
        size_t movieDataSize = 32 * nn::capsrv::movie::MovieDataChunkUnitSize;

        void* imageRgba = nullptr;
        size_t sizeRgba = 0;
        void* imageNv12Y = nullptr;
        size_t sizeNv12Y = 0;
        void* imageNv12Uv = nullptr;
        size_t sizeNv12Uv = 0;
        if(sourceImageFormat == SourceImageFormat_Rgba)
        {
            sizeRgba = 4 * Width * Height;
            imageRgba = aligned_alloc(8, sizeRgba);
        }
        else if(sourceImageFormat == SourceImageFormat_Nv12)
        {
            const int Unit = 128 * 1024;
            sizeNv12Y = ((Width * Height + Unit - 1) / Unit) * Unit;
            sizeNv12Uv = ((Width * Height / 2 + Unit - 1) / Unit) * Unit;
            imageNv12Y = aligned_alloc(4096, sizeNv12Y);
            imageNv12Uv = aligned_alloc(4096, sizeNv12Uv);
        }
        else
        {
            NN_ABORT("unknown sourceImageFormat(%d)", sourceImageFormat);
        }
        NN_UTIL_SCOPE_EXIT
        {
            if(imageRgba)
            {
                free(imageRgba);
                imageRgba = nullptr;
            }
            if(imageNv12Y)
            {
                free(imageNv12Y);
                imageNv12Y = nullptr;
            }
            if(imageNv12Uv)
            {
                free(imageNv12Uv);
                imageNv12Uv = nullptr;
            }
        };

        std::mt19937 rand(0xBCF98765);
        nnt::capsrv::MovieData movieData = nnt::capsrv::MovieFileCreator::CreateRandomMovieData(movieDataSize, rand);
        nnt::capsrv::MovieMeta movieMeta = {};
        nnt::capsrv::MovieDataHash dataHash = nnt::capsrv::MovieFileCreator::CreateValidMovieDataHash(movieData);;
        nnt::capsrv::MovieMetaHash metaHash = {};
        nnt::capsrv::MovieSignature signature = {};
        nnt::capsrv::MovieVersion version = nnt::capsrv::MovieFileCreator::CreateValidMovieVersion(nn::capsrv::movie::MovieMetaDataVersion_1);

        struct Data
        {
            int frameCount;
            int numerator;
            int denominator;
            int duration;
            int keyInterval;
            bool isExplicitAppletDataUsed;
        };

        std::vector<Data> DataList;
        DataList.push_back({1800, 1, 60, 30, 30, false});
        DataList.push_back({1800, 1, 60, 30, 30, true});
        DataList.push_back({ 900, 1, 30, 30, 30, true});
        DataList.push_back({ 900, 2, 60, 30, 30, true});
        DataList.push_back({ 900, 1, 60, 15, 30, true});
        DataList.push_back({1800, 1, 60, 30, 60, true});

        for(auto& d : DataList)
        {
            NN_LOG("variation: c%dframes, r%d/%d, dur%dms, key%dframes\n", d.frameCount, d.numerator, d.denominator, d.duration, d.keyInterval);
            nn::capsrv::AlbumFileId fileId = {};
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
                NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };
                NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::GenerateCurrentAlbumFileId(&fileId, {0xAB123451234512}, nn::capsrv::AlbumFileContents_Movie));
            }

            std::uniform_int_distribution<int> dist;
            nn::capsrv::AppletData srcAppletData = {};
            if(d.isExplicitAppletDataUsed)
            {
                for(size_t i = 0; i < sizeof(srcAppletData); i++)
                {
                    srcAppletData.value[i] = static_cast<char>(dist(rand));
                }
            }

            // メタを作成
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeScreenShotControl());
                NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeScreenShotControl(); };
                movieMeta.value.resize(nn::capsrv::movie::MovieMetaDataSize);
                nn::capsrv::movie::MovieMetaDataBuilder builder = {};
                builder.Initialize(nn::capsrv::movie::MovieMetaDataVersion_1, movieMeta.value.data(), movieMeta.value.size(), fileId);
                builder.SetDescription(nn::capsrv::AlbumFileDescription_MovieContinuous);
                switch(sourceImageFormat)
                {
                case SourceImageFormat_Rgba:
                    builder.SetImageDataRgba(imageRgba, sizeRgba, Width, Height);
                    break;
                case SourceImageFormat_Nv12:
                    builder.SetImageDataNv12(imageNv12Y, sizeNv12Y, imageNv12Uv, sizeNv12Uv, Width, Height);
                    break;
                default: NN_UNEXPECTED_DEFAULT;
                }
                builder.SetMovieDataSize(movieDataSize);
                builder.SetAttribute(d.frameCount, d.numerator, d.denominator, d.duration, d.keyInterval, false, nn::capsrv::ScreenShotSize_1280x720, nn::capsrv::ScreenShotOrientation_Default);
                if(d.isExplicitAppletDataUsed)
                {
                    builder.SetAppletData(srcAppletData);
                }
                NNT_EXPECT_RESULT_SUCCESS(builder.Build());
                movieMeta.makerNoteVersion = builder.GetMakerNoteVersion();
                movieMeta.makerNoteOffset = builder.GetMakerNoteOffset();
                movieMeta.makerNoteSize = builder.GetMakerNoteSize();
                movieMeta.pMeta = reinterpret_cast<nn::capsrv::movie::MovieMetaData*>(movieMeta.value.data());
            }

            // 直接書き出すために静止画署名をつける
            nnt::capsrv::MovieFileCreator::SetValidMetaSignature(movieMeta);

            // ファイル作成
            metaHash = nnt::capsrv::MovieFileCreator::CreateValidMovieMetaHash(movieMeta);
            signature = nnt::capsrv::MovieFileCreator::CreateValidMovieSignature(dataHash, metaHash);
            auto path = nnt::capsrv::DirectAlbumAccessor::GetFilePath(fileId);
            nnt::capsrv::DirectAlbumAccessor::CreateFile(0, path.c_str());
            nn::fs::FileHandle hFile = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&hFile, path.c_str(), nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
            NN_ABORT_UNLESS_RESULT_SUCCESS(nnt::capsrv::MovieFileCreator::WriteFileData(
                [&](int64_t offset, const void* buf, size_t size)->nn::Result
                {
                    return nn::fs::WriteFile(hFile, offset, buf, size, nn::fs::WriteOption::MakeValue(0));
                },
                movieData,
                movieMeta,
                dataHash,
                metaHash,
                signature,
                version
            ));

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::FlushFile(hFile));
            nn::fs::CloseFile(hFile);

            // 読込
            NN_LOG("  checking loading image section\n");
            nn::capsrv::ScreenShotAttribute imageAttribute = {};
            nn::capsrv::AppletData imageAppletData = {};
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
                NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
                int w = 0;
                int h = 0;
                std::vector<uint8_t> imageData;
                imageData.resize(4 * Width * Height);
                nn::capsrv::ScreenShotDecodeOption opt = {};
                std::vector<uint8_t> work;
                work.resize(nn::capsrv::AlbumFileSizeLimit_ScreenShot);
                NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::LoadAlbumScreenShotImage(&w, &h, &imageAttribute, &imageAppletData, imageData.data(), imageData.size(), &fileId, opt, work.data(), work.size()));
            }

            NN_LOG("    dst:");
            for(int i = 0; i < 8; i++)
            {
                NN_LOG("%02X", imageAppletData.value[i]);
            }
            NN_LOG("\n");
            NN_LOG("    src:");
            for(int i = 0; i < 8; i++)
            {
                NN_LOG("%02X", srcAppletData.value[i]);
            }
            NN_LOG("\n");

            EXPECT_EQ(0, std::memcmp(&srcAppletData, &imageAppletData, sizeof(nn::capsrv::AppletData)));
            EXPECT_EQ(nn::capsrv::AlbumFileDescription_MovieContinuous, imageAttribute.description);
            EXPECT_EQ(d.frameCount, imageAttribute.frameCount);
            EXPECT_EQ(d.numerator, imageAttribute.frameRateNumerator);
            EXPECT_EQ(d.denominator, imageAttribute.frameRateDenominator);
            EXPECT_EQ(d.duration, imageAttribute.dataDurationMilliseconds);
            EXPECT_EQ(d.keyInterval, imageAttribute.keyFrameInterval);

            // ストリームからの属性読込
            NN_LOG("  checking reading attribute from stream(caps:a)\n");
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccess());
                NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumAccess(); };
                nn::capsrv::AlbumMovieReadStreamHandle hStream = {};
                NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieReadStream(&hStream, fileId));
                NN_UTIL_SCOPE_EXIT{ nn::capsrv::CloseAlbumMovieReadStream(hStream); };

                nn::capsrv::ScreenShotAttribute streamAttribute = {};
                NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::ReadAttributeFromAlbumMovieReadStream(&streamAttribute, hStream));

                EXPECT_EQ(0, std::memcmp(&imageAttribute, &streamAttribute, sizeof(nn::capsrv::ScreenShotAttribute)));
            }

            NN_LOG("  checking reading attribute from stream(caps:c)\n");
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControl());
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumControlExtension());
                NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeAlbumControl(); };

                nn::capsrv::AlbumMovieReadStreamHandle hStream = {};
                NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::OpenAlbumMovieReadStream(&hStream, fileId));
                NN_UTIL_SCOPE_EXIT{ nn::capsrv::CloseAlbumMovieReadStream(hStream); };

                nn::capsrv::ScreenShotAttribute streamAttribute = {};
                NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::ReadAttributeFromAlbumMovieReadStream(&streamAttribute, hStream));

                EXPECT_EQ(0, std::memcmp(&imageAttribute, &streamAttribute, sizeof(nn::capsrv::ScreenShotAttribute)));
            }
        }
    }// NOLINT(impl/function_size)
}


TEST(ScreenShotControlApi, CreateProtoMovieMetaDataNv12)
{
    TestCreateProtoMovieMetaDataImpl(SourceImageFormat_Nv12);
}

TEST(ScreenShotControlApi, CreateProtoMovieMetaDataRgba)
{
    TestCreateProtoMovieMetaDataImpl(SourceImageFormat_Rgba);
}
