﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include "testCapsrv_TQueue.h"
#include "testCapsrv_StaticRange.h"

namespace nnt{ namespace capsrv{ namespace threading{

    namespace detail {

        enum WorkerRequestTag : uint32_t
        {
            WorkerRequestTag_Exit = 0,
            WorkerRequestTag_InvokeFunction,
        };

        enum WorkerReplyTag : uint32_t
        {
            WorkerReplyTag_Complete = 0,
        };

        struct WorkerRequest
        {
            WorkerRequestTag m_Tag;
            void (*m_pFunction)(int, void*);
            void* m_pArgument;
        };

        struct WorkerReply
        {
            WorkerReplyTag m_Tag;
        };

        struct WorkerThreadContext
        {
            int m_ThreadNumber;
            TQueue<WorkerRequest, 2, true> m_RequestQueue;
            TQueue<WorkerReply, 2, true> m_ReplyQueue;
        };

        template<typename F>
        void InvokeFuncionObject(int tid, void* p) NN_NOEXCEPT
        {
            auto pFunc = reinterpret_cast<F*>(p);
            (*pFunc)(tid);
        }

    }

    class WorkerThreadPool
    {
    public:
        static const int CoreNumberMin = 0;
        static const int CoreNumberMax = 2;

        static const int ThreadCountMax = 12;

    public:
        typedef uint64_t SequenceId;

    public:
        WorkerThreadPool() NN_NOEXCEPT;

        void Initialize(int threadCount, size_t stackSize, int workerThreadPriority) NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        void SetActiveThreadCount(int value) NN_NOEXCEPT;
        int GetActiveThreadCount() const NN_NOEXCEPT;

        StaticRange<ThreadCountMax> GetStaticRangeDivision(int n) const NN_NOEXCEPT;

        void StartWorker() NN_NOEXCEPT;
        void StopWorker() NN_NOEXCEPT;

        SequenceId InvokeFunctionAsync(void (*pFunction)(int threadId, void* arg), void* arg) NN_NOEXCEPT;

        void WaitComplete(SequenceId id) NN_NOEXCEPT;

        void InvokeFunction(void (*pFunction)(int threadId, void* arg), void* arg) NN_NOEXCEPT;
        template<typename F>
        void InvokeFunction(F func) NN_NOEXCEPT
        {
            InvokeFunction(detail::InvokeFuncionObject<F>, &func);
        }

    private:
        static void WorkerThreadFunction(void* p) NN_NOEXCEPT;

    private:
        int m_ThreadCount;
        int m_ActiveThreadCount;
        void* m_pStackMemory;
        SequenceId m_RequestedSequenceCount;

        nn::os::ThreadType m_ThreadList[ThreadCountMax];
        detail::WorkerThreadContext m_ContextList[ThreadCountMax];
    };

}}}
