﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>

namespace nnt{ namespace capsrv{ namespace threading{

    template<typename T, int TCapacity, bool TThreadSafe>
    class TQueue;

    template<typename T, int TCapacity>
    class TQueue<T, TCapacity, false>
    {
    public:
        static const int Capacity = TCapacity;

    protected:
        static const int InternalSize = Capacity + 1;

    public:
        TQueue() NN_NOEXCEPT
            : m_EntryList()
            , m_HeadIndex(0)
            , m_TailIndex(0)
        {
        }

        void Initialize() NN_NOEXCEPT
        {
            m_HeadIndex = 0;
            m_TailIndex = 0;
        }

        void Finalize() NN_NOEXCEPT
        {
        }

        bool IsFull() const NN_NOEXCEPT
        {
            return ((m_TailIndex + 1) % InternalSize) == m_HeadIndex;
        }

        bool IsEmpty() const NN_NOEXCEPT
        {
            return m_TailIndex == m_HeadIndex;
        }

        int GetCount() const NN_NOEXCEPT
        {
            int tail = m_TailIndex;
            if(m_TailIndex < m_HeadIndex)
            {
                tail += InternalSize;
            }
            NN_ASSERT_GREATER_EQUAL(tail, m_HeadIndex);
            return tail - m_HeadIndex;
        }

        void Clear() NN_NOEXCEPT
        {
            m_HeadIndex = 0;
            m_TailIndex = 0;
        }

        bool TryEnqueue(T entry) NN_NOEXCEPT
        {
            int nextTail = (m_TailIndex + 1) % InternalSize;
            if(nextTail == m_HeadIndex)
            {
                return false;
            }

            m_EntryList[m_TailIndex] = entry;
            m_TailIndex = nextTail;
            return true;
        }

        bool TryDequeue(T* pOutEntry) NN_NOEXCEPT
        {
            if(m_HeadIndex == m_TailIndex)
            {
                return false;
            }

            *pOutEntry = m_EntryList[m_HeadIndex];
            m_EntryList[m_HeadIndex] = T();
            m_HeadIndex = (m_HeadIndex + 1) % InternalSize;
            return true;
        }

        //// 次にデキューされる要素のポインタを取得する。この関数ではデキューされない。
        //// キューが空の場合 nullptr が返る。
        //T* GetHead() NN_NOEXCEPT
        //{
        //    if(IsEmpty())
        //    {
        //        return nullptr;
        //    }
        //    return &m_EntryList[m_HeadIndex];
        //}

        // 次にデキューされる要素のポインタを取得する。この関数ではデキューされない。
        // キューが空の場合 nullptr が返る。
        const T* GetHead() const NN_NOEXCEPT
        {
            if(IsEmpty())
            {
                return nullptr;
            }
            return &m_EntryList[m_HeadIndex];
        }

        //// 最後にエンキューされた要素のポインタを取得する。
        //// キューが空の場合 nullptr が返る。
        //T* GetTail() NN_NOEXCEPT
        //{
        //    if(IsEmpty())
        //    {
        //        return nullptr;
        //    }

        //    int tail = m_TailIndex - 1;
        //    if(tail < 0)
        //    {
        //        tail = InternalSize - 1;
        //    }
        //    return &m_EntryList[tail];
        //}

        // 最後にエンキューされた要素のポインタを取得する。
        // キューが空の場合 nullptr が返る。
        const T* GetTail() const NN_NOEXCEPT
        {
            if(IsEmpty())
            {
                return nullptr;
            }

            int tail = m_TailIndex - 1;
            if(tail < 0)
            {
                tail = InternalSize - 1;
            }
            return &m_EntryList[tail];
        }

    private:
        T m_EntryList[InternalSize];
        int m_HeadIndex;
        int m_TailIndex;
    };

    template<typename T, int TCapacity>
    class TQueue<T, TCapacity, true>
        : private TQueue<T, TCapacity, false>
    {
    private:
        typedef TQueue<T, TCapacity, false> BaseType;

    public:
        static const int Capacity = BaseType::Capacity;
        static const int InternalSize = BaseType::InternalSize;

    public:
        TQueue() NN_NOEXCEPT
            : BaseType()
            , m_Mutex()
            , m_EnqueueSemaphore()
            , m_DequeueSemaphore()
        {
        }

        void Initialize() NN_NOEXCEPT
        {
            BaseType::Initialize();
            nn::os::InitializeMutex(&m_Mutex, false, 0);
            nn::os::InitializeSemaphore(&m_EnqueueSemaphore, Capacity, Capacity);
            nn::os::InitializeSemaphore(&m_DequeueSemaphore,        0, Capacity);
        }

        void Finalize() NN_NOEXCEPT
        {
            nn::os::FinalizeSemaphore(&m_DequeueSemaphore);
            nn::os::FinalizeSemaphore(&m_EnqueueSemaphore);
            nn::os::FinalizeMutex(&m_Mutex);
            BaseType::Finalize();
        }

        // キューが空くまでブロックします。
        void Enqueue(T entry) NN_NOEXCEPT
        {
            nn::os::AcquireSemaphore(&m_EnqueueSemaphore);
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

            auto result = BaseType::TryEnqueue(entry);
            NN_ASSERT(result);

            nn::os::ReleaseSemaphore(&m_DequeueSemaphore);
        }

        // エンキューされるまでブロックします。
        void Dequeue(T* pOutEntry) NN_NOEXCEPT
        {
            nn::os::AcquireSemaphore(&m_DequeueSemaphore);
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

            auto result = BaseType::TryDequeue(pOutEntry);
            NN_ASSERT(result);

            nn::os::ReleaseSemaphore(&m_EnqueueSemaphore);
        }

        // デキュー可能になった際にシグナルする MultiWaitHolder を初期化します。
        void InitializeDequeueableMultiWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
        {
            NN_ASSERT_NOT_NULL(pHolder);
            nn::os::InitializeMultiWaitHolder(pHolder, &m_DequeueSemaphore);
        }

        // デキュー可能になった際にシグナルする MultiWaitHolder を破棄します。
        void FinalizeDequeueableMultiWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
        {
            NN_ASSERT_NOT_NULL(pHolder);
            nn::os::FinalizeMultiWaitHolder(pHolder);
        }

    private:
        nn::os::MutexType m_Mutex;
        nn::os::SemaphoreType m_EnqueueSemaphore;
        nn::os::SemaphoreType m_DequeueSemaphore;
    };

}}}
