﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <random>
#include <nn/fs.h>
#include <nn/capsrv/capsrv_AlbumEntry.h>
#include "testCapsrv_TestFileUtility.h"

namespace nnt{ namespace capsrv{

    struct FileInfo
    {
    public:
        static const int ImageWidth = 1280;
        static const int ImageHeight = 720;
        static const int ImageBlockSize = 16;
    public:
        // @brief 設定されたデータに基づいてファイルを作成します。
        // @retval true  作成に成功
        // @retval false ストレージがマウントされていない
        // @details
        // ファイルの内容はランダムです。
        bool Create() const;

        // 設定されたデータに基づいて JPEG ファイルを作成します。
        // @retval true  作成に成功
        // @retval false ストレージがマウントされていない
        // @details
        //  ファイルの内容はランダムな画像の JPEG です。
        //  このオブジェクトの fileSize の値は無視されます。
        bool CreateJpeg(TestFileSignaturePattern signaturePattern = TestFileSignaturePattern_Valid) const;

        // 設定されたデータに基づいて動画ファイルを作成します。
        bool CreateMovie() const;

        // 渡されたデータがこのファイルの内容に一致するかをチェックします。
        bool CheckData(const void* targetData, int64_t dataSize) const;

        // 渡された JPEG データとこのファイルの内容の距離を取得します。
        // 距離はピクセル差分の平均値(0～255)です。
        double CheckDataJpeg(const std::vector<uint8_t>& jpeg) const;

        // 時刻が一致するかを返します。
        bool IsMatchTime(const nn::capsrv::AlbumEntry& entry) const;

        // ファイルパスを取得します。
        std::string GetDirectFilePath() const;

        // ファイル ID を取得します。
        nn::capsrv::AlbumFileId GetAlbumFileId() const;

        // Create() で生成するファイルの内容を取得します
        std::vector<uint8_t> GetData() const;

    public:
        int64_t filesize;
        nn::capsrv::AlbumStorageType storage;
        nn::capsrv::AlbumFileDateTime time;
        nn::ncm::ApplicationId appId;
        char extension[10];
        uint64_t seed;
    };


// 0 で始まっていても 10 進数として解釈させるためのマクロ。2桁用。
#define NNT_CAPSRV_DECIMAL2(val) ( 1 ## val - 100)
// 0 で始まっていても 10 進数として解釈させるためのマクロ。4桁用。
#define NNT_CAPSRV_DECIMAL4(val) ( 1 ## val - 10000)
// ストレージを取得するためのマクロ
#define NNT_CAPSRV_STORAGE_VALUES_NA nn::capsrv::AlbumStorage_Nand
#define NNT_CAPSRV_STORAGE_VALUES_SD nn::capsrv::AlbumStorage_Sd
#define NNT_CAPSRV_STORAGE_VALUE_I(a,b) a ## b
#define NNT_CAPSRV_STORAGE_VALUE(storage) NNT_CAPSRV_STORAGE_VALUE_I(NNT_CAPSRV_STORAGE_VALUES_, storage)

// FileInfo の初期化子をつくるためのマクロ。
//  storage … NA, SD のいずれか
//  year    … 4 桁の年
//  month, day, hour, min, sec, id … 2 桁の数値。1 桁の場合は 0 詰めすること。
//  appId   … uint64_t の値。
//  ext     … .jpg
//  filesize … 作成するファイルサイズ
//  seed    … ファイルの内容を生成するための乱数のシード
#define NNT_CAPSRV_FILEINFO(storage,year,month,day,hour,min,sec,id,appId,ext,filesize,seed)    \
    {\
        filesize, \
        NNT_CAPSRV_STORAGE_VALUE(storage), \
        { \
            NNT_CAPSRV_DECIMAL4(year), \
            NNT_CAPSRV_DECIMAL2(month), \
            NNT_CAPSRV_DECIMAL2(day), \
            NNT_CAPSRV_DECIMAL2(hour), \
            NNT_CAPSRV_DECIMAL2(min), \
            NNT_CAPSRV_DECIMAL2(sec), \
            NNT_CAPSRV_DECIMAL2(id), \
        }, \
        { appId }, \
        #ext, \
        seed   \
    }

}}
