﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include "testCapsrv_AlbumEntryUtility.h"

#include <algorithm>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nnt.h>

namespace nnt{ namespace capsrv{

#define NNT_CAPSRV_COMPARE_CHECK(a,b) if(a != b) { return static_cast<int>(a) - static_cast<int>(b); }
    int CompareTime(const nn::capsrv::AlbumFileDateTime& lhs, const nn::capsrv::AlbumFileDateTime& rhs)
    {
        NNT_CAPSRV_COMPARE_CHECK(lhs.year, rhs.year);
        NNT_CAPSRV_COMPARE_CHECK(lhs.month, rhs.month);
        NNT_CAPSRV_COMPARE_CHECK(lhs.day, rhs.day);
        NNT_CAPSRV_COMPARE_CHECK(lhs.hour, rhs.hour);
        NNT_CAPSRV_COMPARE_CHECK(lhs.minute, rhs.minute);
        NNT_CAPSRV_COMPARE_CHECK(lhs.second, rhs.second);
        NNT_CAPSRV_COMPARE_CHECK(lhs.id, rhs.id);
        return 0;
    }

    int CompareAlbumEntryStorageTime(const nn::capsrv::AlbumEntry& lhs, const nn::capsrv::AlbumEntry& rhs)
    {
        NNT_CAPSRV_COMPARE_CHECK(lhs.fileId.storage, rhs.fileId.storage);
        return CompareTime(lhs.fileId.time, rhs.fileId.time);
    }
#undef NNT_CAPSRV_COMPARE_CHECK

    std::vector<nn::capsrv::AlbumEntry> GetAlbumEntryList(nn::capsrv::AlbumStorageType storage)
    {
        if(!nn::capsrv::IsAlbumMounted(storage))
        {
            NN_LOG("GetAlbumEntryList skipped. storage %d is not mounted.\n", static_cast<int>(storage));
            return std::vector<nn::capsrv::AlbumEntry>();
        }

        std::vector<nn::capsrv::AlbumEntry> list;
        // ファイル数を取得
        int count = 0;
        nn::Result capsrvResult = nn::capsrv::GetAlbumFileCount(&count, storage);
        EXPECT_TRUE(capsrvResult.IsSuccess());
        if(count == 0)
        {
            return list;
        }
        list.resize(static_cast<size_t>(count));
        // ファイルリストを取得
        capsrvResult = nn::capsrv::GetAlbumFileList(&count, list.data(), count, storage);
        EXPECT_TRUE(capsrvResult.IsSuccess());
        if(count == 0)
        {
            list.clear();
        }
        else
        {
            list.resize(static_cast<size_t>(count));
        }
        return list;
    }


    void SortAlbumEntryListStorageTime(std::vector<nn::capsrv::AlbumEntry>& list)
    {
        std::sort(
            list.begin(),
            list.end(),
            [](const nn::capsrv::AlbumEntry& lhs, const nn::capsrv::AlbumEntry& rhs)
            {
                return CompareAlbumEntryStorageTime(lhs, rhs) < 0;
            }
        );
    }

    void SubtractAlbumEntryList(
        AlbumEntryListDifference* pOutValue,
        const std::vector<nn::capsrv::AlbumEntry>& lhs,
        const std::vector<nn::capsrv::AlbumEntry>& rhs
        )
    {
        std::vector<nn::capsrv::AlbumEntry>& outPositive = pOutValue->positive;
        std::vector<nn::capsrv::AlbumEntry>& outNegative = pOutValue->negative;
        outPositive.clear();
        outNegative.clear();

        auto lhsIt = lhs.begin();
        auto rhsIt = rhs.begin();
        while(lhsIt != lhs.end() && rhsIt != rhs.end())
        {
            int compareResult = CompareAlbumEntryStorageTime(*lhsIt, *rhsIt);
            if(compareResult < 0)
            {
                outPositive.push_back(*lhsIt);
                ++lhsIt;
            }
            else if(compareResult > 0)
            {
                outNegative.push_back(*rhsIt);
                ++rhsIt;
            }
            else
            {
                ++lhsIt;
                ++rhsIt;
            }
        }
        outPositive.insert(outPositive.end(), lhsIt, lhs.end());
        outNegative.insert(outNegative.end(), rhsIt, rhs.end());
    }


}}
