﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
// NintendoSDK のヘッダファイルをインクルードする前に、NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET マクロを
// 定義することで、DebugFontWriter::Print() の入力文字コードを Windows のロケールのデフォルト
// (日本語の場合、CP932)に変更できます。
// #define NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET

#include <cstdlib>
#include <cstdio>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/os.h>
#include <nn/oe.h>
#include <nn/init.h>

#include <nn/util/util_Color.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#include <nn/btm/btm.h>
#include <nn/btm/system/btm_SystemApi.h>

#include "../Common/BtmTools_GraphicsSystem.h"
#include "../Common/BtmTools_Pad.h"

namespace
{
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    nnt::btm::GraphicsSystem* g_pGraphicsSystem;

    nn::btm::DeviceInfoList g_List;
    int g_Cursor = 0;
}

char* BdaddrToString(const nn::btm::BdAddress &bdaddr)
{
    static char strBdaddr[18];
    snprintf(strBdaddr, sizeof(strBdaddr), "%02X:%02X:%02X:%02X:%02X:%02X",
        bdaddr.address[0], bdaddr.address[1], bdaddr.address[2], bdaddr.address[3], bdaddr.address[4], bdaddr.address[5]);
    return strBdaddr;
}

//------------------------------------------------------------------------------
//  キーイベント処理
//------------------------------------------------------------------------------
void KeyEventFunc(nn::hid::NpadButtonSet& state)
{
    if (state.Test<nn::hid::NpadButton::Down>() || state.Test<nn::hid::NpadButton::StickLDown>())
    {
        ++g_Cursor;
        if (g_Cursor > g_List.deviceCount - 1)
        {
            g_Cursor = 0;
        }
    }
    if (state.Test<nn::hid::NpadButton::Up>() || state.Test<nn::hid::NpadButton::StickLUp>())
    {
        --g_Cursor;
        if (g_Cursor < 0)
        {
            g_Cursor = g_List.deviceCount - 1;
        }
    }
}

void DrawInfo(nn::btm::DeviceInfo* pInfo, nn::gfx::util::DebugFontTextWriter* pWriter)
{
    pWriter->SetScale(1.3f, 1.3f);
    const float Left = pWriter->GetCursorX();
    const float Top = pWriter->GetCursorY();

    pWriter->SetCursor(Left, Top);
    pWriter->SetTextColor(nn::util::Color4u8::White());
    pWriter->Print("%s", pInfo->bdName.name);
    pWriter->Print("\t%s\n", BdaddrToString(pInfo->bdAddress));
    pWriter->SetCursorX(Left);
    for (auto val : pInfo->linkKey.key)
    {
        pWriter->Print("%02X ", val);
    }
}

void DrawDeviceList(nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::PrimitiveRenderer::Renderer* pRenderer, nn::gfx::util::DebugFontTextWriter* pWriter)
{
    pWriter->SetScale(1.3f, 1.3f);
    const float Left = 5;
    const float Top = 5;
    const float Margin = 7;
    float w = pWriter->GetFontWidth();
    float h = pWriter->GetFontHeight();

    float textPosX = Left + w;
    float textPosY = Top + Margin;

    pWriter->SetTextColor(nn::util::Color4u8::White());
    pWriter->SetCursor(textPosX, textPosY);

    if (g_List.deviceCount == 0)
    {
        nn::util::Color4u8Type color1 = { { 255, 128, 0, 255 } };
        pWriter->SetTextColor(color1);
        pWriter->Print("No devices paired");
    }

    for (int i = 0; i < g_List.deviceCount; ++i)
    {
        pWriter->Print("%d\t", i + 1);
        DrawInfo(&g_List.device[i], pWriter);
        textPosY = pWriter->GetCursorY() + h + Margin * 2;
        pWriter->SetCursor(textPosX, textPosY);
    }
}

void MakeCommandFunc(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData)
{
    auto& gfxsys = *reinterpret_cast<nnt::btm::GraphicsSystem*>(pUserData);

    auto* renderer = gfxsys.GetPrimitiveRenderer();
    auto* writer = gfxsys.GetDebugFontTextWriter();

    renderer->Update(bufferIndex);

    pGraphicsFramework->BeginFrame(bufferIndex);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = pGraphicsFramework->GetRootCommandBuffer(bufferIndex);

        nn::gfx::ColorTargetView* target = pGraphicsFramework->GetColorTargetView();
        rootCommandBuffer->ClearColor(target,
            nn::util::Color4u8::Black().GetR(),
            nn::util::Color4u8::Black().GetG(),
            nn::util::Color4u8::Black().GetB(),
            nn::util::Color4u8::Black().GetA(), 0);
        rootCommandBuffer->SetRenderTargets(1, &target, nullptr);
        rootCommandBuffer->SetViewportScissorState(pGraphicsFramework->GetViewportScissorState());

        renderer->SetDefaultParameters();

        nn::util::Matrix4x3fType viewMatrix;
        nn::util::Matrix4x4fType projectionMatrix;
        nn::util::Matrix4x3f modelMatrix;

        nn::util::MatrixIdentity(&viewMatrix);
        nn::util::MatrixIdentity(&projectionMatrix);
        nn::util::MatrixIdentity(&modelMatrix);
        renderer->SetViewMatrix(&viewMatrix);
        renderer->SetProjectionMatrix(&projectionMatrix);
        renderer->SetModelMatrix(&modelMatrix);

        DrawDeviceList(rootCommandBuffer, renderer, writer);

        writer->Draw(rootCommandBuffer);
    }
    pGraphicsFramework->EndFrame(bufferIndex, true);
} // NOLINT(impl/function_size)

bool ExitApplicationRequest()
{
    nn::oe::Message message;
    if (nn::oe::TryPopNotificationMessage(&message))
    {
        switch(message)
        {
        // フォーカス状態変更通知
        case nn::oe::MessageFocusStateChanged:
            {
                auto state = nn::oe::GetCurrentFocusState();
                //  HOME メニューから復帰した
                if (state == nn::oe::FocusState_InFocus)
                {
                    // ペアリング済みのデバイス情報を取得します
                    nn::btm::GetDeviceInfo(&g_List);
                }
                break;
            }
        case nn::oe::MessageExitRequest:
            {
                NN_LOG("Received MessageExitRequest\n");
                return true;
            }
        // 動作モード（携帯／据置）が変更
        case nn::oe::MessageOperationModeChanged:
            break;
        // 性能モード（ノーマル／ブースト）が変更
        case nn::oe::MessagePerformanceModeChanged:
            break;
        // 未定義メッセージ
        default:
            {
                NN_LOG("Received unknown message= 0x%08x", message);
            }
            break;
        }
    }
    return false;
}
//------------------------------------------------------------------------------
//  メイン 関数
//------------------------------------------------------------------------------
extern "C" void nnMain()
{
    NN_LOG( "Start demo.\n" );
    // FS を使用する場合はフレームワークよりも前に初期化します

    nn::oe::Initialize();
    nn::oe::SetFocusHandlingMode(nn::oe::FocusHandlingMode_Notify);
    nn::oe::EnterExitRequestHandlingSection();

    // フレームワーク初期化
    g_pGraphicsSystem = new nnt::btm::GraphicsSystem();
    g_pGraphicsSystem->Initialize(GraphicsSystemMemorySize, MakeCommandFunc);

    // デバッグフォント初期化
    g_pGraphicsSystem->InitializeDebugFont();

    // プリミティブレンダラ初期化
    g_pGraphicsSystem->InitializePrimitiveRenderer();

    nnt::btm::Pad pad;
    pad.Initialize();

    nn::btm::InitializeBtmInterface();

    // ペアリング済みのデバイス情報を取得します
    nn::btm::GetDeviceInfo(&g_List);

    for (int frame = 0; !ExitApplicationRequest(); ++frame)
    {
        nn::hid::NpadButtonSet buttonState = pad.Update();
        KeyEventFunc(buttonState);
        g_pGraphicsSystem->Update();
    }

    nn::btm::FinalizeBtmInterface();

    // デバッグフォント終了
    g_pGraphicsSystem->FinalizeDebugFont();

    // プリミティブレンダラ終了
    g_pGraphicsSystem->FinalizePrimitiveRenderer();

    // フレームワーク終了
    g_pGraphicsSystem->Finalize();
    delete g_pGraphicsSystem;

    NN_LOG("End demo.\n");
    nn::oe::LeaveExitRequestHandlingSection();
}
