﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>

#include "TestBtmModule_ApiBtmSystem.h"

#include <nn/btm/system/btm_SystemApi.h>
#include <nn/btm/btm_Types.h>
#include <nn/btm/system/btm_SystemResult.h>

#include <nn/btm/btm_Api.h>
#include <nn/btm/btm_Result.h>

namespace ApiBtmSystem
{
    void TestBtmModule_ApiBtmSystem::TestGamepadPairing()
    {
        nn::btm::DeviceConditionList deviceConditionList;

        NN_TEST_BTM_MODULE_LOG("%s: Start gamepad pairing\n", NN_CURRENT_FUNCTION_NAME);

        nn::os::ClearEvent(&m_GamepadPairingEvent);
        nn::btm::system::StartGamepadPairing();
        nn::os::WaitEvent(&m_GamepadPairingEvent);

        NN_ABORT_UNLESS(nn::btm::system::IsGamepadPairingStarted());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Cancel gamepad pairing\n", NN_CURRENT_FUNCTION_NAME);

        nn::os::ClearEvent(&m_GamepadPairingEvent);
        nn::btm::system::CancelGamepadPairing();
        nn::os::WaitEvent(&m_GamepadPairingEvent);

        NN_ABORT_UNLESS(!nn::btm::system::IsGamepadPairingStarted());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Start gamepad pairing multiple times\n", NN_CURRENT_FUNCTION_NAME);

        nn::os::ClearEvent(&m_GamepadPairingEvent);
        nn::btm::system::StartGamepadPairing();
        nn::os::WaitEvent(&m_GamepadPairingEvent);

        NN_ABORT_UNLESS(nn::btm::system::IsGamepadPairingStarted());

        nn::os::ClearEvent(&m_GamepadPairingEvent);
        nn::btm::system::StartGamepadPairing();
        NN_ABORT_UNLESS(!nn::os::TryWaitEvent(&m_GamepadPairingEvent));     //!< 2回目はイベントがシグナルされない

        NN_ABORT_UNLESS(nn::btm::system::IsGamepadPairingStarted());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Cancel gamepad pairing multiple times\n", NN_CURRENT_FUNCTION_NAME);

        nn::os::ClearEvent(&m_GamepadPairingEvent);
        nn::btm::system::CancelGamepadPairing();
        nn::os::WaitEvent(&m_GamepadPairingEvent);

        NN_ABORT_UNLESS(!nn::btm::system::IsGamepadPairingStarted());

        nn::os::ClearEvent(&m_GamepadPairingEvent);
        nn::btm::system::CancelGamepadPairing();
        NN_ABORT_UNLESS(!nn::os::TryWaitEvent(&m_GamepadPairingEvent));     //!< 2回目はイベントがシグナルされない

        NN_ABORT_UNLESS(!nn::btm::system::IsGamepadPairingStarted());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllControllers();
        UnpairAllControllers();
        WaitControllerConnection(1);
        DisconnectAllControllers();

        NN_TEST_BTM_MODULE_LOG("%s: Check paired device count\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_EQUAL(nn::btm::system::GetPairedGamepadCount(), 1);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Clear gamepad pairing database without connection\n", NN_CURRENT_FUNCTION_NAME);

        nn::os::ClearEvent(&m_RegisterDeviceInfoEvent);
        nn::btm::system::ClearGamepadPairingDatabase();
        nn::os::WaitEvent(&m_RegisterDeviceInfoEvent);

        NN_ABORT_UNLESS_EQUAL(nn::btm::system::GetPairedGamepadCount(), 0);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        WaitControllerConnection(1);

        NN_TEST_BTM_MODULE_LOG("%s: Clear gamepad pairing database with connection\n", NN_CURRENT_FUNCTION_NAME);

        nn::os::ClearEvent(&m_RegisterDeviceInfoEvent);
        nn::btm::system::ClearGamepadPairingDatabase();
        nn::os::WaitEvent(&m_RegisterDeviceInfoEvent);

        NN_ABORT_UNLESS_EQUAL(nn::btm::system::GetPairedGamepadCount(), 0);

        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, 0);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllControllers();
        UnpairAllControllers();
    }

    void TestBtmModule_ApiBtmSystem::TestRadio()
    {
        nn::btm::DeviceConditionList deviceConditionList;

        NN_TEST_BTM_MODULE_LOG("%s: Disable radio without connection\n", NN_CURRENT_FUNCTION_NAME);

        nn::os::ClearEvent(&m_RadioEvent);
        nn::btm::system::EnableRadio(false);
        nn::os::WaitEvent(&m_RadioEvent);

        NN_ABORT_UNLESS(!nn::btm::system::IsRadioEnabled());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Enable radio\n", NN_CURRENT_FUNCTION_NAME);

        nn::os::ClearEvent(&m_RadioEvent);
        nn::btm::system::EnableRadio(true);
        nn::os::WaitEvent(&m_RadioEvent);

        NN_ABORT_UNLESS(nn::btm::system::IsRadioEnabled());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Disable radio for multiple times\n", NN_CURRENT_FUNCTION_NAME);

        nn::os::ClearEvent(&m_RadioEvent);
        nn::btm::system::EnableRadio(false);
        nn::os::WaitEvent(&m_RadioEvent);

        NN_ABORT_UNLESS(!nn::btm::system::IsRadioEnabled());

        nn::os::ClearEvent(&m_RadioEvent);
        nn::btm::system::EnableRadio(false);
        NN_ABORT_UNLESS(!nn::os::TryWaitEvent(&m_RadioEvent));      //!< 2回目はイベントがシグナルされない

        NN_ABORT_UNLESS(!nn::btm::system::IsRadioEnabled());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Enable radio for multiple times\n", NN_CURRENT_FUNCTION_NAME);

        nn::os::ClearEvent(&m_RadioEvent);
        nn::btm::system::EnableRadio(true);
        nn::os::WaitEvent(&m_RadioEvent);

        NN_ABORT_UNLESS(nn::btm::system::IsRadioEnabled());

        nn::os::ClearEvent(&m_RadioEvent);
        nn::btm::system::EnableRadio(true);
        NN_ABORT_UNLESS(!nn::os::TryWaitEvent(&m_RadioEvent));      //!< 2回目はイベントがシグナルされない

        NN_ABORT_UNLESS(nn::btm::system::IsRadioEnabled());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        WaitControllerConnection(1);

        NN_TEST_BTM_MODULE_LOG("%s: Disable radio with connection\n", NN_CURRENT_FUNCTION_NAME);

        nn::os::ClearEvent(&m_RadioEvent);
        nn::btm::system::EnableRadio(false);
        nn::os::WaitEvent(&m_RadioEvent);

        NN_ABORT_UNLESS(!nn::btm::system::IsRadioEnabled());

        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, 0);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        nn::os::ClearEvent(&m_RadioEvent);
        nn::btm::system::EnableRadio(true);
        nn::os::WaitEvent(&m_RadioEvent);

        NN_ABORT_UNLESS(nn::btm::system::IsRadioEnabled());

        DisconnectAllControllers();
        UnpairAllControllers();
    }

    void TestBtmModule_ApiBtmSystem::TestGamepadPairingRadioOff()
    {
        if (nn::btm::system::IsRadioEnabled())
        {
            nn::os::ClearEvent(&m_RadioEvent);
            nn::btm::EnableRadio(false);
            nn::os::WaitEvent(&m_RadioEvent);

            NN_ABORT_UNLESS(!nn::btm::system::IsRadioEnabled());
        }

        NN_TEST_BTM_MODULE_LOG("%s: Start gamepad pairing\n", NN_CURRENT_FUNCTION_NAME);

        nn::os::ClearEvent(&m_GamepadPairingEvent);

        NN_ABORT_UNLESS(nn::btm::system::ResultPairingFailureRadioOff().Includes(nn::btm::system::StartGamepadPairing()));
        NN_ABORT_UNLESS(!nn::btm::system::IsGamepadPairingStarted());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Cancel gamepad pairing\n", NN_CURRENT_FUNCTION_NAME);

        nn::os::ClearEvent(&m_GamepadPairingEvent);
        nn::btm::system::CancelGamepadPairing();
        nn::os::WaitEvent(&m_GamepadPairingEvent);      // Radio OFF 中もイベントはシグナルされる

        NN_ABORT_UNLESS(!nn::btm::system::IsGamepadPairingStarted());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        UnpairAllControllers();

        WaitControllerConnection(1);
        DisconnectAllControllers();

        // Radio OFF 中でも動く
        NN_TEST_BTM_MODULE_LOG("%s: Check paired device count\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_EQUAL(nn::btm::system::GetPairedGamepadCount(), 1);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Radio OFF 中でも動く
        NN_TEST_BTM_MODULE_LOG("%s: Clear gamepad pairing database without connection\n", NN_CURRENT_FUNCTION_NAME);

        nn::os::ClearEvent(&m_RegisterDeviceInfoEvent);
        nn::btm::system::ClearGamepadPairingDatabase();
        nn::os::WaitEvent(&m_RegisterDeviceInfoEvent);

        NN_ABORT_UNLESS_EQUAL(nn::btm::system::GetPairedGamepadCount(), 0);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllControllers();
        UnpairAllControllers();
    }
}   // namespace ApiBtmSystem

