﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>

#include <nn/btm/btm_Api.h>
#include <nn/btm/btm_Types.h>
#include <nn/btm/btm_Result.h>

#include <nn/btm/system/btm_SystemApi.h>
#include <nn/btm/system/btm_SystemResult.h>

#include <nn/btm/debug/btm_DebugApi.h>

#include "TestBtmModule_ApiBtm.h"

namespace ApiBtm
{
    void TestBtmModule_ApiBtm::TestInterface()
    {
        NN_TEST_BTM_MODULE_LOG("%s: Simple IPC sesstion establishment\n", NN_CURRENT_FUNCTION_NAME);

        nn::btm::InitializeBtmInterface();
        nn::btm::FinalizeBtmInterface();

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Multiple IPC sesstion establishments\n", NN_CURRENT_FUNCTION_NAME);

        nn::btm::InitializeBtmInterface();
        nn::btm::InitializeBtmInterface();
        nn::btm::InitializeBtmInterface();
        nn::btm::FinalizeBtmInterface();
        nn::btm::FinalizeBtmInterface();
        nn::btm::FinalizeBtmInterface();

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
    }

    void TestBtmModule_ApiBtm::TestBtmState(nn::btm::BtmState expectedState) const
    {
        NN_TEST_BTM_MODULE_LOG("%s: Check BTM state. Expected state is %d\n", NN_CURRENT_FUNCTION_NAME, expectedState);

        nn::btm::BtmState state;
        nn::btm::GetState(&state);

        NN_ABORT_UNLESS_EQUAL(state, expectedState);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
    }

    void TestBtmModule_ApiBtm::TestHostDeviceProperty(const char* pHostName, size_t hostNameSize, const nn::btm::ClassOfDevice classOfDevice, uint8_t featureSet)
    {
        NN_TEST_BTM_MODULE_LOG("%s: Check host device property. Expected value is Host Name = %s, CoD = %02X:%02X:%02X, Feature Set = %02X\n",
            NN_CURRENT_FUNCTION_NAME,
            pHostName, classOfDevice.cod[0], classOfDevice.cod[1], classOfDevice.cod[2], featureSet);

        nn::btm::HostDeviceProperty hostDeviceProperty;
        nn::btm::GetHostDeviceProperty(&hostDeviceProperty);

        NN_ABORT_UNLESS(memcmp(pHostName, hostDeviceProperty.bdName.name, hostNameSize) == 0);

        NN_ABORT_UNLESS(memcmp(classOfDevice.cod, hostDeviceProperty.classOfDevice.cod, NN_ARRAY_SIZE(classOfDevice.cod)) == 0);

        NN_ABORT_UNLESS_EQUAL(featureSet, hostDeviceProperty.featureSet);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);
    }

    void TestBtmModule_ApiBtm::TestDeviceInfo()
    {
        // ダミーのDeviceInfo 生成用に一旦ペアリングを要求する
        WaitControllerConnection(1);
        DisconnectAllControllers();

        nn::btm::DeviceInfoList     deviceInfoList;
        nn::btm::DeviceInfoList     updatedDeviceInfoList;      // 操作後の比較用
        nn::btm::GeneralInfoList    generalInfoList;            // プロテクトされているDeviceInfo のリスト確認用

        // ダミーのDeviceInfo
        nn::btm::DeviceInfo dummyDeviceInfo[16];

        for (uint8_t i = 0; i < NN_ARRAY_SIZE(dummyDeviceInfo); ++i)
        {
            dummyDeviceInfo[i] = deviceInfoList.device[0];

            dummyDeviceInfo[i].bdAddress.address[0] = 0xAA;
            dummyDeviceInfo[i].bdAddress.address[0] = 0xBB;
            dummyDeviceInfo[i].bdAddress.address[0] = 0xCC;
            dummyDeviceInfo[i].bdAddress.address[0] = 0xDD;
            dummyDeviceInfo[i].bdAddress.address[0] = 0xEE;
            dummyDeviceInfo[i].bdAddress.address[0] = i;
        }

        NN_TEST_BTM_MODULE_LOG("%s: Simple add and remove device info\n", NN_CURRENT_FUNCTION_NAME);

        // 一旦すべてクリアする
        UnpairAllControllers();

        // 念のため、実行前にクリアしておく
        nn::os::ClearEvent(&m_RegisterDeviceInfoEvent);
        nn::btm::AddDeviceInfo(&dummyDeviceInfo[0]);
        nn::os::WaitEvent(&m_RegisterDeviceInfoEvent);

        nn::btm::GetDeviceInfo(&deviceInfoList);        // dummy[0]
        NN_ABORT_UNLESS_EQUAL(deviceInfoList.deviceCount, 1);
        NN_ABORT_UNLESS_EQUAL(deviceInfoList.device[0].bdAddress, dummyDeviceInfo[0].bdAddress);

        // 念のため、実行前にクリアしておく
        nn::os::ClearEvent(&m_RegisterDeviceInfoEvent);
        nn::btm::RemoveDeviceInfo(&dummyDeviceInfo[0].bdAddress);
        nn::os::WaitEvent(&m_RegisterDeviceInfoEvent);

        nn::btm::GetDeviceInfo(&deviceInfoList);        // empty
        NN_ABORT_UNLESS_EQUAL(deviceInfoList.deviceCount, 0);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Add a device info twice\n", NN_CURRENT_FUNCTION_NAME);

        // 念のため、実行前にクリアしておく
        nn::os::ClearEvent(&m_RegisterDeviceInfoEvent);
        nn::btm::AddDeviceInfo(&dummyDeviceInfo[0]);
        nn::os::WaitEvent(&m_RegisterDeviceInfoEvent);

        nn::os::ClearEvent(&m_RegisterDeviceInfoEvent);
        nn::btm::AddDeviceInfo(&dummyDeviceInfo[0]);
        nn::os::WaitEvent(&m_RegisterDeviceInfoEvent);

        nn::btm::GetDeviceInfo(&deviceInfoList);            // dummy[0]
        NN_ABORT_UNLESS_EQUAL(deviceInfoList.deviceCount, 1);
        NN_ABORT_UNLESS_EQUAL(deviceInfoList.device[0].bdAddress, dummyDeviceInfo[0].bdAddress);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Remove a device info twice\n", NN_CURRENT_FUNCTION_NAME);

        // 念のため、実行前にクリアしておく
        nn::os::ClearEvent(&m_RegisterDeviceInfoEvent);
        nn::btm::RemoveDeviceInfo(&dummyDeviceInfo[0].bdAddress);
        nn::os::WaitEvent(&m_RegisterDeviceInfoEvent);

        nn::os::ClearEvent(&m_RegisterDeviceInfoEvent);
        nn::btm::RemoveDeviceInfo(&dummyDeviceInfo[0].bdAddress);
        nn::os::WaitEvent(&m_RegisterDeviceInfoEvent);

        nn::btm::GetDeviceInfo(&deviceInfoList);            // empty
        NN_ABORT_UNLESS_EQUAL(deviceInfoList.deviceCount, 0);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Add device info up to maximum\n", NN_CURRENT_FUNCTION_NAME);

        for (int i = 0; i < nn::btm::COUNT_OF_DI_LIST; ++i)
        {
            // 念のため、実行前にクリアしておく
            nn::os::ClearEvent(&m_RegisterDeviceInfoEvent);
            nn::btm::AddDeviceInfo(&dummyDeviceInfo[i]);
            nn::os::WaitEvent(&m_RegisterDeviceInfoEvent);

            NN_TEST_BTM_MODULE_LOG("%s: %d / %d\n", NN_CURRENT_FUNCTION_NAME, i + 1, nn::btm::COUNT_OF_DI_LIST);
        }

        nn::btm::GetDeviceInfo(&deviceInfoList);        // dummy[0][1][2][3][4][5][6][7][8][9]
        NN_ABORT_UNLESS_EQUAL(deviceInfoList.deviceCount, nn::btm::COUNT_OF_DI_LIST);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Add device info more than maximum\n", NN_CURRENT_FUNCTION_NAME);

        // 念のため、実行前にクリアしておく
        nn::os::ClearEvent(&m_RegisterDeviceInfoEvent);
        nn::btm::AddDeviceInfo(&dummyDeviceInfo[nn::btm::COUNT_OF_DI_LIST]);
        nn::os::WaitEvent(&m_RegisterDeviceInfoEvent);

        nn::btm::GetDeviceInfo(&deviceInfoList);
        NN_ABORT_UNLESS_EQUAL(deviceInfoList.deviceCount, nn::btm::COUNT_OF_DI_LIST);

        for (int i = 0; i < nn::btm::COUNT_OF_DI_LIST; ++i)
        {
            NN_ABORT_UNLESS_EQUAL(deviceInfoList.device[i].bdAddress, dummyDeviceInfo[i + 1].bdAddress);
        }

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Change device info order\n", NN_CURRENT_FUNCTION_NAME);

        // 念のため、実行前にクリアしておく
        nn::os::ClearEvent(&m_RegisterDeviceInfoEvent);
        // 最も古いDeviceInfo を最新へ
        nn::btm::IncreaseDeviceInfoOrder(&deviceInfoList.device[0].bdAddress);
        nn::os::WaitEvent(&m_RegisterDeviceInfoEvent);

        nn::btm::GetDeviceInfo(&updatedDeviceInfoList);         // dummy[2][3][4][5][6][7][8][9][10][1]
        NN_ABORT_UNLESS_EQUAL(updatedDeviceInfoList.deviceCount, nn::btm::COUNT_OF_DI_LIST);
        NN_ABORT_UNLESS_EQUAL(updatedDeviceInfoList.device[nn::btm::COUNT_OF_DI_LIST - 1].bdAddress, deviceInfoList.device[0].bdAddress);
        for (int i = 0; i < nn::btm::COUNT_OF_DI_LIST - 1; ++i)
        {
            NN_ABORT_UNLESS_EQUAL(deviceInfoList.device[i + 1].bdAddress, updatedDeviceInfoList.device[i].bdAddress);
        }

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Protect device info\n", NN_CURRENT_FUNCTION_NAME);

        nn::btm::GetDeviceInfo(&deviceInfoList);            // dummy[2][3][4][5][6][7][8][9][10][1]
        NN_ABORT_UNLESS_EQUAL(deviceInfoList.deviceCount, nn::btm::COUNT_OF_DI_LIST);

        nn::btm::ProtectDeviceInfo(&deviceInfoList.device[0].bdAddress, true);
        // 同期手段がないので、少し待つ
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

        nn::btm::debug::GeneralGet(0, &generalInfoList);        // dummy[{2}]
        NN_ABORT_UNLESS_EQUAL(generalInfoList.count, 1);

        // 念のため、実行前にクリアしておく
        nn::os::ClearEvent(&m_RegisterDeviceInfoEvent);
        nn::btm::AddDeviceInfo(&dummyDeviceInfo[nn::btm::COUNT_OF_DI_LIST + 1]);
        nn::os::WaitEvent(&m_RegisterDeviceInfoEvent);

        nn::btm::GetDeviceInfo(&updatedDeviceInfoList);     // dummy[{2}][4][5][6][7][8][9][10][1][11]
        NN_ABORT_UNLESS_EQUAL(updatedDeviceInfoList.deviceCount, nn::btm::COUNT_OF_DI_LIST);
        NN_ABORT_UNLESS_EQUAL(updatedDeviceInfoList.device[0].bdAddress, deviceInfoList.device[0].bdAddress);

        nn::btm::debug::GeneralGet(0, &generalInfoList);
        NN_ABORT_UNLESS_EQUAL(generalInfoList.count, 1);    // dummy[{2}]

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Explicitly remove protected device info\n", NN_CURRENT_FUNCTION_NAME);

        // 念のため、実行前にクリアしておく
        nn::os::ClearEvent(&m_RegisterDeviceInfoEvent);

        nn::btm::RemoveDeviceInfo(&deviceInfoList.device[0].bdAddress);

        nn::os::WaitEvent(&m_RegisterDeviceInfoEvent);

        nn::btm::GetDeviceInfo(&updatedDeviceInfoList);     // dummy[4][5][6][7][8][9][10][1][11]
        NN_ABORT_UNLESS_EQUAL(updatedDeviceInfoList.deviceCount, nn::btm::COUNT_OF_DI_LIST - 1);
        NN_ABORT_UNLESS_NOT_EQUAL(updatedDeviceInfoList.device[0].bdAddress, deviceInfoList.device[0].bdAddress);

        nn::btm::debug::GeneralGet(0, &generalInfoList);
        NN_ABORT_UNLESS_EQUAL(generalInfoList.count, 1);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Protect a device info for multiple times\n", NN_CURRENT_FUNCTION_NAME);

        nn::btm::GetDeviceInfo(&deviceInfoList);        // dummy[4][5][6][7][8][9][10][1][11]
        NN_ABORT_UNLESS_EQUAL(deviceInfoList.deviceCount, nn::btm::COUNT_OF_DI_LIST - 1);

        for (int i = 0; i < 3; ++i)
        {
            nn::btm::ProtectDeviceInfo(&deviceInfoList.device[0].bdAddress, true);
            // 同期手段がないので、少し待つ
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

            nn::btm::debug::GeneralGet(0, &generalInfoList);        // dummy[{2}][{4}]
            NN_ABORT_UNLESS_EQUAL(generalInfoList.count, 2);
        }
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Unprotect a device info\n", NN_CURRENT_FUNCTION_NAME);

        nn::btm::ProtectDeviceInfo(&deviceInfoList.device[0].bdAddress, false);
        // 同期手段がないので、少し待つ
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

        nn::btm::debug::GeneralGet(0, &generalInfoList);
        NN_ABORT_UNLESS_EQUAL(generalInfoList.count, 1);        // dummy[{2}]
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Protect device info more than maximum\n", NN_CURRENT_FUNCTION_NAME);

        int protectedDeviceCount = 0;
        for (auto deviceInfo : dummyDeviceInfo)
        {
            nn::btm::ProtectDeviceInfo(&deviceInfo.bdAddress, true);
            // 同期手段がないので、少し待つ
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

            NN_TEST_BTM_MODULE_LOG("%s: %d / %d\n", NN_CURRENT_FUNCTION_NAME, ++protectedDeviceCount, NN_ARRAY_SIZE(dummyDeviceInfo));
        }
        NN_UNUSED(protectedDeviceCount);

        nn::btm::debug::GeneralGet(0, &generalInfoList);
        NN_ABORT_UNLESS_EQUAL(generalInfoList.count, nn::btm::COUNT_OF_DI_LIST);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // テスト対象ではないが、後続のテストのために、全プロテクト情報を削除する
        // Radio Off 中は動かないので、一旦Radio On にする
        bool isRadioOn = nn::btm::system::IsRadioEnabled();

        if (!isRadioOn)
        {
            nn::os::ClearEvent(&m_RadioEvent);
            nn::btm::EnableRadio(true);
            nn::os::WaitEvent(&m_RadioEvent);

            NN_ABORT_UNLESS(nn::btm::system::IsRadioEnabled());
        }

        NN_TEST_BTM_MODULE_LOG("%s: Remove all device info protection\n", NN_CURRENT_FUNCTION_NAME);
        nn::btm::debug::GeneralTest(11);
        nn::btm::debug::GeneralGet(0, &generalInfoList);
        NN_ABORT_UNLESS_EQUAL(generalInfoList.count, 0);
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // 元に戻す
        if (!isRadioOn)
        {
            nn::os::ClearEvent(&m_RadioEvent);
            nn::btm::EnableRadio(false);
            nn::os::WaitEvent(&m_RadioEvent);

            NN_ABORT_UNLESS(!nn::btm::system::IsRadioEnabled());
        }

        DisconnectAllControllers();
        UnpairAllControllers();
    } // NOLINT(impl/function_size)

    void TestBtmModule_ApiBtm::TestMode()
    {
        WaitControllerConnection(1);

        nn::btm::DeviceConditionList deviceConditionList;

        // 1 台を除いて、全て切断する
        while (NN_STATIC_CONDITION(true))
        {
            nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);

            nn::btm::GetConnectedDeviceCondition(&deviceConditionList);

            if (deviceConditionList.deviceCount == 1)
            {
                break;
            }
            else if (deviceConditionList.deviceCount > 1)
            {
                nn::btm::HidDisconnect(&deviceConditionList.device[1].bdAddress);

                nn::os::WaitEvent(&m_ConnectedDeviceConditionEvent);
            }
        }

        NN_TEST_BTM_MODULE_LOG("%s: Set bluetooth mode BluetoothMode_Active\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetBluetoothMode(nn::btm::BluetoothMode_Active));

        WaitDeviceCondition(1, nn::btm::BluetoothMode_Active, nn::btm::WlanMode_None, false, false, 0, 10000);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Set bluetooth mode BluetoothMode_Auto\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetBluetoothMode(nn::btm::BluetoothMode_Auto));

        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 0, 10000);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local4\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_Local4));

        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local4, false, false, 1, 10000);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local8\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_Local8));

        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local8, false, false, 1, 10000);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // User8 はアプリに接続数の管理を任せるので、設定後の状態はチェックしない
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_User8\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_User8));

        nn::os::TimerEventType timerEvent;
        nn::TimeSpan oneShotTime;

        nn::os::InitializeTimerEvent(&timerEvent, nn::os::EventClearMode_AutoClear);
        oneShotTime = nn::TimeSpan::FromMilliSeconds(10000);
        nn::os::StartOneShotTimerEvent(&timerEvent, oneShotTime);

        while (NN_STATIC_CONDITION(true))
        {
            if (nn::os::TryWaitTimerEvent(&timerEvent))
            {
                NN_ABORT("[" TEST_NAME "] Timeout to wait device condition list update as expected\n");
            }

            nn::btm::GetConnectedDeviceCondition(&deviceConditionList);

            if (deviceConditionList.wlanMode == nn::btm::WlanMode_User8)
            {
                break;
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
        }

        nn::os::FinalizeTimerEvent(&timerEvent);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_None\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_None));

        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 0, 10000);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Set slot mode SlotMode_4\n", NN_CURRENT_FUNCTION_NAME);

        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, 1);

        nn::btm::DeviceSlotModeList deviceSlotModeList;
        deviceSlotModeList.deviceCount          = deviceConditionList.deviceCount;
        deviceSlotModeList.device[0].bdAddress  = deviceConditionList.device[0].bdAddress;
        deviceSlotModeList.device[0].slotMode   = nn::btm::SlotMode_4;

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetSlotMode(deviceSlotModeList));

        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 1, 10000);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // BluetoothMode_Auto な状態なので、実際はActive にならず、SlotMode2 に落ち着く
        NN_TEST_BTM_MODULE_LOG("%s: Set slot mode SlotMode_Active\n", NN_CURRENT_FUNCTION_NAME);

        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, 1);

        deviceSlotModeList.deviceCount          = deviceConditionList.deviceCount;
        deviceSlotModeList.device[0].bdAddress  = deviceConditionList.device[0].bdAddress;
        deviceSlotModeList.device[0].slotMode   = nn::btm::SlotMode_Active;

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetSlotMode(deviceSlotModeList));

        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 0, 10000);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Set slot mode SlotMode_2\n", NN_CURRENT_FUNCTION_NAME);

        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, 1);

        deviceSlotModeList.deviceCount          = deviceConditionList.deviceCount;
        deviceSlotModeList.device[0].bdAddress  = deviceConditionList.device[0].bdAddress;
        deviceSlotModeList.device[0].slotMode   = nn::btm::SlotMode_2;

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetSlotMode(deviceSlotModeList));

        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 0, 10000);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Enable Zero Retransmission\n", NN_CURRENT_FUNCTION_NAME);

        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
        NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, 1);

        nn::btm::ZeroRetransmissionList zeroRetransmissionList;
        zeroRetransmissionList.enabledReportId[0]   = 0x01;
        zeroRetransmissionList.enabledReportIdCount = 1;

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetRetransmissionMode(deviceConditionList.device[0].bdAddress, zeroRetransmissionList));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllControllers();
        UnpairAllControllers();
    } // NOLINT(impl/function_size)

    void TestBtmModule_ApiBtm::TestEnableSlotSaving()
    {
        WaitControllerConnection(1);

        nn::btm::DeviceConditionList deviceConditionList;

        NN_TEST_BTM_MODULE_LOG("%s: Enable Slot Saving\n", NN_CURRENT_FUNCTION_NAME);
        // 念のため、実行前にクリアしておく
        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);
        nn::btm::EnableSlotSaving(true);
        nn::os::WaitEvent(&m_ConnectedDeviceConditionEvent);

        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);

        NN_ABORT_UNLESS(deviceConditionList.isSlotSaving);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Disable Slot Saving\n", NN_CURRENT_FUNCTION_NAME);
        // 念のため、実行前にクリアしておく
        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);

        nn::btm::EnableSlotSaving(false);

        nn::os::WaitEvent(&m_ConnectedDeviceConditionEvent);

        nn::btm::GetConnectedDeviceCondition(&deviceConditionList);

        NN_ABORT_UNLESS(!deviceConditionList.isSlotSaving);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllControllers();
        UnpairAllControllers();
    }

    void TestBtmModule_ApiBtm::TestEnableRadio()
    {
        WaitControllerConnection(1);

        NN_TEST_BTM_MODULE_LOG("%s: Radio OFF with connection\n", NN_CURRENT_FUNCTION_NAME);
        // 念のため、実行前にクリアしておく
        nn::os::ClearEvent(&m_RadioEvent);
        nn::btm::EnableRadio(false);
        nn::os::WaitEvent(&m_RadioEvent);

        NN_ABORT_UNLESS(!nn::btm::system::IsRadioEnabled());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: Radio ON\n", NN_CURRENT_FUNCTION_NAME);

        // 念のため、実行前にクリアしておく
        nn::os::ClearEvent(&m_RadioEvent);
        nn::btm::EnableRadio(true);
        nn::os::WaitEvent(&m_RadioEvent);

        NN_ABORT_UNLESS(nn::btm::system::IsRadioEnabled());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllControllers();
        UnpairAllControllers();
    }

    void TestBtmModule_ApiBtm::TestLlrNotify()
    {
        nn::btm::DeviceConditionList    deviceConditionList;
        nn::btm::DeviceInfoList         deviceInfoList;

        DisconnectAllControllers();
        UnpairAllControllers();

        NN_TEST_BTM_MODULE_LOG("%s: LLR for an unpaired controller\n", NN_CURRENT_FUNCTION_NAME);

        nn::btm::BdAddress dummyAddress = { { 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF} };

        NN_ABORT_UNLESS(nn::btm::ResultFailureLowLayer().Includes(TryLlr(dummyAddress)));

        // Bluetooth がエラーを返した場合は、LlrStateEvent はシグナルされない
        NN_ABORT_UNLESS(!nn::os::TryWaitEvent(&m_LlrStateEvent));

        NN_ABORT_UNLESS(!nn::btm::IsLlrStarted());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        WaitControllerConnection(1);
        DisconnectAllControllers();

        NN_TEST_BTM_MODULE_LOG("%s: LLR for an paired but unexisting controller\n", NN_CURRENT_FUNCTION_NAME);

        nn::btm::GetDeviceInfo(&deviceInfoList);
        NN_ABORT_UNLESS_GREATER(deviceInfoList.deviceCount, 0);

        // ダミーのデバイスのペアリング情報を追加する
        deviceInfoList.device[0].bdAddress.address[3] = 0xAA;
        deviceInfoList.device[0].bdAddress.address[4] = 0xAA;
        deviceInfoList.device[0].bdAddress.address[5] = 0xAA;

        // 念のため、実行前にクリアしておく
        nn::os::ClearEvent(&m_RegisterDeviceInfoEvent);
        nn::btm::AddDeviceInfo(&deviceInfoList.device[0]);
        nn::os::WaitEvent(&m_RegisterDeviceInfoEvent);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TryLlr(deviceInfoList.device[0].bdAddress));

        while (NN_STATIC_CONDITION(true))
        {
            nn::os::WaitEvent(&m_LlrStateEvent);

            if (!nn::btm::IsLlrStarted())
            {
                // タイムアウトしてデバイスは接続されていないはず
                nn::btm::GetConnectedDeviceCondition(&deviceConditionList);
                NN_ABORT_UNLESS_EQUAL(deviceConditionList.deviceCount, 0);

                break;
            }
        }

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        WaitControllerConnection(1);
        DisconnectAllControllers();

        NN_TEST_BTM_MODULE_LOG("%s: LLR for a paired and unconnected device\n", NN_CURRENT_FUNCTION_NAME);

        nn::btm::GetDeviceInfo(&deviceInfoList);
        NN_ABORT_UNLESS_GREATER(deviceInfoList.deviceCount, 0);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TryLlr(deviceInfoList.device[0].bdAddress));

        // 接続されるはず
        WaitDeviceCondition(1, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 0, 10000);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        NN_TEST_BTM_MODULE_LOG("%s: LLR for a connected device\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS(nn::btm::ResultFailureLowLayer().Includes(TryLlr(deviceConditionList.device[0].bdAddress)));

        // Bluetooth がエラーを返した場合は、LlrStateEvent はシグナルされない
        NN_ABORT_UNLESS(!nn::os::TryWaitEvent(&m_LlrStateEvent));

        NN_ABORT_UNLESS(!nn::btm::IsLlrStarted());

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllControllers();
        UnpairAllControllers();
    } // NOLINT(impl/function_size)

    void TestBtmModule_ApiBtm::TestBtmStateRadioOff(nn::btm::BtmState expectedState)
    {
        DisableRadio();

        TestBtmState(expectedState);
    }

    void TestBtmModule_ApiBtm::TestHostDevicePropertyRadioOff(const char* pHostName, size_t hostNameSize, const nn::btm::ClassOfDevice classOfDevice, uint8_t featureSet)
    {
        DisableRadio();

        // Radio 状態に無関係
        TestHostDeviceProperty(pHostName, hostNameSize, classOfDevice, featureSet);
    }

    void TestBtmModule_ApiBtm::TestDeviceInfoRadioOff()
    {
        DisableRadio();

        // Radio 状態に無関係
        TestDeviceInfo();
    }

    void TestBtmModule_ApiBtm::TestModeRadioOff()
    {
        nn::btm::DeviceConditionList    deviceConditionList;
        nn::btm::DeviceInfoList         deviceInfoList;

        DisableRadio();

        WaitControllerConnection(1);

        // 期待通りに動くはず。ただし、DeviceCount は0。
        NN_TEST_BTM_MODULE_LOG("%s: Set bluetooth mode BluetoothMode_Active\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetBluetoothMode(nn::btm::BluetoothMode_Active));

        WaitDeviceCondition(0, nn::btm::BluetoothMode_Active, nn::btm::WlanMode_None, false, false, 0, 10000);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // 期待通りに動くはず。ただし、DeviceCount は0。
        NN_TEST_BTM_MODULE_LOG("%s: Set bluetooth mode BluetoothMode_Auto\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetBluetoothMode(nn::btm::BluetoothMode_Auto));

        WaitDeviceCondition(0, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 0, 10000);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // 期待通りに動くはず。ただし、DeviceCount は0。
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local4\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_Local4));

        WaitDeviceCondition(0, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local4, false, false, 0, 10000);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // 期待通りに動くはず。ただし、DeviceCount は0。
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_Local8\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_Local8));

        WaitDeviceCondition(0, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_Local8, false, false, 0, 10000);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // 期待通りに動くはず。ただし、DeviceCount は0。
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_User8\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_User8));

        nn::os::TimerEventType timerEvent;
        nn::TimeSpan oneShotTime;

        nn::os::InitializeTimerEvent(&timerEvent, nn::os::EventClearMode_AutoClear);
        oneShotTime = nn::TimeSpan::FromMilliSeconds(10000);
        nn::os::StartOneShotTimerEvent(&timerEvent, oneShotTime);

        while (NN_STATIC_CONDITION(true))
        {
            if (nn::os::TryWaitTimerEvent(&timerEvent))
            {
                NN_ABORT("[" TEST_NAME "] Timeout to wait device condition list update as expected\n");
            }

            nn::btm::GetConnectedDeviceCondition(&deviceConditionList);

            if (deviceConditionList.wlanMode == nn::btm::WlanMode_User8)
            {
                break;
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
        }

        nn::os::FinalizeTimerEvent(&timerEvent);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // 期待通りに動くはず。ただし、DeviceCount は0。
        NN_TEST_BTM_MODULE_LOG("%s: Set wlan mode WlanMode_None\n", NN_CURRENT_FUNCTION_NAME);

        NN_ABORT_UNLESS_RESULT_SUCCESS(TrySetWlanMode(nn::btm::WlanMode_None));

        WaitDeviceCondition(0, nn::btm::BluetoothMode_Auto, nn::btm::WlanMode_None, false, false, 0, 10000);

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Radio Off 中は動かない
        NN_TEST_BTM_MODULE_LOG("%s: Set slot mode SlotMode_4\n", NN_CURRENT_FUNCTION_NAME);

        // Radio OFF 中はDeviceConditionList は使えないので、代わりにDeviceInfoList を使う
        nn::btm::GetDeviceInfo(&deviceInfoList);
        NN_ABORT_UNLESS_GREATER(deviceInfoList.deviceCount, 0);

        nn::btm::DeviceSlotModeList deviceSlotModeList;
        deviceSlotModeList.deviceCount          = deviceInfoList.deviceCount;
        deviceSlotModeList.device[0].bdAddress  = deviceInfoList.device[0].bdAddress;
        deviceSlotModeList.device[0].slotMode   = nn::btm::SlotMode_4;

        NN_ABORT_UNLESS(nn::btm::ResultInvalidStateRadioOff().Includes(TrySetSlotMode(deviceSlotModeList)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Radio Off 中は動かない
        NN_TEST_BTM_MODULE_LOG("%s: Set slot mode SlotMode_Active\n", NN_CURRENT_FUNCTION_NAME);

        deviceSlotModeList.deviceCount          = deviceInfoList.deviceCount;
        deviceSlotModeList.device[0].bdAddress  = deviceInfoList.device[0].bdAddress;
        deviceSlotModeList.device[0].slotMode   = nn::btm::SlotMode_Active;

        NN_ABORT_UNLESS(nn::btm::ResultInvalidStateRadioOff().Includes(TrySetSlotMode(deviceSlotModeList)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Radio Off 中は動かない
        NN_TEST_BTM_MODULE_LOG("%s: Set slot mode SlotMode_2\n", NN_CURRENT_FUNCTION_NAME);

        deviceSlotModeList.deviceCount          = deviceInfoList.deviceCount;
        deviceSlotModeList.device[0].bdAddress  = deviceInfoList.device[0].bdAddress;
        deviceSlotModeList.device[0].slotMode   = nn::btm::SlotMode_2;

        NN_ABORT_UNLESS(nn::btm::ResultInvalidStateRadioOff().Includes(TrySetSlotMode(deviceSlotModeList)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        // Radio Off 中は動かない
        NN_TEST_BTM_MODULE_LOG("%s: Enable Zero Retransmission\n", NN_CURRENT_FUNCTION_NAME);

        nn::btm::ZeroRetransmissionList zeroRetransmissionList;
        zeroRetransmissionList.enabledReportId[0]   = 0x01;
        zeroRetransmissionList.enabledReportIdCount = 1;

        NN_ABORT_UNLESS(nn::btm::ResultInvalidStateRadioOff().Includes(TrySetRetransmissionMode(deviceInfoList.device[0].bdAddress, zeroRetransmissionList)));

        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllControllers();
        UnpairAllControllers();
    } // NOLINT(impl/function_size)

    void TestBtmModule_ApiBtm::TestEnableSlotSavingRadioOff()
    {
        DisableRadio();

        // Radio 状態に無関係
        TestEnableSlotSaving();
    }

    void TestBtmModule_ApiBtm::TestLlrNotifyRadioOff()
    {
        nn::btm::DeviceInfoList deviceInfoList;

        DisableRadio();

        WaitControllerConnection(1);

        // Radio OFF 中はDeviceConditionList は使えないので、代わりにDeviceInfoList を使う
        nn::btm::GetDeviceInfo(&deviceInfoList);
        NN_ABORT_UNLESS_GREATER(deviceInfoList.deviceCount, 0);

        // Radio Off 中は動かない
        NN_TEST_BTM_MODULE_LOG("%s: LLR for a paired but unconnected device\n", NN_CURRENT_FUNCTION_NAME);
        NN_ABORT_UNLESS(nn::btm::ResultInvalidStateRadioOff().Includes(TryLlr(deviceInfoList.device[0].bdAddress)));
        NN_TEST_BTM_MODULE_LOG("%s: Complete\n\n", NN_CURRENT_FUNCTION_NAME);

        DisconnectAllControllers();
        UnpairAllControllers();
    }


    nn::Result TestBtmModule_ApiBtm::TryLlr(const nn::btm::BdAddress& address)
    {
        // 念のため、実行前にクリアしておく
        nn::os::ClearEvent(&m_LlrStateEvent);

        while (NN_STATIC_CONDITION(true))
        {
            auto result = nn::btm::LlrNotify(&address);

            if (nn::btm::ResultBusy().Includes(result))
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
            }
            else
            {
                return result;
            }
        }
    }

    nn::Result TestBtmModule_ApiBtm::TrySetBluetoothMode(nn::btm::BluetoothMode bluetoothMode)
    {
        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);

        while (NN_STATIC_CONDITION(true))
        {
            auto result = nn::btm::SetBluetoothMode(bluetoothMode);

            if (nn::btm::ResultBusy().Includes(result))
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
            }
            else
            {
                return result;
            }
        }
    }

    nn::Result TestBtmModule_ApiBtm::TrySetWlanMode(nn::btm::WlanMode wlanMode)
    {
        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);

        while (NN_STATIC_CONDITION(true))
        {
            auto result = nn::btm::SetWlanMode(wlanMode);

            if (nn::btm::ResultBusy().Includes(result))
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
            }
            else
            {
                return result;
            }
        }
    }

    nn::Result TestBtmModule_ApiBtm::TrySetSlotMode(const nn::btm::DeviceSlotModeList& slotModeList)
    {
        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);

        while (NN_STATIC_CONDITION(true))
        {
            auto result = nn::btm::SetSlotMode(&slotModeList);

            if (nn::btm::ResultBusy().Includes(result))
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
            }
            else
            {
                return result;
            }
        }
    }

    nn::Result TestBtmModule_ApiBtm::TrySetRetransmissionMode(const nn::btm::BdAddress& address, const nn::btm::ZeroRetransmissionList& zeroRetransmissionList)
    {
        nn::os::ClearEvent(&m_ConnectedDeviceConditionEvent);

        while (NN_STATIC_CONDITION(true))
        {
            auto result = nn::btm::HidSetRetransmissionMode(&address, &zeroRetransmissionList);

            if (nn::btm::ResultBusy().Includes(result))
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
            }
            else
            {
                return result;
            }
        }
    }
}   // namespace ApiBtm

