﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_Nfc.h>

#include <nn/nfp/nfp_Lib.h>
#include <nn/nfp/nfp_TagApi.h>
#include <nn/nfp/nfp_MiscApi.h>
#include <nn/nfp/nfp_NpadApi.h>

#include "testBtm_UtilHid.h"

namespace nnt { namespace btm {
nn::hid::NpadIdType g_NpadIds[] = {nn::hid::NpadId::No1,
                                   nn::hid::NpadId::No2,
                                   nn::hid::NpadId::No3,
                                   nn::hid::NpadId::No4,
                                   nn::hid::NpadId::Handheld,
                                   };
const int NpadIdCountMax = sizeof(g_NpadIds) / sizeof(nn::hid::NpadIdType);
nn::hid::NpadJoyDualState oldNpadJoyDualState[NpadIdCountMax * 2];
nn::hid::NpadJoyDualState currentNpadJoyDualState[NpadIdCountMax * 2];
nn::hid::NpadHandheldState oldNpadHandheldState;
nn::hid::NpadHandheldState currentNpadHandheldState;

void HidInit()
{
    NN_LOG("[TestBtm]Init hid Interface\n");
    nn::hid::InitializeNpad();
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleJoyDual::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(g_NpadIds, NpadIdCountMax);

    nn::nfp::Initialize();
}

template <typename TState>
HidButton HidDetectButton(const TState& oldState, const TState& currentState)
{
    nn::hid::NpadButtonSet oldButton = oldState.buttons;
    nn::hid::NpadButtonSet currentButton = currentState.buttons;

    if(currentButton.Test<nn::hid::NpadJoyButton::ZR>() == true)
    {
        if(oldButton.Test<nn::hid::NpadJoyButton::A>() == false &&
                currentButton.Test<nn::hid::NpadJoyButton::A>() == true)
            return HidButton_ZR_A;

        if(oldButton.Test<nn::hid::NpadJoyButton::B>() == false &&
                currentButton.Test<nn::hid::NpadJoyButton::B>() == true)
            return HidButton_ZR_B;

        if(oldButton.Test<nn::hid::NpadJoyButton::X>() == false &&
                currentButton.Test<nn::hid::NpadJoyButton::X>() == true)
            return HidButton_ZR_X;

        if(oldButton.Test<nn::hid::NpadJoyButton::Y>() == false &&
                currentButton.Test<nn::hid::NpadJoyButton::Y>() == true)
            return HidButton_ZR_Y;
    }
    if(currentButton.Test<nn::hid::NpadJoyButton::ZL>() == true)
    {
        if(oldButton.Test<nn::hid::NpadJoyButton::A>() == false &&
                currentButton.Test<nn::hid::NpadJoyButton::A>() == true)
            return HidButton_ZL_A;

        if(oldButton.Test<nn::hid::NpadJoyButton::B>() == false &&
                currentButton.Test<nn::hid::NpadJoyButton::B>() == true)
            return HidButton_ZL_B;

        if(oldButton.Test<nn::hid::NpadJoyButton::X>() == false &&
                currentButton.Test<nn::hid::NpadJoyButton::X>() == true)
            return HidButton_ZL_X;

        if(oldButton.Test<nn::hid::NpadJoyButton::Y>() == false &&
                currentButton.Test<nn::hid::NpadJoyButton::Y>() == true)
            return HidButton_ZL_Y;
    }

    if(oldButton.Test<nn::hid::NpadJoyButton::A>() == false &&
            currentButton.Test<nn::hid::NpadJoyButton::A>() == true)
        return HidButton_A;

    if(oldButton.Test<nn::hid::NpadJoyButton::R>() == false &&
            currentButton.Test<nn::hid::NpadJoyButton::R>() == true)
        return HidButton_R;

    if(oldButton.Test<nn::hid::NpadJoyButton::L>() == false &&
            currentButton.Test<nn::hid::NpadJoyButton::L>() == true)
        return HidButton_L;

    if(oldButton.Test<nn::hid::NpadJoyButton::Up>() == false &&
            currentButton.Test<nn::hid::NpadJoyButton::Up>() == true)
        return HidButton_Up;

    if(oldButton.Test<nn::hid::NpadJoyButton::Down>() == false &&
            currentButton.Test<nn::hid::NpadJoyButton::Down>() == true)
        return HidButton_Down;

    if(oldButton.Test<nn::hid::NpadJoyButton::Right>() == false &&
            currentButton.Test<nn::hid::NpadJoyButton::Right>() == true)
        return HidButton_Right;

    if(oldButton.Test<nn::hid::NpadJoyButton::Left>() == false &&
            currentButton.Test<nn::hid::NpadJoyButton::Left>() == true)
        return HidButton_Left;

    return HidButton_None;
}

HidInput HidUpdateDummy()
{
    HidInput hidInput;
    hidInput.hidButton = HidButton_None;
    hidInput.isUpdated = false;

    for(int i = 0; i < NpadIdCountMax; i++)
    {
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(g_NpadIds[i]);
        if (style.Test<nn::hid::NpadStyleJoyDual>() == true)
        {
            hidInput.hidButton = HidButton_ZR_B;
            hidInput.npadIdType = g_NpadIds[i];
            hidInput.isUpdated = true;
            return hidInput;
        }
    }
    return hidInput;
}

HidInput HidUpdate()
{
    HidInput hidInput;
    hidInput.hidButton = HidButton_None;
    hidInput.isUpdated = false;

    for(int i = 0; i < NpadIdCountMax; i++)
    {
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(g_NpadIds[i]);

        if (style.Test<nn::hid::NpadStyleJoyDual>() == true)
        {
            oldNpadJoyDualState[i] = currentNpadJoyDualState[i];

            //最新のNpadのステートを取得
            nn::hid::GetNpadState(&(currentNpadJoyDualState[i]), g_NpadIds[i]);

            //メニュー操作目的のボタン押しを検出（連続入力を弾く）
            HidButton hidButton;
            hidButton = HidDetectButton(oldNpadJoyDualState[i], currentNpadJoyDualState[i]);
            if(currentNpadJoyDualState[i].buttons.IsAnyOn())
            {
                hidInput.hidButton = hidButton;
                hidInput.npadIdType = g_NpadIds[i];
                hidInput.joyDualState = currentNpadJoyDualState[i];//メニュー操作以外の用途目的で、生データも入れておく
                hidInput.isUpdated = true;
                return hidInput;
            }
        }

        if (style.Test<nn::hid::NpadStyleHandheld>() == true)
        {
            oldNpadHandheldState = currentNpadHandheldState;

            //最新のNpadのステートを取得
            nn::hid::GetNpadState(&(currentNpadHandheldState), g_NpadIds[i]);
            HidButton hidButton;
            hidButton = HidDetectButton(oldNpadHandheldState, currentNpadHandheldState);
            if(currentNpadHandheldState.buttons.IsAnyOn())
            {
                hidInput.hidButton = hidButton;
                hidInput.npadIdType = g_NpadIds[i];
                hidInput.handheldState = currentNpadHandheldState;//メニュー操作以外の用途目的で、生データも入れておく
                hidInput.isUpdated = true;
                return hidInput;
            }
        }
    }
    return hidInput;
}

nn::nfp::DeviceHandle SearchNfpHandle(nn::hid::NpadIdType usingPad)
{
    static const int maxCountOfJoyConR = 4;//右コンは4台まで想定
    nn::nfp::DeviceHandle deviceHandle[maxCountOfJoyConR];
    int count;
    nn::Result result;

    //検出失敗し得るので、成功した場合のみ実際の制御に進む
    result = nn::nfp::ListDevices(&deviceHandle[0], &count, maxCountOfJoyConR);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    //if(!result.IsSuccess())
    //    return result;

    //ボタンを押したジョイコンの、NFCdeviceHandleを確定させる
    for(int i=0;i<count;i++)
    {
        nn::hid::NpadIdType npadIdType;
        result = nn::nfp::GetNpadId(&npadIdType, deviceHandle[i]);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        if(usingPad == npadIdType)
        {
            return deviceHandle[i];
        }
    }
    NN_ABORT("[testBtm_ManualDisp]Can't find NFC in controller.\n");
    return deviceHandle[0];//ここには来ない
}

nn::Result StartNfpScan(nn::hid::NpadIdType usingPad)
{
    nn::nfp::DeviceHandle handle;
    handle = SearchNfpHandle(usingPad);

    auto result = nn::nfp::StartDetection(handle);
    return result;
}

nn::Result StopNfpScan(nn::hid::NpadIdType usingPad)
{
    nn::nfp::DeviceHandle handle;
    handle = SearchNfpHandle(usingPad);

    auto result = nn::nfp::StopDetection(handle);
    return result;
}

}}
