﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_Color.h>

#include "testBtm_UtilGraphic.h"

namespace nnt { namespace btm {

GraphicsSystem::GraphicsSystem() NN_NOEXCEPT
    : m_GraphicsSystemMemorySize(0)
    , m_pMemoryPool(0)
    , m_pMemoryPoolOffset(nn::gfx::util::MemoryPoolAllocator::InvalidOffset)
    , m_TextureDescriptorIndex(0)
    , m_SamplerDescriptorIndex(0)
    , m_pPrimitiveRenderer(0)
    , m_FrameCount(0)
{
}

void GraphicsSystem::MakeCommandFunc(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData) NN_NOEXCEPT
{
    auto& gfxsys = *reinterpret_cast<GraphicsSystem*>(pUserData);

    pGraphicsFramework->BeginFrame(bufferIndex);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = pGraphicsFramework->GetRootCommandBuffer(bufferIndex);

        nn::gfx::ColorTargetView* target = pGraphicsFramework->GetColorTargetView();
        rootCommandBuffer->ClearColor(target,
            static_cast<float>(nn::util::Color4u8::Black().GetR()) / nn::util::Color4u8::White().GetR(),
            static_cast<float>(nn::util::Color4u8::Black().GetG()) / nn::util::Color4u8::White().GetG(),
            static_cast<float>(nn::util::Color4u8::Black().GetB()) / nn::util::Color4u8::White().GetB(),
            static_cast<float>(nn::util::Color4u8::Black().GetA()) / nn::util::Color4u8::White().GetA(),
            0);
        rootCommandBuffer->SetRenderTargets(1, &target, nullptr);
        rootCommandBuffer->SetViewportScissorState(pGraphicsFramework->GetViewportScissorState());

        gfxsys.GetDebugFontTextWriter()->Draw(rootCommandBuffer);
    }
    pGraphicsFramework->EndFrame(bufferIndex, true);
}

void GraphicsSystem::Initialize(size_t graphicsSystemMemorySize) NN_NOEXCEPT
{
    Initialize(graphicsSystemMemorySize, nullptr);
}

void GraphicsSystem::Initialize(size_t graphicsSystemMemorySize, void (*makeCommandFunc)(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData)) NN_NOEXCEPT
{
    m_GraphicsSystemMemorySize = graphicsSystemMemorySize;
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(m_GraphicsSystemMemorySize);

    nns::gfx::GraphicsFramework::FrameworkInfo info;
    info.SetDefault();
    info.SetDisplayWidth(DisplayWidth);
    info.SetDisplayHeight(DisplayHeight);
    info.SetBufferCount(BufferCount);
    info.SetSwapChainBufferCount(BufferCount);
    info.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, (BufferCount * 8 + 8) * 1024 * 1024);
    info.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_CommandBuffer, (BufferCount * 8 + 8) * 1024 * 1024);
    // info.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_Shader, 0 * 1024 * 1024);
    // info.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_Data, 1 * 1024 * 1024);
    info.SetRootCommandBufferCommandMemorySize(8 * 1024 * 1024);

    m_GraphicsFramework.Initialize(info);

    if (makeCommandFunc != nullptr)
    {
        m_GraphicsFramework.SetMakeCommandCallback(makeCommandFunc, this);
    }
    else
    {
        m_GraphicsFramework.SetMakeCommandCallback(MakeCommandFunc, this);
    }
}

void GraphicsSystem::InitializeDebugFont() NN_NOEXCEPT
{
    m_GraphicsFramework.InitializeDebugFontTextWriter(&m_DebugFontTextWriter, CharCountMax, BufferCount);
}

void GraphicsSystem::InitializePrimitiveRenderer() NN_NOEXCEPT
{
    nns::gfx::PrimitiveRenderer::RendererInfo info;
    info.SetDefault();
    info.SetAllocator(nns::gfx::GraphicsFramework::DefaultAllocateFunction, nullptr);
    info.SetAdditionalBufferSize(1024 * 4);
    info.SetDrawCallCountMax(1024 * 4);
    info.SetViewFunctionCallCountMax(1024 * 4);

    info.SetMultiBufferQuantity(BufferCount);

    m_pPrimitiveRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer(m_GraphicsFramework.GetDevice(), info);
    m_pPrimitiveRenderer->SetScreenWidth(m_GraphicsFramework.GetDisplayWidth());
    m_pPrimitiveRenderer->SetScreenHeight(m_GraphicsFramework.GetDisplayHeight());
}

void GraphicsSystem::Finalize() NN_NOEXCEPT
{
    m_GraphicsFramework.QueueFinish();

    m_GraphicsFramework.Finalize();
}

void GraphicsSystem::FinalizeDebugFont() NN_NOEXCEPT
{
    m_GraphicsFramework.FinalizeDebugFontTextWriter(&m_DebugFontTextWriter);
}

void GraphicsSystem::FinalizePrimitiveRenderer() NN_NOEXCEPT
{
    nns::gfx::PrimitiveRenderer::DestroyRenderer(m_pPrimitiveRenderer, m_GraphicsFramework.GetDevice(), m_GraphicsFramework.DefaultFreeFunction, NULL);
}

void GraphicsSystem::Update() NN_NOEXCEPT
{
    m_GraphicsFramework.ProcessFrame();
    m_FrameCount++;
}

nn::gfx::util::DebugFontTextWriter* GraphicsSystem::GetDebugFontTextWriter() NN_NOEXCEPT
{
    return &m_DebugFontTextWriter.object;
}

nns::gfx::PrimitiveRenderer::Renderer* GraphicsSystem::GetPrimitiveRenderer() NN_NOEXCEPT
{
    return m_pPrimitiveRenderer;
}
}}
