﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/nn_TimeSpan.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/TargetConfigs/build_Platform.h>
#include <nn/nn_Log.h>
#include <nns/hid.h>
#include <nn/btm/debug/btm_DebugApi.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#include <nn/bluetooth.h>

#include "../Common/BluetoothTools_ApplicationHeap.h"
#include "../Common/BluetoothTools_GraphicsSystem.h"
#include "../Common/BluetoothTools_WindowMessage.h"
#include "../Common/BluetoothTools_FontSystem.h"
#include "../Common/BluetoothTools_Color.h"
#include "BleTestTool_Scene.h"
#include "BleTestTool_ScanScene.h"
#include "BleTestTool_GattClientScene.h"
#include "BleTestTool_Uuid.h"

namespace
{
    nn::os::SystemEventType g_SystemEventForBleScan;                        //!< BLE スキャン でデバイスが発見された際にシグナルされるイベント
    nn::os::SystemEventType g_SystemEventForBleConnection;                  //!< BLE の接続状態が変化した際にシグナルされるイベント
    nn::os::SystemEventType g_SystemEventForBleServiceDiscovery;            //!< GATT Service Discovery でGATT Attribute が発見された際にシグナルされるイベント
    nn::os::SystemEventType g_SystemEventForBleMtuConfig;                   //!< GATT Server との間でMTU 設定が完了した際にシグナルされるイベント
    nn::os::SystemEventType g_SystemEventForBleGattOperation;               //!< GATT Attribute に対する操作が完了した際にシグナルされるイベント

    nn::os::MultiWaitType g_MultiWait;
    nn::os::MultiWaitHolderType g_MultiWaitHolederForBleScan;
    nn::os::MultiWaitHolderType g_MultiWaitHolederForBleConnection;
    nn::os::MultiWaitHolderType g_MultiWaitHolederForBleServiceDiscovery;
    nn::os::MultiWaitHolderType g_MultiWaitHolederForBleMtuConfig;
    nn::os::MultiWaitHolderType g_MultiWaitHolederForBleGattOperation;

    typedef void(*Callback)();

    void BleScanEventHandler();                 //!< BLE スキャンイベントを処理
    void BleConnectionEventHandler();           //!< BLE 接続イベントを処理
    void BleServiceDiscoveryHandler();          //!< GATT Service Discovery イベントを処理
    void BleMtuEventHandler();                  //!< BLE MTU 設定イベントを処理
    void BleGattOperationHandler();             //!< GATT Attribute に対するイベントを処理

    const std::string g_ScanSceneTitle          = "SCAN";
    const std::string g_GattClientScene0Title   = "CONNECTION0";
    const std::string g_GattClientScene1Title   = "CONNECTION1";
    const std::string g_GattClientScene2Title   = "CONNECTION2";
    const std::string g_GattClientScene3Title   = "CONNECTION3";
    const std::string g_DebugSceneTitle         = "DEBUG";

    SceneBase* gp_CurrentScene;
    ScanScene g_ScanScene = ScanScene(g_ScanSceneTitle);
    GattClientScene g_GattClientScene0 = GattClientScene(g_GattClientScene0Title, 0);
    GattClientScene g_GattClientScene1 = GattClientScene(g_GattClientScene1Title, 1);
    GattClientScene g_GattClientScene2 = GattClientScene(g_GattClientScene2Title, 2);
    GattClientScene g_GattClientScene3 = GattClientScene(g_GattClientScene3Title, 3);

    nn::os::ThreadType g_MultiWaitThread;                               //!< イベントの多重待ちを行うスレッド
    void MultiWaitThreadFunction(void* arg);                            //!< イベントの多重待ちを行うスレッド関数
    const size_t ThreadStackSize = 8192;                                //!< イベント待ちスレッドのスタックサイズ
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ThreadStackSize];     //!< イベント待ちスレッドのスタック

    const size_t ApplicationHeapSize = 64 * 1024 * 1024;

    const int FrameBufferWidth = 1280;
    const int FrameBufferHeight = 720;
    const int FrameRate = 60;

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    const size_t GraphicsMemorySize = 8 * 1024 * 1024;

    void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
        return aligned_alloc(alignment, nn::util::align_up(size, alignment));
    }

    void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
        free(addr);
    }

    void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
        return realloc(addr, newSize);
    }

    void ChangeCurrentScene(bool toNext)
    {
        if (toNext)
        {
            if (gp_CurrentScene->GetTitle().compare(g_ScanSceneTitle) == 0)
            {
                gp_CurrentScene = &g_GattClientScene0;
            }
            else if (gp_CurrentScene->GetTitle().compare(g_GattClientScene0Title) == 0)
            {
                gp_CurrentScene = &g_GattClientScene1;
            }
            else if (gp_CurrentScene->GetTitle().compare(g_GattClientScene1Title) == 0)
            {
                gp_CurrentScene = &g_GattClientScene2;
            }
            else if (gp_CurrentScene->GetTitle().compare(g_GattClientScene2Title) == 0)
            {
                gp_CurrentScene = &g_GattClientScene3;
            }
            else if (gp_CurrentScene->GetTitle().compare(g_GattClientScene3Title) == 0)
            {
                gp_CurrentScene = &g_ScanScene;
            }
        }
        else
        {
            if (gp_CurrentScene->GetTitle().compare(g_ScanSceneTitle) == 0)
            {
                gp_CurrentScene = &g_GattClientScene3;
            }
            else if (gp_CurrentScene->GetTitle().compare(g_GattClientScene0Title) == 0)
            {
                gp_CurrentScene = &g_ScanScene;
            }
            else if (gp_CurrentScene->GetTitle().compare(g_GattClientScene1Title) == 0)
            {
                gp_CurrentScene = &g_GattClientScene0;
            }
            else if (gp_CurrentScene->GetTitle().compare(g_GattClientScene2Title) == 0)
            {
                gp_CurrentScene = &g_GattClientScene1;
            }
            else if (gp_CurrentScene->GetTitle().compare(g_GattClientScene3Title) == 0)
            {
                gp_CurrentScene = &g_GattClientScene2;
            }
        }
    }

    nns::hid::ButtonSet GetControllerInput(nns::hid::ControllerManager* pControllerManager)
    {
        NN_ASSERT_NOT_NULL(pControllerManager);

        pControllerManager->Update();

        nns::hid::ButtonSet buttonSet;
        buttonSet.Reset();

        for (int i = 0; i < 4; ++i)
        {
            nns::hid::GamePad* gamePad = reinterpret_cast<nns::hid::GamePad*>(pControllerManager->GetController(nns::hid::ControllerId_GamePad, i));

            buttonSet |= gamePad->GetButtonsDown();
        }

        nns::hid::DebugPad* debugPad = reinterpret_cast<nns::hid::DebugPad*>(pControllerManager->GetController(nns::hid::ControllerId_DebugPad, 0));
        buttonSet |= debugPad->GetButtonsDown();

        return buttonSet;
    }

    void BleScanEventHandler()
    {
        nn::os::ClearSystemEvent(&g_SystemEventForBleScan);

        g_ScanScene.UpdateScanResult();
    }

    void BleConnectionEventHandler()
    {
        nn::os::ClearSystemEvent(&g_SystemEventForBleConnection);

        g_GattClientScene0.UpdateConnectionState();
        g_GattClientScene1.UpdateConnectionState();
        g_GattClientScene2.UpdateConnectionState();
        g_GattClientScene3.UpdateConnectionState();
    }

    void BleServiceDiscoveryHandler()
    {
        nn::os::ClearSystemEvent(&g_SystemEventForBleServiceDiscovery);

        g_GattClientScene0.UpdateServerProfile();
        g_GattClientScene1.UpdateServerProfile();
        g_GattClientScene2.UpdateServerProfile();
        g_GattClientScene3.UpdateServerProfile();
    }

    void BleMtuEventHandler()
    {
        nn::os::ClearSystemEvent(&g_SystemEventForBleMtuConfig);

        g_GattClientScene0.UpdateMtu();
        g_GattClientScene1.UpdateMtu();
        g_GattClientScene2.UpdateMtu();
        g_GattClientScene3.UpdateMtu();
    }

    void BleGattOperationHandler()
    {
        nn::os::ClearSystemEvent(&g_SystemEventForBleGattOperation);

        // GATT Operation に関しては、即結果を取得する
        nn::bluetooth::BleClientGattOperationInfo info;
        nn::bluetooth::GetGattOperationResult(&info);

        if (info.connectionHandle == g_GattClientScene0.GetConnectionHandle())
        {
            g_GattClientScene0.UpdateGattOperationStatus(info);
        }
        else if (info.connectionHandle == g_GattClientScene1.GetConnectionHandle())
        {
            g_GattClientScene1.UpdateGattOperationStatus(info);
        }
        else if (info.connectionHandle == g_GattClientScene2.GetConnectionHandle())
        {
            g_GattClientScene2.UpdateGattOperationStatus(info);
        }
        else if (info.connectionHandle == g_GattClientScene3.GetConnectionHandle())
        {
            g_GattClientScene3.UpdateGattOperationStatus(info);
        }
    }

    //!< BLE 関連の初期化を行います
    void initializeBle()
    {
        // ライブラリの初期化
        nn::bluetooth::InitializeBle();

        // BTM のデバッグIPC の初期化
        nn::btm::debug::InitializeBtmDebugInterface();

        // システムイベントの取得
        nn::bluetooth::AcquireBleScanEvent(&g_SystemEventForBleScan);
        nn::bluetooth::AcquireBleConnectionStateChangedEvent(&g_SystemEventForBleConnection);
        nn::bluetooth::AcquireBleServiceDiscoveryEvent(&g_SystemEventForBleServiceDiscovery);
        nn::bluetooth::AcquireBleMtuConfigEvent(&g_SystemEventForBleMtuConfig);
        nn::bluetooth::AcquireBleGattOperationEvent(&g_SystemEventForBleGattOperation);

        // 多重待ちオブジェクトの初期化とリンク
        nn::os::InitializeMultiWaitHolder(&g_MultiWaitHolederForBleScan, &g_SystemEventForBleScan);
        nn::os::SetMultiWaitHolderUserData(&g_MultiWaitHolederForBleScan, reinterpret_cast<uintptr_t>(BleScanEventHandler));

        nn::os::InitializeMultiWaitHolder(&g_MultiWaitHolederForBleConnection, &g_SystemEventForBleConnection);
        nn::os::SetMultiWaitHolderUserData(&g_MultiWaitHolederForBleConnection, reinterpret_cast<uintptr_t>(BleConnectionEventHandler));

        nn::os::InitializeMultiWaitHolder(&g_MultiWaitHolederForBleServiceDiscovery, &g_SystemEventForBleServiceDiscovery);
        nn::os::SetMultiWaitHolderUserData(&g_MultiWaitHolederForBleServiceDiscovery, reinterpret_cast<uintptr_t>(BleServiceDiscoveryHandler));

        nn::os::InitializeMultiWaitHolder(&g_MultiWaitHolederForBleMtuConfig, &g_SystemEventForBleMtuConfig);
        nn::os::SetMultiWaitHolderUserData(&g_MultiWaitHolederForBleMtuConfig, reinterpret_cast<uintptr_t>(BleMtuEventHandler));

        nn::os::InitializeMultiWaitHolder(&g_MultiWaitHolederForBleGattOperation, &g_SystemEventForBleGattOperation);
        nn::os::SetMultiWaitHolderUserData(&g_MultiWaitHolederForBleGattOperation, reinterpret_cast<uintptr_t>(BleGattOperationHandler));

        nn::os::InitializeMultiWait(&g_MultiWait);
        nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_MultiWaitHolederForBleScan);
        nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_MultiWaitHolederForBleConnection);
        nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_MultiWaitHolederForBleServiceDiscovery);
        nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_MultiWaitHolederForBleMtuConfig);
        nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_MultiWaitHolederForBleGattOperation);

        // 多重待ちスレッドを生成、開始
        auto result = nn::os::CreateThread(&g_MultiWaitThread, MultiWaitThreadFunction, nullptr, g_ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority);
        NN_ASSERT(result.IsSuccess());
        nn::os::StartThread(&g_MultiWaitThread);
    }

    //!< BLE 関連のファイナライズを行います
    void finalizeBle()
    {
        // スレッドの終了を待つ
        nn::os::WaitThread(&g_MultiWaitThread);
        nn::os::DestroyThread(&g_MultiWaitThread);

        // 多重待ちオブジェクトのアンリンクと破棄
        nn::os::UnlinkAllMultiWaitHolder(&g_MultiWait);
        nn::os::FinalizeMultiWait(&g_MultiWait);

        nn::os::FinalizeMultiWaitHolder(&g_MultiWaitHolederForBleScan);
        nn::os::FinalizeMultiWaitHolder(&g_MultiWaitHolederForBleConnection);
        nn::os::FinalizeMultiWaitHolder(&g_MultiWaitHolederForBleServiceDiscovery);
        nn::os::FinalizeMultiWaitHolder(&g_MultiWaitHolederForBleMtuConfig);
        nn::os::FinalizeMultiWaitHolder(&g_MultiWaitHolederForBleGattOperation);

        // システムイベントの破棄
        nn::os::DestroySystemEvent(&g_SystemEventForBleScan);
        nn::os::DestroySystemEvent(&g_SystemEventForBleConnection);
        nn::os::DestroySystemEvent(&g_SystemEventForBleServiceDiscovery);
        nn::os::DestroySystemEvent(&g_SystemEventForBleMtuConfig);
        nn::os::DestroySystemEvent(&g_SystemEventForBleGattOperation);

        // BTM のデバッグIPC の末期化
        nn::btm::debug::FinalizeBtmDebugInterface();
    }

    //!< イベントの多重待ちを行うスレッド関数
    void MultiWaitThreadFunction(void* arg)
    {
        NN_UNUSED(arg);

        for (;;)
        {
            nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&g_MultiWait);
            auto calllback = reinterpret_cast<Callback>(nn::os::GetMultiWaitHolderUserData(holder));

            if (calllback)
            {
                // コールバックの呼び出し
                calllback();
            }
        }
    }

#endif
}   // namespace

extern "C" void nnMain()
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::InitializeGraphics(std::malloc(GraphicsMemorySize), GraphicsMemorySize);
#endif
    nns::hid::ControllerManager controllerManager;
    nns::hid::util::SetControllerManagerWithDefault(&controllerManager);

    // BLE の開始処理
    initializeBle();

    ApplicationHeap applicationHeap;
    applicationHeap.Initialize(ApplicationHeapSize);

    GraphicsSystem* pGraphicsSystem = new GraphicsSystem();
    pGraphicsSystem->Initialize(&applicationHeap, FrameBufferWidth, FrameBufferHeight);

    FontSystem* pFontSystem = new FontSystem();
    pFontSystem->Initialize(&applicationHeap, pGraphicsSystem);

    nn::gfx::util::DebugFontTextWriter& textWriter = pFontSystem->GetDebugFontTextWriter();

    g_ScanScene.UpdateScanResult();

    gp_CurrentScene = &g_ScanScene;

    while (NN_STATIC_CONDITION(true))
    {
        nns::hid::ButtonSet controllerInput = GetControllerInput(&controllerManager);

        if ((controllerInput & nns::hid::Button::R::Mask).IsAnyOn())
        {
            ChangeCurrentScene(true);
        }
        else if ((controllerInput & nns::hid::Button::L::Mask).IsAnyOn())
        {
            ChangeCurrentScene(false);
        }
        else if ((controllerInput & nns::hid::Button::Plus::Mask).IsAnyOn())
        {
            NN_LOG("Exit.\n");
            break;
        }

        gp_CurrentScene->ProcessControllerInput(controllerInput);
        gp_CurrentScene->Draw(&textWriter);

        pGraphicsSystem->BeginDraw();
        pFontSystem->Draw();
        pGraphicsSystem->EndDraw();
        pGraphicsSystem->Synchronize(nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate));
    }

    // BLE の終了処理
    finalizeBle();

    pFontSystem->Finalize();
    delete pFontSystem;

    pGraphicsSystem->Finalize();
    delete pGraphicsSystem;

    applicationHeap.Finalize();
}
