﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/bluetooth_Client.h"
#include <nn/util/util_BitPack.h>
#include <cstdlib>

#include <nnt/nntest.h>
#include <nn/nn_Log.h>

using namespace nn::bluetooth;

// TSI Test 1 specific states
typedef enum
{
    StateWaitOnSetTsiCallback,
    StateWaitOnInputReport,
    StateVerifyOutputReportType,
    StateGatherHidPackets,
    StateDisplayResults,
    StateSetNewTsi
} TestStateType;

// Valid report types by TSI and amount of valid types
// Format: [Size][Type1][Type2]...[TypeN]
int aTsi0ReportTypes[4] = {3, ReportId_Basic, ReportId_MCU, ReportId_Attachment};                  // 10 ms interval
int aTsi1ReportTypes[3] = {2, ReportId_Basic, ReportId_MCU};                                       // 5  ms interval
int aTsi2ReportTypes[4] = {3, ReportId_Basic, ReportId_MCU, ReportId_Attachment};                  // 10 ms interval
int aTsi3ReportTypes[5] = {4, ReportId_Basic, ReportId_MCU, ReportId_Attachment, ReportId_Audio};  // 15 ms interval
int aTsi4ReportTypes[4] = {3, ReportId_Basic, ReportId_MCU, ReportId_Attachment};                  // 10 ms interval
int aTsi5ReportTypes[5] = {4, ReportId_Basic, ReportId_MCU, ReportId_Attachment, ReportId_Audio};  // 15 ms interval
int aTsi6ReportTypes[5] = {4, ReportId_Basic, ReportId_MCU, ReportId_Attachment, ReportId_Audio};  // 15 ms interval
int aTsi7ReportTypes[5] = {4, ReportId_Basic, ReportId_MCU, ReportId_Attachment, ReportId_Audio};  // 15 ms interval
int aTsi8ReportTypes[5] = {4, ReportId_Basic, ReportId_MCU, ReportId_Attachment, ReportId_Audio};  // 15 ms interval
int aTsi9ReportTypes[4] = {3, ReportId_Basic, ReportId_MCU, ReportId_Attachment};                  // 10 ms interval
int aTsi10ReportTypes[5] = {4, ReportId_Basic, ReportId_MCU, ReportId_Attachment, ReportId_Audio}; // 15 ms interval

int *aValidReportTypeByTsi[11] = {aTsi0ReportTypes, aTsi1ReportTypes, aTsi2ReportTypes, aTsi3ReportTypes,
                                  aTsi4ReportTypes, aTsi5ReportTypes, aTsi6ReportTypes, aTsi7ReportTypes,
                                  aTsi8ReportTypes, aTsi9ReportTypes, aTsi10ReportTypes};

class tsiTest2Client : public bluetoothClient
{
    public:
        int deviceConnected = 0;
        TestStateType TestState;

        /*******************************************************************************
        * EventFromConnectionStateCallback()
        *******************************************************************************/
        void EventFromConnectionStateCallback(const nn::bluetooth::InfoFromConnectionStateCallback* pInfo)
        {
            const char* connStrP;
            switch(pInfo->state)
            {
                case BTHH_CONN_STATE_CONNECTED:               connStrP = "BTHH_CONN_STATE_CONNECTED"; break;
                case BTHH_CONN_STATE_CONNECTING:              connStrP = "BTHH_CONN_STATE_CONNECTING"; break;
                case BTHH_CONN_STATE_DISCONNECTED:            connStrP = "BTHH_CONN_STATE_DISCONNECTED"; break;
                case BTHH_CONN_STATE_DISCONNECTING:           connStrP = "BTHH_CONN_STATE_DISCONNECTING"; break;
                case BTHH_CONN_STATE_FAILED_MOUSE_FROM_HOST:  connStrP = "BTHH_CONN_STATE_FAILED_MOUSE_FROM_HOST"; break;
                case BTHH_CONN_STATE_FAILED_KBD_FROM_HOST:    connStrP = "BTHH_CONN_STATE_FAILED_KBD_FROM_HOST"; break;
                case BTHH_CONN_STATE_FAILED_TOO_MANY_DEVICES: connStrP = "BTHH_CONN_STATE_FAILED_TOO_MANY_DEVICES"; break;
                case BTHH_CONN_STATE_FAILED_NO_BTHID_DRIVER:  connStrP = "BTHH_CONN_STATE_FAILED_NO_BTHID_DRIVER"; break;
                case BTHH_CONN_STATE_FAILED_GENERIC:          connStrP = "BTHH_CONN_STATE_FAILED_GENERIC"; break;
                case BTHH_CONN_STATE_UNKNOWN:                 connStrP = "BTHH_CONN_STATE_UNKNOWN"; break;
                default:
                    NN_LOG("Invalid ConnectionState: 0x%X\n", pInfo->state);
                    connStrP = "UNKNOWN";
            }
            NN_LOG("ConnectionState(HID): %s %s\n", connStrP, toHexString(pInfo->bluetoothAddress));

            if(pInfo->state==BTHH_CONN_STATE_CONNECTED)
            {
                // Stop discovery and other devices from connecting
                deviceConnected++;
                NN_LOG("\n***********Controller connected************\n");
                // Connected from HID view.

                // Get values for first test cycle
                GetRandomTestParams(aValidReportTypeByTsi);

                // Change the Tsi mode.
                NN_LOG("HID sending new TSI:[%d]\n", tsiMode);
                nn::bluetooth::ExtSetTsi(&pInfo->bluetoothAddress, tsiMode);

                TestState = StateWaitOnSetTsiCallback;
            }
            if(pInfo->state==BTHH_CONN_STATE_DISCONNECTED)
            {
                NN_LOG("**********Controller disconnected**********\n\n");
            }
        }
        /*******************************************************************************
        * EventFromExtensionCallbacks()
        *******************************************************************************/
        void EventFromExtensionCallbacks(const nn::bluetooth::InfoFromExtensionCallbacks* pInfo)
        {
            const char* extStrP;
            switch (pInfo->eventType)
            {
                case nn::bluetooth::EventFromTsiSetCallback:    extStrP = "EventFromTsiSetCallback"; break;
                case nn::bluetooth::EventFromTsiExitCallback:   extStrP = "EventFromTsiExitCallback"; break;
                case nn::bluetooth::EventFromBurstSetCallback:  extStrP = "EventFromBurstSetCallback"; break;
                case nn::bluetooth::EventFromBurstExitCallback: extStrP = "EventFromBurstExitCallback"; break;
                default:
                    NN_LOG("Invalid Extension Event: 0x%X\n", pInfo->eventType);
                    extStrP = "UNKNOWN";
            }
            NN_LOG("Extension(HID): %s, %d\n", extStrP, pInfo->eventType);

            if(pInfo->eventType==nn::bluetooth::EventFromTsiSetCallback)
            {
                // TSI set, now set the output report type
                nn::bluetooth::HidData hidData;
                memset(&hidData, 0, sizeof(hidData));
                hidData.length = 49;
                hidData.data[0] = 0x01;
                hidData.data[10] = 0x03;
                hidData.data[11] = outputReport;
                NN_LOG("HID sending new output report:[0x%02x] to device:[%s]\n", outputReport, toHexString(pInfo->bluetoothAddress));
                nn::bluetooth::HidSendData(&pInfo->bluetoothAddress, &hidData);

                TestState = StateWaitOnInputReport;
            }
            if(pInfo->eventType==nn::bluetooth::EventFromTsiExitCallback)
            {
                // Get new values for next test cycle
                GetRandomTestParams(aValidReportTypeByTsi);

                // Change the Tsi mode.
                NN_LOG("HID sending new TSI:[%d]\n", tsiMode);
                nn::bluetooth::ExtSetTsi(&pInfo->bluetoothAddress, tsiMode);

                TestState = StateWaitOnSetTsiCallback;
            }
        }
        /*******************************************************************************
        * EventFromGetReportDataCallback()
        *******************************************************************************/
        void EventFromGetReportDataCallback(const nn::bluetooth::InfoFromGetReportCallback* pInfo)
        {
            const nn::bluetooth::HidReport* pReport = &pInfo->rptData; // NOLINT

            int packetCount;

            // Wait for INPUT report 0x21 to guarantee the new OUTPUT report was set
            if(TestState == StateWaitOnInputReport)
                TestState = StateVerifyOutputReportType;

            else if(TestState == StateVerifyOutputReportType)
                VerifyOutputReportType(pReport);

            else if(TestState == StateGatherHidPackets)
                packetCount = GatherHidPackets(pReport);

            if(TestState == StateDisplayResults)
            {
                static int testCycle = 0;
                int expected, lost;
                int tsiInterval = GetTsiIntervalMs(tsiMode);

                expected = 5000 / tsiInterval;
                lost = expected - packetCount;
                testCycle++;

                NN_LOG("[Packets received over 5 seconds: %d] [Expected: %d] [Estimated loss: %d]\n", packetCount, expected, lost);
                NN_LOG("\n***********Completed Test Cycle [%d]************\n\n", testCycle);

                TestState = StateSetNewTsi;
            }
            if(TestState == StateSetNewTsi)
            {
                TestState = StateWaitOnSetTsiCallback;

                // Exit current TSI
                nn::bluetooth::ExtSetTsi(&pInfo->bluetoothAddress, 0xff);
                NN_LOG("HID exiting current TSI:[%d]\n", tsiMode);
            }
        }

    private:
        // Default TSI and OUTPUT report
        int tsiMode = 0;
        int outputReport = ReportId_Basic;

        /*******************************************************************************
        * GetRandomTestParams()
        *
        * Set tsiMode & outputReport to random valid values
        *******************************************************************************/
        void GetRandomTestParams(int **aReportTypesByTsi)
        {
            int *ptr;

            // Get a random TSI value between 0-10
            tsiMode = rand() % 11;

            ptr = aReportTypesByTsi[tsiMode];

            // Get random output report, skip ptr[0] which contains the size
            outputReport = ptr[rand() % ptr[0] + 1];

            NN_LOG("\n*****New Test Cycle Parameters TSI:[%d], Output Report:[0x%02x]*****\n", tsiMode, outputReport);
        }
        /*******************************************************************************
        * GetTsiIntervalMs()
        *
        * Returns the interval time for the inputed TSI number
        *******************************************************************************/
        int GetTsiIntervalMs(int tsiNum)
        {
            if(tsiNum == 1)
                return 5;
            if(tsiNum == 0 || tsiNum == 2 || tsiNum == 4 || tsiNum == 9)
                return 10;
            if(tsiNum == 3 || tsiNum == 5 || tsiNum == 6 || tsiNum == 7 || tsiNum == 8 || tsiNum == 10)
                return 15;

            return 0;
        }
        /*******************************************************************************
        * VerifyOutputReportType()
        *
        * Counts the HID packets received over 500 ms
        * Prints the average, max, and min packet interval times
        *******************************************************************************/
        void VerifyOutputReportType(const nn::bluetooth::HidReport* pReport)
        {
            const int MS_500 = 500;       // 500 Milliseconds

            static int packetCount = 0;   // number of hid events
            static nn::TimeSpan end = 0;
            static nn::TimeSpan max = 0;
            static nn::TimeSpan min = nn::TimeSpan::FromMilliSeconds(MS_500);
            static nn::TimeSpan previous;
            static bool firstRun = false;

            nn::TimeSpan current, delta;

            current = nn::os::GetSystemTick().ToTimeSpan();
            delta = current - previous;
            previous = current;

            if(end == 0)
                end = current + nn::TimeSpan::FromMilliSeconds(MS_500);

            packetCount++;

            // Need to skip first calculation to get a valid max interval
            if(firstRun == true)
            {
                if (delta > max)
                    max = delta;

                if (delta < min)
                    min = delta;
            }
            else
                firstRun = true;

            // Calculate and print packet interval based on packets counted over 500 ms
            if(current >= end)
            {
                int packetInterval = MS_500 / packetCount;

                NN_LOG("[Report Type: 0x%02X] [Average Packet Interval: %dms] [Max Packet Interval: %dms] [Min Packet Interval: %dms]\n",
                        pReport->reportData.data[0], packetInterval, max.GetMilliSeconds(), min.GetMilliSeconds());

                // Warn if the report ID is wrong
                if(pReport->reportData.data[0] != outputReport)
                {
                    NN_LOG("\n                    !!!!!!!!TEST FAILURE!!!!!!!!\n");
                    FAIL();
                    NN_LOG("->OUTPUT Report received:[0x%02X] does not match the OUTPUT Report that was set:[0x%02X]\n\n", pReport->reportData.data[0], outputReport);
                }

                // Warn if the report length is wrong
                if(pReport->reportData.data[0] == ReportId_Basic)
                {
                    if(pReport->reportLength != ReportIdDataLength_Basic)
                    {
                        NN_LOG("\nKnown issue: http://spdlybra.nintendo.co.jp/jira/browse/SIGLONTD-2626 \n");
                        NN_LOG("Report length received:[%d] does not match the length expected:[%d](ReportIdDataLength_Basic)\n\n", pReport->reportLength, ReportIdDataLength_Basic);
                    }
                }
                if(pReport->reportData.data[0] >= ReportId_MCU  && pReport->reportData.data[0] <= ReportId_Audio)
                {
                    if(pReport->reportLength != ReportIdDataLength_MCU)
                    {
                        NN_LOG("\nKnown issue: http://spdlybra.nintendo.co.jp/jira/browse/SIGLONTD-2626 \n");
                        NN_LOG("Report length received:[%d] does not match the length expected:[%d](ReportIdDataLength_MCU)\n\n", pReport->reportLength, ReportIdDataLength_MCU);
                    }
                }

                TestState = StateGatherHidPackets;

                // Reset statics
                packetCount = 0;
                end = 0;
                max = 0;
                firstRun = false;
                min = nn::TimeSpan::FromMilliSeconds(MS_500);
            }
        }
        /*******************************************************************************
        * GatherHidPackets()
        *
        * Counts the HID packets received over 5 seconds
        * Returns the final packet count
        *******************************************************************************/
        int GatherHidPackets(const nn::bluetooth::HidReport* pReport)
        {
            static int packetCount = 0;     // number of hid events
            static nn::TimeSpan end = 0;

            nn::TimeSpan current;

            current = nn::os::GetSystemTick().ToTimeSpan();

            if(end == 0)
            {
                end = current + nn::TimeSpan::FromSeconds(5);
                packetCount = 0;
                NN_LOG("Buttons pressed:\n");
            }

            packetCount++;

            // Display controller button press values
            PrintButtonValues(pReport);

            // After 5 second calculate packet rate
            if(current >= end)
            {
                TestState = StateDisplayResults;

                // Reset statics
                end = 0;

                return packetCount;
            }
            return 0;
        }
        /*******************************************************************************
        * PrintButtonValues()
        *
        * Prints currently pressed buttons
        *******************************************************************************/
        void PrintButtonValues(const nn::bluetooth::HidReport* pReport)
        {
            nn::util::BitPack8 ButtonByte1, ButtonByte2, ButtonByte3;
            static nn::util::BitPack8 prevButtonByte1, prevButtonByte2, prevButtonByte3;
            char buttons[23];

            ButtonByte1.SetMaskedBits(0xFF, pReport->reportData.data[3]);
            ButtonByte2.SetMaskedBits(0xFF, pReport->reportData.data[4]);
            ButtonByte3.SetMaskedBits(0xFF, pReport->reportData.data[5]);

            if((prevButtonByte3.storage != ButtonByte3.storage) || (prevButtonByte2.storage != ButtonByte2.storage)
               || (prevButtonByte1.storage != ButtonByte1.storage))
            {
                buttons[0]  = (ButtonByte1.GetBit(0))    ? 'Y' : ' ';
                buttons[1]  = (ButtonByte1.GetBit(1))    ? 'X' : ' ';
                buttons[2]  = (ButtonByte1.GetBit(2))    ? 'B' : ' ';
                buttons[3]  = (ButtonByte1.GetBit(3))    ? 'A' : ' ';
                buttons[4]  = (ButtonByte1.GetBit(4))    ? 'S' : ' ';
                buttons[5]  = (ButtonByte1.GetBit(5))    ? 's' : ' ';
                buttons[6]  = (ButtonByte1.GetBit(6))    ? 'R' : ' ';
                buttons[7]  = (ButtonByte1.GetBit(7))    ? 'r' : ' ';
                buttons[8]  = (ButtonByte2.GetBit(0))    ? '-' : ' ';
                buttons[9]  = (ButtonByte2.GetBit(1))    ? '+' : ' ';
                buttons[10]  = (ButtonByte2.GetBit(2))   ? 'J' : ' ';
                buttons[11]  = (ButtonByte2.GetBit(3))   ? 'j' : ' ';
                buttons[12]  = (ButtonByte2.GetBit(4))   ? 'H' : ' ';
                buttons[13]  = (ButtonByte2.GetBit(5))   ? 'C' : ' ';
                buttons[14]  = (ButtonByte3.GetBit(0))   ? 'v' : ' ';
                buttons[15]  = (ButtonByte3.GetBit(1))   ? '^' : ' ';
                buttons[16]  = (ButtonByte3.GetBit(2))   ? '>' : ' ';
                buttons[17]  = (ButtonByte3.GetBit(3))   ? '<' : ' ';
                buttons[18]  = (ButtonByte3.GetBit(4))   ? 'S' : ' ';
                buttons[19]  = (ButtonByte3.GetBit(5))   ? 's' : ' ';
                buttons[20]  = (ButtonByte3.GetBit(6))   ? 'L' : ' ';
                buttons[21]  = (ButtonByte3.GetBit(7))   ? 'l' : ' ';
                buttons[22] = '\0';

                NN_LOG("%s\n", buttons);
            }

            prevButtonByte1.storage = ButtonByte1.storage;
            prevButtonByte2.storage = ButtonByte2.storage;
            prevButtonByte3.storage = ButtonByte3.storage;
        }
}; // tsiTest2Client class

static tsiTest2Client client;
/*******************************************************************************
* nnMain()
*******************************************************************************/
extern "C" void nnMain()
{
    NN_LOG("bluetoothTsiTest2_Client NOW RUNNING \n");

    client.startBluetooth();
    nn::bluetooth::ExtSetVisibility(false,true);

    srand(time(NULL));

    int loopTimeMs = 0;

    for(;;)
    {
        if(loopTimeMs == 0 && client.deviceConnected == 0)
        {
            //Discovery
            NN_LOG("  [StartDiscovery]\n");
            nn::bluetooth::StartDiscovery();
        }

        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100));

        loopTimeMs += 100;
        if(loopTimeMs >= 30000)
        {
            loopTimeMs = 0;
        }
    }
}



