﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/bluetooth_Client.h"
#include <nn/util/util_BitPack.h>
#include <nnt/nnt_Argument.h>
#include <nnt/nntest.h>
#include <nn/nn_Log.h>

using namespace nn::bluetooth;

// TSI Test 1 specific states
typedef enum
{
    StateWaitOnSetTsiCallback,
    StateWaitOnInputReport,
    StateVerifyOutputReportType,
    StateGatherHidPackets,
    StateDisplayResults,
    StateDisconnect
} TestStateType;

class tsiTest1Client : public bluetoothClient
{
    public:
        int deviceConnected = 0;
        TestStateType TestState;
        nn::bluetooth::BluetoothAddress btTestAddr;

        /*******************************************************************************
        * GetUserInputValues()
        *
        * Parse the input args for valid TSI and OUTPUT report values
        *******************************************************************************/
        void GetUserInputValues()
        {
            int argc = nnt::GetHostArgc();
            char** argv = nnt::GetHostArgv();

            if(argc == 3)
            {
                tsiMode = convertArgvToInt(argv[1]);
                outputReport = convertArgvToInt(argv[2]);

                if((tsiMode > 10 || tsiMode < 0) || (outputReport != ReportId_Generic
                   && (outputReport < ReportId_Basic || outputReport > ReportId_Audio)))
                {
                    NN_LOG("Invalid TSI:[%d] or OUTPUT report:[0x%02x] value entered!\n", tsiMode, outputReport);
                    NN_LOG("Values reverted to default TSI:[0], OUTPUT report:[0x30]\n");
                    tsiMode = 0;
                    outputReport = 0x01;
                }
            }
            else if(argc != 1)
                NN_LOG("Invalid arguement length!\n");

            NN_LOG("Beginning test with values: TSI:[%d], OUTPUT report:[0x%02x]\n", tsiMode, outputReport);
        }
        /*******************************************************************************
        * EventFromConnectionStateCallback()
        *******************************************************************************/
        void EventFromConnectionStateCallback(const nn::bluetooth::InfoFromConnectionStateCallback* pInfo)
        {
            const char* connStrP;
            switch(pInfo->state)
            {
                case BTHH_CONN_STATE_CONNECTED:               connStrP = "BTHH_CONN_STATE_CONNECTED"; break;
                case BTHH_CONN_STATE_CONNECTING:              connStrP = "BTHH_CONN_STATE_CONNECTING"; break;
                case BTHH_CONN_STATE_DISCONNECTED:            connStrP = "BTHH_CONN_STATE_DISCONNECTED"; break;
                case BTHH_CONN_STATE_DISCONNECTING:           connStrP = "BTHH_CONN_STATE_DISCONNECTING"; break;
                case BTHH_CONN_STATE_FAILED_MOUSE_FROM_HOST:  connStrP = "BTHH_CONN_STATE_FAILED_MOUSE_FROM_HOST"; break;
                case BTHH_CONN_STATE_FAILED_KBD_FROM_HOST:    connStrP = "BTHH_CONN_STATE_FAILED_KBD_FROM_HOST"; break;
                case BTHH_CONN_STATE_FAILED_TOO_MANY_DEVICES: connStrP = "BTHH_CONN_STATE_FAILED_TOO_MANY_DEVICES"; break;
                case BTHH_CONN_STATE_FAILED_NO_BTHID_DRIVER:  connStrP = "BTHH_CONN_STATE_FAILED_NO_BTHID_DRIVER"; break;
                case BTHH_CONN_STATE_FAILED_GENERIC:          connStrP = "BTHH_CONN_STATE_FAILED_GENERIC"; break;
                case BTHH_CONN_STATE_UNKNOWN:                 connStrP = "BTHH_CONN_STATE_UNKNOWN"; break;
                default:
                    NN_LOG("Invalid ConnectionState: 0x%X\n", pInfo->state);
                    connStrP = "UNKNOWN";
            }
            NN_LOG("ConnectionState(HID): %s %s\n", connStrP, toHexString(pInfo->bluetoothAddress));

            if(pInfo->state==BTHH_CONN_STATE_CONNECTED)
            {
                // Store the address of the connected device for disconnecting in nnMain()
                btTestAddr = pInfo->bluetoothAddress;

                // Stop discovery and other devices from connecting
                deviceConnected++;
                //NN_LOG("\n***********Controller connected************\n");

                // Connected from HID view.
                // Change the Tsi mode.
                NN_LOG("Set TSI: 0x%x\n", tsiMode);
                nn::bluetooth::ExtSetTsi(&pInfo->bluetoothAddress, tsiMode);

                TestState = StateWaitOnSetTsiCallback;
            }
            if(pInfo->state==BTHH_CONN_STATE_DISCONNECTED)
            {
                //NN_LOG("**********Controller disconnected**********\n\n");
            }
        }
        /*******************************************************************************
        * EventFromExtensionCallbacks()
        *******************************************************************************/
        void EventFromExtensionCallbacks(const nn::bluetooth::InfoFromExtensionCallbacks* pInfo)
        {
            const char* extStrP;
            switch (pInfo->eventType)
            {
                case nn::bluetooth::EventFromTsiSetCallback:    extStrP = "EventFromTsiSetCallback"; break;
                case nn::bluetooth::EventFromTsiExitCallback:   extStrP = "EventFromTsiExitCallback"; break;
                case nn::bluetooth::EventFromBurstSetCallback:  extStrP = "EventFromBurstSetCallback"; break;
                case nn::bluetooth::EventFromBurstExitCallback: extStrP = "EventFromBurstExitCallback"; break;
                default:
                    NN_LOG("Invalid Extension Event: 0x%X\n", pInfo->eventType);
                    extStrP = "UNKNOWN";
            }
            NN_LOG("Extension(HID): %s, %d\n", extStrP, pInfo->eventType);

            if(pInfo->eventType==nn::bluetooth::EventFromTsiSetCallback)
            {
                // TSI set, now set the output report type
                nn::bluetooth::HidData hidData;
                memset(&hidData, 0, sizeof(hidData));
                hidData.length = 49;
                hidData.data[0] = 0x01;
                hidData.data[10] = 0x03;
                hidData.data[11] = outputReport;
                NN_LOG("HID sending output report:[0x%02x] to device:[%s]\n", outputReport, toHexString(pInfo->bluetoothAddress));
                nn::bluetooth::HidSendData(&pInfo->bluetoothAddress, &hidData);

                TestState = StateWaitOnInputReport;
            }
        }
        /*******************************************************************************
        * EventFromGetReportDataCallback()
        *******************************************************************************/
        void EventFromGetReportDataCallback(const nn::bluetooth::InfoFromGetReportCallback* pInfo)
        {
            const nn::bluetooth::HidReport* pReport = &pInfo->rptData; // NOLINT

            int packetCount;

            if(TestState == StateWaitOnInputReport)
                TestState = StateVerifyOutputReportType;

            else if(TestState == StateVerifyOutputReportType)
                VerifyOutputReportType(pReport);

            else if(TestState == StateGatherHidPackets)
                packetCount = GatherHidPackets(pReport);

            if(TestState == StateDisplayResults)
            {
                static int testCycle = 0;
                int expected, lost;
                int tsiInterval = getTsiIntervalMs(tsiMode);

                expected = 5000 / tsiInterval;
                lost = expected - packetCount;
                testCycle++;

                NN_LOG("[Packets received over 5 seconds: %d] [Expected: %d] [Estimated loss: %d]\n", packetCount, expected, lost);
                NN_LOG("\n***********Completed Test Cycle [%d]************\n\n", testCycle);
                NN_LOG("Disconnect the controller via reset button or moving out of range, then reconnect to start the test over\n");
                NN_LOG("Controller will automatically disconnect in 30 seconds\n");

                TestState = StateDisconnect;
            }
        }

    private:
        // Default TSI and OUTPUT report
        int tsiMode = 0;
        int outputReport = ReportId_Basic;

        /*******************************************************************************
        * convertArgvToInt()
        *
        * Converts argv char string to int
        *******************************************************************************/
        int convertArgvToInt(char *argv)
        {
            int num = 0;

            for (int i = 0; argv[i] != '\0'; ++i)
            {
                num *= 10;
                num += argv[i] - '0';
            }
            return num;
        }
        /*******************************************************************************
        * getTsiIntervalMs()
        *
        * Returns the interval time for the inputed TSI number
        *******************************************************************************/
        int getTsiIntervalMs(int tsiNum)
        {
            if(tsiNum == 1)
                return 5;
            if(tsiNum == 0 || tsiNum == 2 || tsiNum == 4 || tsiNum == 9)
                return 10;
            if(tsiNum == 3 || tsiNum == 5 || tsiNum == 6 || tsiNum == 7 || tsiNum == 8 || tsiNum == 10)
                return 15;

            return 0;
        }
        /*******************************************************************************
        * VerifyOutputReportType()
        *
        * Counts the HID packets received over 500 ms
        * Prints the average, max, and min packet interval times
        *******************************************************************************/
        void VerifyOutputReportType(const nn::bluetooth::HidReport* pReport)
        {
            const int MS_500 = 500;       // 500 Milliseconds

            static int packetCount = 0;   // number of hid events
            static nn::TimeSpan end = 0;
            static nn::TimeSpan max = 0;
            static nn::TimeSpan min = nn::TimeSpan::FromMilliSeconds(MS_500);
            static nn::TimeSpan previous;
            static bool firstRun = false;

            nn::TimeSpan current, delta;

            current = nn::os::GetSystemTick().ToTimeSpan();
            delta = current - previous;
            previous = current;

            if(end == 0)
                end = current + nn::TimeSpan::FromMilliSeconds(MS_500);

            packetCount++;

            // Need to skip first calculation to get a valid max interval
            if(firstRun == true)
            {
                if (delta > max)
                    max = delta;

                if (delta < min)
                    min = delta;
            }
            else
                firstRun = true;

            // Calculate and print packet interval based on packets counted over 500 ms
            if(current >= end)
            {
                int packetInterval = MS_500 / packetCount;

                NN_LOG("[Report Type: 0x%02X] [Average Packet Interval: %dms] [Max Packet Interval: %dms] [Min Packet Interval: %dms]\n",
                        pReport->reportData.data[0], packetInterval, max.GetMilliSeconds(), min.GetMilliSeconds());

                TestState = StateGatherHidPackets;

                // Reset statics
                packetCount = 0;
                end = 0;
                max = 0;
                firstRun = false;
                min = nn::TimeSpan::FromMilliSeconds(MS_500);
            }
        }
        /*******************************************************************************
        * GatherHidPackets()
        *
        * Counts the HID packets received over 5 seconds
        * Returns the final packet count
        *******************************************************************************/
        int GatherHidPackets(const nn::bluetooth::HidReport* pReport)
        {
            static int packetCount = 0;     // number of hid events
            static nn::TimeSpan end = 0;

            nn::TimeSpan current;

            current = nn::os::GetSystemTick().ToTimeSpan();

            if(end == 0)
            {
                end = current + nn::TimeSpan::FromSeconds(5);
                packetCount = 0;
                NN_LOG("Buttons pressed:\n");
            }

            packetCount++;

            // Display controller button press values
            PrintButtonValues(pReport);

            // After 5 second calculate packet rate
            if(current >= end)
            {
                TestState = StateDisplayResults;

                // Reset statics
                end = 0;

                return packetCount;
            }
            return 0;
        }
        /*******************************************************************************
        * PrintButtonValues()
        *
        * Prints currently pressed buttons
        *******************************************************************************/
        void PrintButtonValues(const nn::bluetooth::HidReport* pReport)
        {
            nn::util::BitPack8 ButtonByte1, ButtonByte2, ButtonByte3;
            static nn::util::BitPack8 prevButtonByte1, prevButtonByte2, prevButtonByte3;
            char buttons[23];

            ButtonByte1.SetMaskedBits(0xFF, pReport->reportData.data[3]);
            ButtonByte2.SetMaskedBits(0xFF, pReport->reportData.data[4]);
            ButtonByte3.SetMaskedBits(0xFF, pReport->reportData.data[5]);

            if((prevButtonByte3.storage != ButtonByte3.storage) || (prevButtonByte2.storage != ButtonByte2.storage)
               || (prevButtonByte1.storage != ButtonByte1.storage))
            {
                buttons[0]  = (ButtonByte1.GetBit(0))    ? 'Y' : ' ';
                buttons[1]  = (ButtonByte1.GetBit(1))    ? 'X' : ' ';
                buttons[2]  = (ButtonByte1.GetBit(2))    ? 'B' : ' ';
                buttons[3]  = (ButtonByte1.GetBit(3))    ? 'A' : ' ';
                buttons[4]  = (ButtonByte1.GetBit(4))    ? 'S' : ' ';
                buttons[5]  = (ButtonByte1.GetBit(5))    ? 's' : ' ';
                buttons[6]  = (ButtonByte1.GetBit(6))    ? 'R' : ' ';
                buttons[7]  = (ButtonByte1.GetBit(7))    ? 'r' : ' ';
                buttons[8]  = (ButtonByte2.GetBit(0))    ? '-' : ' ';
                buttons[9]  = (ButtonByte2.GetBit(1))    ? '+' : ' ';
                buttons[10]  = (ButtonByte2.GetBit(2))   ? 'J' : ' ';
                buttons[11]  = (ButtonByte2.GetBit(3))   ? 'j' : ' ';
                buttons[12]  = (ButtonByte2.GetBit(4))   ? 'H' : ' ';
                buttons[13]  = (ButtonByte2.GetBit(5))   ? 'C' : ' ';
                buttons[14]  = (ButtonByte3.GetBit(0))   ? 'v' : ' ';
                buttons[15]  = (ButtonByte3.GetBit(1))   ? '^' : ' ';
                buttons[16]  = (ButtonByte3.GetBit(2))   ? '>' : ' ';
                buttons[17]  = (ButtonByte3.GetBit(3))   ? '<' : ' ';
                buttons[18]  = (ButtonByte3.GetBit(4))   ? 'S' : ' ';
                buttons[19]  = (ButtonByte3.GetBit(5))   ? 's' : ' ';
                buttons[20]  = (ButtonByte3.GetBit(6))   ? 'L' : ' ';
                buttons[21]  = (ButtonByte3.GetBit(7))   ? 'l' : ' ';
                buttons[22] = '\0';

                NN_LOG("%s\n", buttons);
            }

            prevButtonByte1.storage = ButtonByte1.storage;
            prevButtonByte2.storage = ButtonByte2.storage;
            prevButtonByte3.storage = ButtonByte3.storage;
        }
}; // tsiTest1Client class

static tsiTest1Client client;
/*******************************************************************************
* nnMain()
*******************************************************************************/
extern "C" void nnMain()
{
    NN_LOG("bluetoothTsiTest1_Client NOW RUNNING \n");

    client.GetUserInputValues();
    client.startBluetooth();
    nn::bluetooth::ExtSetVisibility(false,true);

    int loopTimeMs = 0;
    int disconnectTimeMs = 0;

    for(;;)
    {
        if(loopTimeMs == 0 && client.deviceConnected == 0)
        {
            //Discovery
            NN_LOG("  [StartDiscovery]\n");
            nn::bluetooth::StartDiscovery();
        }

        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100));

        // Wait 30 seconds to disconnect controller
        loopTimeMs += 100;
        if(loopTimeMs >= 30000)
        {
            loopTimeMs = 0;
        }

        if(client.TestState == StateDisconnect)
        {
            disconnectTimeMs += 100;
            if(disconnectTimeMs >= 30000)
            {
                client.TestState = StateWaitOnSetTsiCallback;
                nn::bluetooth::HidDisconnect(&client.btTestAddr);
                disconnectTimeMs = 0;
            }
        }
    }

}



