﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>

#include <nnt/nntest.h>
#include <nn/nn_Log.h>

#include <nn/os.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/settings/system/settings_BluetoothDevices.h>
#include <nn/bluetooth/bluetooth_Api.h>
#include <nn/bluetooth/bluetooth_TypesHal.h>

#include "../../Common/testBluetooth_TargetDevices.h"
#include "../testBluetooth_Bonding.h"

using namespace nn::bluetooth;

extern nn::os::SystemEventType g_SystemEvent;
extern nn::os::SystemEventType g_SystemEventForHid;
extern nn::os::MultiWaitType g_MultiWait;
extern nn::os::MultiWaitHolderType g_HolderForBluetoothCore;
extern nn::os::MultiWaitHolderType g_HolderForHid;

void BondingTest::SetUpTestCase()
{
    NN_LOG("\n\t Setting up test case for BondingTest BT_TC_3_1\n");
    NN_LOG("-----------------------------------------------------------------------\n");
    NN_LOG("InitializeBluetoothDriver.!\n");
    nn::bluetooth::InitializeBluetoothDriver();

    nnt::bluetooth::CheckFilterOption();

    NN_LOG("InitializeBluetooth...\n");
    ASSERT_TRUE( nn::bluetooth::InitializeBluetooth(&g_SystemEvent).IsSuccess() );
    NN_LOG("EnableBluetooth...\n");
    ASSERT_TRUE( nn::bluetooth::EnableBluetooth().IsSuccess() );
    NN_LOG("InitializeHid...\n");
    ASSERT_TRUE( nn::bluetooth::InitializeHid(&g_SystemEventForHid).IsSuccess() );

    NN_LOG("InitializeMultiWaitHolder EVENTTYPE_FROM_CORE...\n");
    nn::os::InitializeMultiWaitHolder(&g_HolderForBluetoothCore, &g_SystemEvent);
    nn::os::SetMultiWaitHolderUserData(&g_HolderForBluetoothCore, BondingTestSpace::EVENTTYPE_FROM_CORE);

    NN_LOG("InitializeMultiWaitHolder EVENTTYPE_FROM_HID...\n");
    nn::os::InitializeMultiWaitHolder(&g_HolderForHid, &g_SystemEventForHid);
    nn::os::SetMultiWaitHolderUserData(&g_HolderForHid, BondingTestSpace::EVENTTYPE_FROM_HID);

    NN_LOG("InitializeMultiWait...\n");
    nn::os::InitializeMultiWait(&g_MultiWait);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_HolderForBluetoothCore);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_HolderForHid);

    NN_LOG("Leaving SetUpTestCase.\n");
}

void BondingTest::TearDownTestCase()
{
    nn::os::UnlinkAllMultiWaitHolder(&g_MultiWait);
    nn::os::FinalizeMultiWaitHolder(&g_HolderForBluetoothCore);
    nn::os::FinalizeMultiWaitHolder(&g_HolderForHid);
    nn::os::FinalizeMultiWait(&g_MultiWait);
    nn::os::DestroySystemEvent(&g_SystemEvent);
    nn::os::DestroySystemEvent(&g_SystemEventForHid);

    nn::bluetooth::DisableBluetooth();
    nn::bluetooth::CleanupBluetooth();
    nn::bluetooth::CleanupHid();
    nn::bluetooth::FinalizeBluetoothDriver();
}


TEST_F(BondingTest, BT_TC_3_1)
{
    nn::TimeSpan start;
    nn::bluetooth::BluetoothAddress foundAddr;
    nn::bluetooth::BluetoothName foundName;

    NN_LOG("\n\t Please long press reset button on a controller to begin bonding\n");
    NN_LOG("-----------------------------------------------------------------------\n");
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(3000));
    NN_LOG("Starting discovery ...\n");
    nn::bluetooth::StartDiscovery();

    BondingTestSpace::WaitDeviceFoundCallback(&foundAddr, &foundName);
    start = nn::os::GetSystemTick().ToTimeSpan();

    BluetoothBondState bondState;

    for (;;)
    {
        nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&g_MultiWait);
        int holderType = nn::os::GetMultiWaitHolderUserData(holder);
        switch (holderType)
        {
        case BondingTestSpace::EVENTTYPE_FROM_CORE:
            {
                nn::os::TryWaitSystemEvent(&g_SystemEvent);
                nn::bluetooth::EventType type;
                uint8_t buffer[nn::bluetooth::BUFFER_SIZE_OF_CORE_OUT] = {0};
                nn::bluetooth::GetEventInfo(&type, buffer, nn::bluetooth::BUFFER_SIZE_OF_CORE_OUT);
                BondingTestSpace::PrintEventInfo(type, buffer);
                if (type == nn::bluetooth::EventFromSspRequestCallback)
                {
                    nn::bluetooth::InfoFromSspRequestCallback* info = reinterpret_cast<nn::bluetooth::InfoFromSspRequestCallback*>(buffer);
                    for (int i = 0; i < sizeof(foundAddr.address) / sizeof(foundAddr.address[0]); i++)
                    {
                        EXPECT_EQ(info->bluetoothAddress.address[i], foundAddr.address[i]);
                    }
                    for (int i = 0; i < sizeof(foundName.name) / sizeof(foundName.name[0]); i++)
                    {
                        EXPECT_EQ(info->bluetoothName.name[i], foundName.name[i]);
                    }
                    nn::bluetooth::SspReply(&info->bluetoothAddress, BT_SSP_VARIANT_PASSKEY_CONFIRMATION, true, 0);
                }
                else if (type == nn::bluetooth::EventFromDiscoveryStateChangedCallback)
                {
                    BluetoothDiscoveryState state = reinterpret_cast<const nn::bluetooth::InfoFromDiscoveryStateChangedCallback*>(&buffer[0])->state;
                    if (state == BT_DISCOVERY_STOPPED)
                    {
                        nn::bluetooth::CreateBond(&foundAddr, nn::bluetooth::NoPreference);
                    }
                }
                else if (type == nn::bluetooth::EventFromDeviceFoundCallback)
                {}
                else if (type == nn::bluetooth::EventFromBondStateChangedCallback)
                {
                    bondState = reinterpret_cast<nn::bluetooth::InfoFromBondStateChangedCallback*>(buffer)->state;
                }
                else
                {
                    // Unexpected cb
                    NN_LOG("Unexpected nn::bluetooth event callback [%d].\n", type);
                    FAIL();
                    return;
                }
            }
            break;
        case BondingTestSpace::EVENTTYPE_FROM_HID:
            {
                nn::os::TryWaitSystemEvent(&g_SystemEventForHid);
                nn::bluetooth::HidEventType type;
                uint8_t buffer[nn::bluetooth::BUFFER_SIZE_OF_HID_OUT] = {0};
                nn::bluetooth::HidGetEventInfo(&type, buffer, nn::bluetooth::BUFFER_SIZE_OF_HID_OUT);
                BondingTestSpace::PrintEventInfo(type, buffer);
                if (type == nn::bluetooth::EventFromConnectionStateCallback)
                {
                    nn::bluetooth::InfoFromConnectionStateCallback* info = reinterpret_cast<nn::bluetooth::InfoFromConnectionStateCallback*>(buffer);
                    if (info->state == BTHH_CONN_STATE_CONNECTED)
                    {
                        NN_LOG("%lld ms\n", (nn::os::GetSystemTick().ToTimeSpan() - start).GetMilliSeconds());
                        EXPECT_EQ(bondState, BT_BOND_STATE_BONDED);
                         for (int i = 0; i < sizeof(foundAddr.address) / sizeof(foundAddr.address[0]); i++)
                         {
                             EXPECT_EQ(info->bluetoothAddress.address[i], foundAddr.address[i]);
                         }
                        nn::bluetooth::HidDisconnect(&foundAddr);
                        NN_LOG("\n\tSuccessfully connected to a controller! Test exiting now...\n");
                        NN_LOG("-----------------------------------------------------------------------\n");
                        SUCCEED();
                        goto OUT_OF_LOOP_1;
                    }
                    else
                    {
                        for (int i = 0; i < sizeof(foundAddr.address) / sizeof(foundAddr.address[0]); i++)
                        {
                            EXPECT_EQ(info->bluetoothAddress.address[i], foundAddr.address[i]);
                        }
                        goto OUT_OF_LOOP_1;
                    }
                }
                else if (type == nn::bluetooth::EventFromGetReportCallback)
                {
                    nn::bluetooth::HidDisconnect(&foundAddr);
                }
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }


OUT_OF_LOOP_1:
    nn::bluetooth::RemoveBond(&foundAddr);
} // NOLINT(impl/function_size)

