﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testBcat_Common.h"

void Sync() NN_NOEXCEPT
{
    NN_LOG("Sync ...\n");

    nn::bcat::DeliveryCacheProgress progress;

    nn::Result result = nn::bcat::RequestSyncDeliveryCache(&progress);

    if (result.IsSuccess())
    {
    }
    else if (nn::bcat::ResultNotFound::Includes(result))
    {
        NN_LOG("Please write BCAT-settings in nmeta.\n");
        return;
    }
    else
    {
        NN_LOG("nn::bcat::RequestSyncDeliveryCache failed. result = %03d-%04d\n", result.GetModule(), result.GetDescription());
        return;
    }

    while (progress.GetStatus() != nn::bcat::DeliveryCacheProgressStatus_Done)
    {
        progress.Update();

        switch (progress.GetStatus())
        {
        case nn::bcat::DeliveryCacheProgressStatus_Queued:
            NN_LOG("Progress: status = queued\n");
            break;
        case nn::bcat::DeliveryCacheProgressStatus_Connect:
            NN_LOG("Progress: status = connect\n");
            break;
        case nn::bcat::DeliveryCacheProgressStatus_ProcessList:
            NN_LOG("Progress: status = process list\n");
            break;
        case nn::bcat::DeliveryCacheProgressStatus_Download:
            NN_LOG("Progress: status = download, current = (%s/%s, %lld/%lld), whole = (%lld/%lld)\n",
                progress.GetCurrentDirectoryName().value, progress.GetCurrentFileName().value,
                progress.GetCurrentDownloaded(), progress.GetCurrentTotal(),
                progress.GetWholeDownloaded(), progress.GetWholeTotal());
            break;
        case nn::bcat::DeliveryCacheProgressStatus_Commit:
            NN_LOG("Progress: status = commit, current = (%s), whole = (%lld/%lld)\n",
                progress.GetCurrentDirectoryName().value,
                progress.GetWholeDownloaded(), progress.GetWholeTotal());
            break;
        case nn::bcat::DeliveryCacheProgressStatus_Done:
            NN_LOG("Progress: status = done, whole = (%lld/%lld), result = %03d-%04d\n",
                progress.GetWholeDownloaded(), progress.GetWholeTotal(),
                progress.GetResult().GetModule(), progress.GetResult().GetDescription());
            break;
        default:
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
    }

    NN_LOG("Done!\n");
}

void EnumerateAll() NN_NOEXCEPT
{
    static nn::bcat::DirectoryName s_DirNames[nn::bcat::DeliveryCacheDirectoryCountMax] = {};
    int dirCount = 0;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::bcat::EnumerateDeliveryCacheDirectory(&dirCount, s_DirNames, NN_ARRAY_SIZE(s_DirNames)));

    NN_LOG("----------------------------------------------------------------------------------------------------\n");
    NN_LOG("DirectoryCount = %d\n", dirCount);

    for (int d = 0; d < dirCount; d++)
    {
        NN_LOG("----------------------------------------------------------------------------------------------------\n");
        NN_LOG("Directory[%3d] = %s\n", d, s_DirNames[d].value);

        nn::bcat::DeliveryCacheDirectory directory;

        NN_ABORT_UNLESS_RESULT_SUCCESS(directory.Open(s_DirNames[d]));

        static nn::bcat::DeliveryCacheDirectoryEntry s_Entries[nn::bcat::DeliveryCacheFileCountMaxPerDirectory] = {};
        int entryCount = 0;

        NN_ABORT_UNLESS_RESULT_SUCCESS(directory.Read(&entryCount, s_Entries, NN_ARRAY_SIZE(s_Entries)));

        NN_LOG("    EntryCount = %d\n", entryCount);

        for (int e = 0; e < entryCount; e++)
        {
            NN_LOG("    Entry[%3d] = %32s, %016llx%016llx, %lld\n", e,
                s_Entries[e].name.value, s_Entries[e].digest.value[0], s_Entries[e].digest.value[1], s_Entries[e].size);

            nn::bcat::DeliveryCacheFile file;

            NN_ABORT_UNLESS_RESULT_SUCCESS(file.Open(s_DirNames[d], s_Entries[e].name));

            nn::Bit8 data[8] = {};
            size_t read;

            NN_ABORT_UNLESS_RESULT_SUCCESS(file.Read(&read, 0, data, NN_ARRAY_SIZE(data)));

            NN_LOG("               = ");

            for (int r = 0; r < static_cast<int>(read); r++)
            {
                char c = static_cast<char>(data[r]);

                NN_LOG("%c(%02X) ", (c >= 0x20 && c <= 0x7E) ? c : '?', data[r]);
            }

            NN_LOG("\n");
        }
    }

    NN_LOG("----------------------------------------------------------------------------------------------------\n");
}

TEST(Application, Do)
{
    nn::bcat::Initialize();
    nn::nifm::Initialize();

    NN_LOG("nn::nifm::SubmitNetworkRequestAndWait ...\n");

    nn::nifm::SubmitNetworkRequestAndWait();

    NN_LOG("nn::nifm::SubmitNetworkRequestAndWait done!\n");

    NN_LOG("nn::nifm::IsNetworkAvailable = %s\n", nn::nifm::IsNetworkAvailable() ? "true" : "false");

    if (nn::nifm::IsNetworkAvailable())
    {
        Sync();
    }

    nn::Result result = nn::bcat::MountDeliveryCacheStorage();

    if (result.IsSuccess())
    {
        NN_UTIL_SCOPE_EXIT
        {
            nn::bcat::UnmountDeliveryCacheStorage();
        };

        EnumerateAll();
    }
    else if (nn::bcat::ResultNintendoAccountNotLinked::Includes(result))
    {
        NN_LOG("Please link Nintendo account or register Network service account.\n");
    }
    else if (nn::bcat::ResultApplicationUpdateRequired::Includes(result))
    {
        NN_LOG("Please update application.\n");
    }
    else
    {
        NN_LOG("nn::bcat::MountDeliveryCacheStorage failed. result = %03d-%04d\n", result.GetModule(), result.GetDescription());
    }
}
