﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "AudioOut.h"

#include <algorithm>
#include <cmath>
#include <new>


#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include <nn/audio.h>

#include "../../../Samples/Include/nns/audio/audio_WavFormat.h"

namespace nnt { namespace applet { namespace audioout {

namespace {
    // choose number of files to play
    const int BgmCount = 2;
    const int SeCount = 4;
    const int SineCount = 36;

    // - and add / remove them to / from the files lists
    const char* g_BgmFileNames[BgmCount] =
    {
        "asset:/Audio/SampleBgm0-2ch.wav",
        "asset:/Audio/kart_title.32.wav",
    };

    const char* g_SeFileNames[SeCount] =
    {
        "asset:/Audio/SampleSe0.adpcm",
        "asset:/Audio/SampleSe1.adpcm",
        "asset:/Audio/SampleSe2.adpcm",
        "asset:/Audio/SampleSe3.adpcm",
    };

    const int g_SineMarioFrequencies[6] =
    {
        523, 659, 784,
        1046, 1319, 1568
    };

    const int g_SineKartFrequencies[6] =
    {
        587, 740, 880,
        1174, 1480, 1760
    };

    NN_ALIGNAS(4096) char g_WorkBuffer[8 * 1024 * 1024];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[35 * 1024 * 1024];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_EffectBufferPoolMemory[12 * 1024 * 1024];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_CircularBufferPoolMemory[12 * 1024 * 1024];

    nn::mem::StandardAllocator g_Allocator;
    nn::mem::StandardAllocator g_WaveBufferAllocator;
    nn::mem::StandardAllocator g_EffectBufferAllocator;
    nn::mem::StandardAllocator g_CircularBufferAllocator;

    char* g_MountRomCacheBuffer = NULL;

    void* g_pRendererWorkBuffer = NULL;
    void* g_pConfigBuffer = NULL;
    void* g_pDelayBuffer = NULL;
    void* g_pDataBgm = NULL;
    void* g_pDataSe[SeCount];
    void* g_pDataSineMario[SineCount];
    void* g_pDataSineKart[SineCount];

    nn::audio::AudioOut g_AudioOut;
    nn::audio::AudioRendererHandle g_AudioRendererHandle;
    nn::audio::AudioRendererConfig g_AudioRendererConfig;
    nn::audio::CircularBufferSinkType g_CircularBufferSink;
    nn::audio::FinalMixType g_FinalMix;
    int8_t g_MainBus[2];
    nn::audio::VoiceType g_VoiceBgm;
    nn::audio::VoiceType g_VoiceSe[SeCount];
    nn::audio::VoiceType g_VoiceSineMario[SineCount];
    nn::audio::VoiceType g_VoiceSineKart[SineCount];
    nn::audio::WaveBuffer g_WaveBufferSe[SeCount];
    nn::audio::WaveBuffer g_WaveBufferSineMario[SineCount];
    nn::audio::WaveBuffer g_WaveBufferSineKart[SineCount];
    nn::audio::AdpcmHeaderInfo* g_pSeHeader[SeCount];
    bool g_IsBgmLoaded = false;
    Sine g_Sine = Sine_Disabled;

    const auto BufferCount = 2;
    nn::audio::AudioOutBuffer g_AudioOutBuffer[BufferCount];
    int16_t* g_pResultData = nullptr;
    int g_ResultDataSize;

    nn::os::SystemEvent g_SystemEventForAudioRenderer;
    nn::os::SystemEvent g_SystemEventForAudioOut;
}

template<typename T>
size_t ConvertToInterlevedFormat2ch(T* pDestBuffer, size_t destBufferSize, const T* pSrcBuffer, size_t srcBufferSize, int blockSampleCount) NN_NOEXCEPT
{
    const auto channelCount = 2;
    const auto copySize = std::min(destBufferSize, srcBufferSize);
    const auto sampleCount = copySize / sizeof(T) / channelCount;
    const auto blockCount = sampleCount / blockSampleCount;

    auto blockBuffer0 = &pSrcBuffer[blockSampleCount * 0];
    auto blockBuffer1 = &pSrcBuffer[blockSampleCount * 1];
    const auto strideBlockSampleCount = blockSampleCount * (channelCount - 1);

    for (size_t i = 0; i < blockCount; ++i)
    {
        for (int j = 0; j < blockSampleCount; ++j)
        {
            *pDestBuffer++ = *blockBuffer0++;
            *pDestBuffer++ = *blockBuffer1++;
        }

        blockBuffer0 += strideBlockSampleCount;
        blockBuffer1 += strideBlockSampleCount;
    }

    return copySize;
}

std::size_t ReadAdpcmFile(nn::audio::AdpcmHeaderInfo* header, void** adpcmData, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t size;
    uint8_t adpcmheader[nn::audio::AdpcmHeaderSize];

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    *adpcmData = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size) - sizeof(adpcmheader), nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*adpcmData);

    result = nn::fs::ReadFile(handle, 0, adpcmheader, sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    result = nn::fs::ReadFile(handle, sizeof(adpcmheader), *adpcmData, static_cast<size_t>(size) - sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(handle);

    nn::audio::ParseAdpcmHeader(header, adpcmheader, sizeof(adpcmheader));

    return static_cast<std::size_t>(size) - sizeof(adpcmheader);
}

std::size_t ReadWavFile(nns::audio::WavFormat* format, void** data, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t size;

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    *data = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size), nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*data);

    // DATA チャンクを読む必要がありますが、ここではそれが 1024 バイト以内に見つかると仮定しています
    const std::size_t WavHeaderDataSize = 1024;

    result = nn::fs::ReadFile(handle, 0, *data, WavHeaderDataSize);
    NN_ABORT_UNLESS(result.IsSuccess());

    nns::audio::WavResult wavResult = nns::audio::ParseWavFormat(format, *data, WavHeaderDataSize);
    NN_ABORT_UNLESS_EQUAL(wavResult, nns::audio::WavResult_Success);
    NN_ABORT_UNLESS_EQUAL(format->bitsPerSample, 16);  // このサンプルでは 16bit PCM を仮定しています

    result = nn::fs::ReadFile(handle, static_cast<std::size_t>(format->dataOffset), *data, static_cast<std::size_t>(format->dataSize));
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(handle);

    return static_cast<std::size_t>(format->dataSize);
}

std::size_t GenerateSineWave(void** data, int sampleRate, int frequency, int sampleCount)
{
    int16_t* p = static_cast<int16_t*>(g_WaveBufferAllocator.Allocate(sampleCount * sizeof(int16_t), nn::audio::BufferAlignSize));
    NN_ABORT_UNLESS_NOT_NULL(p);
    const float Pi = 3.1415926535897932384626433f;
    for( auto i = 0; i < sampleCount; ++i )
    {
        p[i] = static_cast<int16_t>(std::numeric_limits<int16_t>::max() * sinf(2 * Pi * frequency * i / sampleRate));
    }
    *data = p;
    return sampleCount * sizeof(int16_t);
}

void InitializeFileSystem()
{
    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("asset", g_MountRomCacheBuffer, cacheSize)
    );
}

void FinalizeFileSystem()
{
    nn::fs::Unmount("asset");

    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;
}

void InitializeAudio() NN_NOEXCEPT
{
    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
    g_WaveBufferAllocator.Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    g_EffectBufferAllocator.Initialize(g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory));
    g_CircularBufferAllocator.Initialize(g_CircularBufferPoolMemory, sizeof(g_CircularBufferPoolMemory));

    InitializeFileSystem();

    // レンダリング結果を再生するためのオーディオ出力を準備します。
    const int channelCount = 2;
    nn::audio::AudioOutParameter audioOutParameter;
    nn::audio::InitializeAudioOutParameter(&audioOutParameter);
    audioOutParameter.channelCount = channelCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(&g_AudioOut, &g_SystemEventForAudioOut, audioOutParameter));
    const auto sampleRate = nn::audio::GetAudioOutSampleRate(&g_AudioOut);

    // レンダラのパラメータを指定します。
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = sampleRate;
    parameter.sampleCount = sampleRate / 200;
    parameter.mixBufferCount = 6 + 2; // FinalMix(6) + SubMix(2)
    parameter.voiceCount = 78;
    parameter.subMixCount = 2;
    parameter.sinkCount = 1;
    parameter.effectCount = 3;
    parameter.performanceFrameCount = 0;
    parameter.executionMode = nn::audio::AudioRendererExecutionMode_ManualExecution;
    parameter.renderingDevice = nn::audio::AudioRendererRenderingDevice_Cpu;

    // ミックスバッファとオーディオバスの関係を定義します。
    int8_t *pMainBus = g_MainBus;
    pMainBus[nn::audio::ChannelMapping_FrontLeft] = 4;
    pMainBus[nn::audio::ChannelMapping_FrontRight] = 5;
    int8_t auxBusA[2];
    auxBusA[nn::audio::ChannelMapping_FrontLeft] = 0;
    auxBusA[nn::audio::ChannelMapping_FrontRight] = 1;

    // パラメータがシステムでサポートされているかどうかを確認します。
    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(parameter),
        "Invalid AudioRendererParameter specified."
    );

    // レンダラのワークバッファを準備します。
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    g_pRendererWorkBuffer = g_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(g_pRendererWorkBuffer);

    // レンダラを初期化します。
    nn::audio::AudioRendererHandle *pHandle = &g_AudioRendererHandle;
    NN_ABORT_UNLESS(
        nn::audio::OpenAudioRenderer(pHandle, &g_SystemEventForAudioRenderer, parameter, g_pRendererWorkBuffer, workBufferSize).IsSuccess(),
        "Failed to open AudioRenderer"
    );

    // AudioRendererConfig を初期化します。
    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    g_pConfigBuffer = g_Allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(g_pConfigBuffer);
    nn::audio::AudioRendererConfig *pConfig = &g_AudioRendererConfig;
    nn::audio::InitializeAudioRendererConfig(pConfig, parameter, g_pConfigBuffer, configBufferSize);

    nn::audio::FinalMixType *pFinalMix = &g_FinalMix;
    nn::audio::AcquireFinalMix(pConfig, pFinalMix, 6);

    nn::audio::SubMixType subMix0;
    nn::audio::AcquireSubMix(pConfig, &subMix0, parameter.sampleRate, 1);
    nn::audio::SubMixType subMix1;
    nn::audio::AcquireSubMix(pConfig, &subMix1, parameter.sampleRate, 1);

    // オーディオレンダラのレンダリング結果を取得するために、CircularBufferSink を準備します。
    const auto bufferingFrameCount = 16;
    const auto circularBufferSinkBufferSize = nn::audio::GetRequiredBufferSizeForCircularBufferSink(&parameter, channelCount, bufferingFrameCount, nn::audio::SampleFormat_PcmInt16);
    auto circularBufferSinkBuffer = g_CircularBufferAllocator.Allocate(nn::util::align_up(circularBufferSinkBufferSize, nn::audio::MemoryPoolType::SizeGranularity), nn::audio::MemoryPoolType::AddressAlignment);
    NN_ABORT_UNLESS_NOT_NULL(circularBufferSinkBuffer);
    nn::audio::MemoryPoolType circularBufferMemoryPool;
    auto rt = nn::audio::AcquireMemoryPool(pConfig, &circularBufferMemoryPool, circularBufferSinkBuffer, nn::util::align_up(circularBufferSinkBufferSize, nn::audio::MemoryPoolType::SizeGranularity));
    NN_ABORT_UNLESS(rt);
    rt = nn::audio::RequestAttachMemoryPool(&circularBufferMemoryPool);
    NN_ABORT_UNLESS(rt);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddCircularBufferSink(pConfig, &g_CircularBufferSink, &g_FinalMix, pMainBus, channelCount, circularBufferSinkBuffer, circularBufferSinkBufferSize, nn::audio::SampleFormat_PcmInt16));

    // SubMix(0) を FinalMix へ接続します
    nn::audio::SetSubMixDestination(pConfig, &subMix0, pFinalMix);

    // SubMix(0) の 0 番目のバッファを FinalMix の mainBus[0]、mainBus[1] へ
    ///ボリューム 0.5f でミックスするように設定します
    nn::audio::SetSubMixMixVolume(&subMix0, pFinalMix, 0.5f, 0, pMainBus[0]);
    nn::audio::SetSubMixMixVolume(&subMix0, pFinalMix, 0.5f, 0, pMainBus[1]);

    // SubMix(1) を SubMix(0) へ接続します
    nn::audio::SetSubMixDestination(pConfig, &subMix1, &subMix0);

    // SubMix(1) の 0 番目のバッファを SubMix(0) の 0 番目のバッファへ
    // ボリューム 0.5f でミックスするように設定します
    nn::audio::SetSubMixMixVolume(&subMix1, &subMix0, 0.5f, 0, 0);

    // エフェクトバッファ用のメモリプールを準備します。
    // MemoryPool は nn::audio::AcquireMemoryPool() で渡される address 、 size の領域を書き換えることはありません。
    // そのため g_EffectBufferPoolMemory として準備した領域をすべて、メモリプールに利用することができます。
    // またこのサンプルでは WaveBuffer, Effect とメモリの用途別にアロケータを準備しています。
    // MemoryPool はその領域全体を「アタッチ」「デタッチ」します。
    // MemoryPool の操作に必要な粒度に応じてメモリ領域を分けておくことをおすすめします。
    // 詳しくは @link ../../../Samples/Sources/Applications/AudioMemoryPool
    // Samples/Sources/Applications/AudioMemoryPool @endlink を参照してください。
    nn::audio::MemoryPoolType effectBufferPool;
    rt = AcquireMemoryPool(pConfig, &effectBufferPool, g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory));
    NN_ABORT_UNLESS(rt);
    // メモリプールをアタッチします。設定は次の nn::audio::RequestUpdateAudioRenderer() が呼ばれた以降に反映されます。
    rt = RequestAttachMemoryPool(&effectBufferPool);
    NN_ABORT_UNLESS(rt);

    // Delay と BufferMixer を用意し、Delay, BufferMixer の順に登録します。
    nn::audio::DelayType delay;
    const nn::TimeSpan delayTimeMax = nn::TimeSpan::FromSeconds(1); // delay エフェクトで利用しうる最大のディレイ時間を指定します。
    size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(delayTimeMax, parameter.sampleRate, 2);
    // エフェクトバッファを取得します。g_EffectBufferAllocator が管理する全領域は effectBufferPool メモリプールに登録してあります。
    g_pDelayBuffer = g_EffectBufferAllocator.Allocate(delaySize, nn::audio::BufferAlignSize);
    nn::audio::BufferMixerType mixer1;

    auto result = nn::audio::AddDelay(pConfig, &delay, g_pDelayBuffer, delaySize, pFinalMix, delayTimeMax, 2);
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::audio::SetDelayInputOutput(&delay, auxBusA, auxBusA, channelCount);
    result = nn::audio::AddBufferMixer(pConfig, &mixer1, pFinalMix);
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::audio::SetBufferMixerInputOutput(&mixer1, auxBusA, pMainBus, channelCount);

    // Delay パラメータを設定します。
    nn::audio::SetDelayEnabled(&delay, true);
    nn::audio::SetDelayTime(&delay, nn::TimeSpan::FromMilliSeconds(100));
    nn::audio::SetDelayFeedbackGain(&delay, 0.7f);
    nn::audio::SetDelayChannelSpread(&delay, 0.256f);
    nn::audio::SetDelayLowPassAmount(&delay, 0.9f);
    nn::audio::SetDelayDryGain(&delay, 0.5f);
    nn::audio::SetDelayInGain(&delay, 0.4f);

    // BufferMixer パラメータを設定します。
    nn::audio::SetBufferMixerVolume(&mixer1, 0, 1.0f);
    nn::audio::SetBufferMixerVolume(&mixer1, 1, 1.0f);

    // 設定したパラメータをレンダラに反映させます。
    result = nn::audio::RequestUpdateAudioRenderer(*pHandle, pConfig);
    NN_ABORT_UNLESS(result.IsSuccess());

    // レンダリングを開始します。
    result = nn::audio::StartAudioRenderer(*pHandle);
    NN_ABORT_UNLESS(result.IsSuccess());

    // WaveBuffer に追加するサンプルデータを保持するためのメモリプールを準備します。
    nn::audio::MemoryPoolType waveBufferMemoryPool;
    rt = AcquireMemoryPool(pConfig, &waveBufferMemoryPool, g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    NN_ABORT_UNLESS(rt);
    rt = RequestAttachMemoryPool(&waveBufferMemoryPool);
    NN_ABORT_UNLESS(rt);

    // SE の準備
    nn::audio::VoiceType *pVoiceSe = g_VoiceSe;
    nn::audio::WaveBuffer *pWaveBufferSe = g_WaveBufferSe;
    nn::audio::AdpcmHeaderInfo** ppHeader = g_pSeHeader;

    for (int i = 0; i < SeCount; ++i)
    {
        ppHeader[i] = reinterpret_cast<nn::audio::AdpcmHeaderInfo*>(g_WaveBufferAllocator.Allocate(sizeof(nn::audio::AdpcmHeaderInfo), NN_ALIGNOF(nn::audio::AdpcmHeaderInfo)));
        std::size_t dataSeSize = ReadAdpcmFile(ppHeader[i], &g_pDataSe[i], g_SeFileNames[i]);
        nn::audio::AcquireVoiceSlot(pConfig, &pVoiceSe[i], ppHeader[i]->sampleRate, 1, nn::audio::SampleFormat_Adpcm, nn::audio::VoiceType::PriorityHighest, &ppHeader[i]->parameter, sizeof(nn::audio::AdpcmParameter));
        nn::audio::SetVoiceDestination(pConfig, &pVoiceSe[i], pFinalMix);

        pWaveBufferSe[i].buffer = g_pDataSe[i];
        pWaveBufferSe[i].size = dataSeSize;
        pWaveBufferSe[i].startSampleOffset = 0;
        pWaveBufferSe[i].endSampleOffset = ppHeader[i]->sampleCount;
        pWaveBufferSe[i].loop = false;
        pWaveBufferSe[i].isEndOfStream = false;
        pWaveBufferSe[i].pContext = &ppHeader[i]->loopContext;
        pWaveBufferSe[i].contextSize = sizeof(nn::audio::AdpcmContext);

        nn::audio::AppendWaveBuffer(&pVoiceSe[i], &pWaveBufferSe[i]);
        nn::audio::SetVoicePlayState(&pVoiceSe[i], nn::audio::VoiceType::PlayState_Pause);
        nn::audio::SetVoiceMixVolume(&pVoiceSe[i], pFinalMix, 0.707f / 2, 0, auxBusA[0]);
        nn::audio::SetVoiceMixVolume(&pVoiceSe[i], pFinalMix, 0.707f / 2, 0, auxBusA[1]);
    }

    const int sineSampleRate = 32000;
    const int sineSampleCount = 32000;

    // SineMario の準備
    nn::audio::VoiceType *pVoiceSineMario = g_VoiceSineMario;
    nn::audio::WaveBuffer *pWaveBufferSineMario = g_WaveBufferSineMario;

    for( int i = 0; i < SineCount; ++i )
    {
        std::size_t size = GenerateSineWave(&g_pDataSineMario[i], sineSampleRate, g_SineMarioFrequencies[i % 6], sineSampleCount);
        nn::audio::AcquireVoiceSlot(pConfig, &pVoiceSineMario[i], sineSampleRate, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
        nn::audio::SetVoiceDestination(pConfig, &pVoiceSineMario[i], pFinalMix);

        pWaveBufferSineMario[i].buffer = g_pDataSineMario[i];
        pWaveBufferSineMario[i].size = size;
        pWaveBufferSineMario[i].startSampleOffset = 0;
        pWaveBufferSineMario[i].endSampleOffset = sineSampleCount;
        pWaveBufferSineMario[i].loop = true;
        pWaveBufferSineMario[i].isEndOfStream = false;

        nn::audio::AppendWaveBuffer(&pVoiceSineMario[i], &pWaveBufferSineMario[i]);
        nn::audio::SetVoicePlayState(&pVoiceSineMario[i], nn::audio::VoiceType::PlayState_Pause);
        nn::audio::SetVoiceMixVolume(&pVoiceSineMario[i], pFinalMix, 0.707f / 72, 0, auxBusA[0]);
        nn::audio::SetVoiceMixVolume(&pVoiceSineMario[i], pFinalMix, 0.707f / 72, 0, auxBusA[1]);
    }

    // SineKart の準備
    nn::audio::VoiceType *pVoiceSineKart = g_VoiceSineKart;
    nn::audio::WaveBuffer *pWaveBufferSineKart = g_WaveBufferSineKart;

    for( int i = 0; i < SineCount; ++i )
    {
        std::size_t size = GenerateSineWave(&g_pDataSineKart[i], sineSampleRate, g_SineKartFrequencies[i % 6], sineSampleCount);
        nn::audio::AcquireVoiceSlot(pConfig, &pVoiceSineKart[i], sineSampleRate, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
        nn::audio::SetVoiceDestination(pConfig, &pVoiceSineKart[i], pFinalMix);

        pWaveBufferSineKart[i].buffer = g_pDataSineKart[i];
        pWaveBufferSineKart[i].size = size;
        pWaveBufferSineKart[i].startSampleOffset = 0;
        pWaveBufferSineKart[i].endSampleOffset = sineSampleCount;
        pWaveBufferSineKart[i].loop = true;
        pWaveBufferSineKart[i].isEndOfStream = false;

        nn::audio::AppendWaveBuffer(&pVoiceSineKart[i], &pWaveBufferSineKart[i]);
        nn::audio::SetVoicePlayState(&pVoiceSineKart[i], nn::audio::VoiceType::PlayState_Pause);
        nn::audio::SetVoiceMixVolume(&pVoiceSineKart[i], pFinalMix, 0.707f / 72, 0, auxBusA[0]);
        nn::audio::SetVoiceMixVolume(&pVoiceSineKart[i], pFinalMix, 0.707f / 72, 0, auxBusA[1]);
    }

    result = nn::audio::RequestUpdateAudioRenderer(*pHandle, pConfig);
    NN_ABORT_UNLESS(result.IsSuccess());

    // CircularBufferSink から取得したレンダリング結果を再生するためのオーディオ出力を準備します。
    g_ResultDataSize = parameter.sampleCount * channelCount * sizeof(int16_t) * bufferingFrameCount;
    const auto bufferSize = nn::util::align_up(g_ResultDataSize, nn::audio::AudioOutBuffer::SizeGranularity);
    // AudioOutBuffer を準備します。
    for (int i = 0; i < BufferCount; ++i)
    {
        // 先に空のバッファを追加しておきます。
        auto buffer = reinterpret_cast<int16_t*>(g_Allocator.Allocate(bufferSize, nn::audio::AudioOutBuffer::AddressAlignment));
        NN_ASSERT_NOT_NULL(buffer);
        memset(buffer, 0, bufferSize);

        nn::audio::SetAudioOutBufferInfo(&g_AudioOutBuffer[i], buffer, bufferSize, g_ResultDataSize);
        nn::audio::AppendAudioOutBuffer(&g_AudioOut, &g_AudioOutBuffer[i]);
    }

    // CircularBufferSink からレンダリング結果を受け取るためのバッファを準備します。
    g_pResultData = reinterpret_cast<int16_t*>(g_Allocator.Allocate(g_ResultDataSize, nn::audio::AudioOutBuffer::AddressAlignment));
    NN_ASSERT_NOT_NULL(g_pResultData);

    // オーディオ出力を開始します。
    NN_ABORT_UNLESS(nn::audio::StartAudioOut(&g_AudioOut).IsSuccess());
} // NOLINT(impl/function_size)

void FinalizeAudio() NN_NOEXCEPT
{
    // レンダリングを終了します。
    nn::audio::StopAudioRenderer(g_AudioRendererHandle);
    nn::audio::CloseAudioRenderer(g_AudioRendererHandle);

    nn::audio::StopAudioOut(&g_AudioOut);
    nn::audio::CloseAudioOut(&g_AudioOut);

    // メモリを解放します。
    if (g_pDelayBuffer)
    {
        g_EffectBufferAllocator.Free(g_pDelayBuffer);
        g_pDelayBuffer = nullptr;
    }
    if (g_pDataBgm)
    {
        g_WaveBufferAllocator.Free(g_pDataBgm);
        g_pDataBgm = nullptr;
    }
    if (g_pDataBgm)
    {
        g_WaveBufferAllocator.Free(g_pDataBgm);
        g_pDataBgm = nullptr;
    }
    for (int i = 0; i < SeCount; ++i)
    {
        if (g_pDataSe[i])
        {
            g_WaveBufferAllocator.Free(g_pDataSe[i]);
            g_pDataSe[i] = nullptr;
            g_WaveBufferAllocator.Free(g_pSeHeader[i]);
            g_pSeHeader[i] = nullptr;
        }
    }
    for (int i = 0; i < SineCount; ++i)
    {
        if (g_pDataSineMario[i])
        {
            g_WaveBufferAllocator.Free(g_pDataSineMario[i]);
            g_pDataSineMario[i] = nullptr;
        }
        if (g_pDataSineKart[i])
        {
            g_WaveBufferAllocator.Free(g_pDataSineKart[i]);
            g_pDataSineKart[i] = nullptr;
        }
    }
    if (g_pConfigBuffer)
    {
        g_Allocator.Free(g_pConfigBuffer);
        g_pConfigBuffer = nullptr;
    }
    if (g_pRendererWorkBuffer)
    {
        g_Allocator.Free(g_pRendererWorkBuffer);
        g_pRendererWorkBuffer = nullptr;
    }
    for (int i = 0; i < BufferCount; ++i)
    {
        auto buffer = nn::audio::GetAudioOutBufferDataPointer(&g_AudioOutBuffer[i]);
        g_Allocator.Free(buffer);
    }
    if (g_pResultData)
    {
        g_Allocator.Free(g_pResultData);
        g_pResultData = nullptr;
    }

    FinalizeFileSystem();
}

void WaitAudio() NN_NOEXCEPT
{
    g_SystemEventForAudioOut.Wait();
}

void UpdateAudio() NN_NOEXCEPT
{
    nn::audio::AudioRendererHandle *pHandle = &g_AudioRendererHandle;
    nn::audio::AudioRendererConfig *pConfig = &g_AudioRendererConfig;

    while(auto pReleasedBuffer = nn::audio::GetReleasedAudioOutBuffer(&g_AudioOut))
    {
        size_t currentResultDataOffset = 0;
        while(currentResultDataOffset < g_ResultDataSize)
        {
            auto result = nn::audio::RequestUpdateAudioRenderer(*pHandle, pConfig);
            NN_ABORT_UNLESS(result.IsSuccess());

            // CircularBufferSink からレンダリング結果を取得します。
            currentResultDataOffset += nn::audio::ReadCircularBufferSink(&g_CircularBufferSink,
                g_pResultData + (currentResultDataOffset / sizeof(int16_t)),
                g_ResultDataSize - currentResultDataOffset);

            // オーディオレンダリング処理を実行します。
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::ExecuteAudioRendererRendering(*pHandle));
        }
        NN_ASSERT(currentResultDataOffset == g_ResultDataSize);

        // レンダリング結果を AudioOut に流します。
        const auto releasedBufferBufferSize = nn::audio::GetAudioOutBufferBufferSize(pReleasedBuffer);
        NN_ASSERT(releasedBufferBufferSize >= g_ResultDataSize);
        int16_t* pInterleavedBuffer = reinterpret_cast<int16_t*>(nn::audio::GetAudioOutBufferDataPointer(pReleasedBuffer));

        const auto writtenSize = ConvertToInterlevedFormat2ch(pInterleavedBuffer, releasedBufferBufferSize, g_pResultData, g_ResultDataSize, 240);
        NN_ASSERT(writtenSize == g_ResultDataSize);
        nn::audio::SetAudioOutBufferInfo(pReleasedBuffer, pInterleavedBuffer, releasedBufferBufferSize, writtenSize);
        nn::audio::AppendAudioOutBuffer(&g_AudioOut, pReleasedBuffer);

        memset(g_pResultData, 0, g_ResultDataSize);
        currentResultDataOffset = 0;
    }
}

void LoadBgm( Bgm bgm ) NN_NOEXCEPT
{
    nn::audio::AudioRendererConfig *pConfig = &g_AudioRendererConfig;
    nn::audio::FinalMixType *pFinalMix = &g_FinalMix;
    int8_t *pMainBus = g_MainBus;

    nn::audio::VoiceType *pVoiceBgm = &g_VoiceBgm;
    nn::audio::WaveBuffer waveBufferBgm;

    nns::audio::WavFormat format;
    std::size_t dataSize = ReadWavFile(&format, &g_pDataBgm, g_BgmFileNames[bgm]);
    if (bgm == Bgm_A)
    {
        g_Sine = Sine_Mario;
    }
    else
    {
        g_Sine = Sine_Kart;
    }

    // 読み込んだ WAV 形式ファイルのチャンネル数を指定してボイスを取得します。
    // マルチチャンネルのデータを読み込んだ場合には、チャンネル数分のボイスを消費することに注意してください。
    nn::audio::AcquireVoiceSlot(pConfig, pVoiceBgm, format.sampleRate, format.channelCount, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
    nn::audio::SetVoiceDestination(pConfig, pVoiceBgm, pFinalMix);

    waveBufferBgm.buffer = g_pDataBgm;
    waveBufferBgm.size = dataSize;
    waveBufferBgm.startSampleOffset = 0;
    waveBufferBgm.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t)) / format.channelCount;
    waveBufferBgm.loop = true;
    waveBufferBgm.isEndOfStream = false;
    waveBufferBgm.pContext = nullptr;
    waveBufferBgm.contextSize = 0;

    nn::audio::AppendWaveBuffer(pVoiceBgm, &waveBufferBgm);
    nn::audio::SetVoicePlayState(pVoiceBgm, nn::audio::VoiceType::PlayState_Pause);
    // ボイスの 0 チャンネルを mainBus[0] へ、1 チャンネルを mainBus[1] へ出力するミックスボリュームを設定します。
    nn::audio::SetVoiceMixVolume(pVoiceBgm, pFinalMix, 0.5f, 0, pMainBus[0]);
    nn::audio::SetVoiceMixVolume(pVoiceBgm, pFinalMix, 0.5f, 1, pMainBus[1]);

    // カットオフ 2048Hz のローパスフィルタを設定します
    nn::audio::BiquadFilterParameter firstFilter = { true, {720, 1439, 720}, {22684, -9350} };
    nn::audio::SetVoiceBiquadFilterParameter(pVoiceBgm, 0, firstFilter);
    // カットオフ 1024 Hz のハイパスフィルタを無効状態で設定します。
    nn::audio::BiquadFilterParameter secondFilter = { false, {14041, -28083, 14041}, {29547, -13563} };
    nn::audio::SetVoiceBiquadFilterParameter(pVoiceBgm, 1, secondFilter);

    g_IsBgmLoaded = true;
}

void PlayBgm() NN_NOEXCEPT
{
    if ( !g_IsBgmLoaded )
    {
        return;
    }
    nn::audio::VoiceType *pVoiceBgm = &g_VoiceBgm;
    if ( nn::audio::GetVoicePlayState( pVoiceBgm ) == nn::audio::VoiceType::PlayState_Play )
    {
        return;
    }
    nn::audio::SetVoicePlayState(pVoiceBgm, nn::audio::VoiceType::PlayState_Play);
}

void PauseBgm() NN_NOEXCEPT
{
    if ( !g_IsBgmLoaded )
    {
        return;
    }
    nn::audio::VoiceType *pVoiceBgm = &g_VoiceBgm;
    if ( nn::audio::GetVoicePlayState( pVoiceBgm ) == nn::audio::VoiceType::PlayState_Pause )
    {
        return;
    }
    nn::audio::SetVoicePlayState(pVoiceBgm, nn::audio::VoiceType::PlayState_Pause);
}

void PlaySe( Se se ) NN_NOEXCEPT
{
    nn::audio::VoiceType *pVoiceSe = g_VoiceSe;
    nn::audio::WaveBuffer *pWaveBufferSe = g_WaveBufferSe;
    if ( nn::audio::GetVoicePlayState( &pVoiceSe[se] ) == nn::audio::VoiceType::PlayState_Pause )
    {
        nn::audio::SetVoicePlayState(&pVoiceSe[se], nn::audio::VoiceType::PlayState_Play);
    }
    if (nn::audio::GetReleasedWaveBuffer(&pVoiceSe[se]))
    {
        nn::audio::AppendWaveBuffer(&pVoiceSe[se], &pWaveBufferSe[se]);
    }
}

void PlaySine()
{
    if( !g_IsBgmLoaded )
    {
        g_Sine = Sine_Mario;
    }
    nn::audio::VoiceType* pVoiceSine;
    if( g_Sine == Sine_Mario )
    {
        pVoiceSine = g_VoiceSineMario;
    }
    else
    {
        pVoiceSine = g_VoiceSineKart;
    }

    for( int i = 0; i < SineCount; ++i )
    {
        if( nn::audio::GetVoicePlayState(&pVoiceSine[i]) == nn::audio::VoiceType::PlayState_Play )
        {
            return;
        }
        nn::audio::SetVoicePlayState(&pVoiceSine[i], nn::audio::VoiceType::PlayState_Play);
    }
}

void PauseSine()
{
    if( !g_IsBgmLoaded )
    {
        g_Sine = Sine_Mario;
    }
    nn::audio::VoiceType* pVoiceSine;
    if( g_Sine == Sine_Mario )
    {
        pVoiceSine = g_VoiceSineMario;
    }
    else
    {
        pVoiceSine = g_VoiceSineKart;
    }

    for( int i = 0; i < SineCount; ++i )
    {
        if( nn::audio::GetVoicePlayState(&pVoiceSine[i]) == nn::audio::VoiceType::PlayState_Pause )
        {
            return;
        }
        nn::audio::SetVoicePlayState(&pVoiceSine[i], nn::audio::VoiceType::PlayState_Pause);
    }
}

}}} // nnt::applet::audioout
