﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using CsTestAssistants;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace testAoc_Download
{
    using ExpectEvaluator = ThrowEvaluator<UnexpectFailureException>;

    /// <summary>
    /// テストクラス
    /// </summary>
    [TestClass]
    public class UnitMain
    {
        // `TestContext` プロパティを定義しておけば、UnitTestフレームワークで勝手にインスタンスが提供される.
        public TestContext TestContext { get; set; }

        private const int ReinitAfterInterval = 15000;
        private const string LogPrefix = "[AocDownloadTest]";
        private const string TestMethodLogBegin = "=== Begin the test of [ {0} ] ===";
        private const string TestMethodLogFinish = "=== Finish the test of [ {0} ] ===";
        private const string TestMethodLogFinally = "=== Finally the test of [ {0} ] ===";
        private static System.Net.WebProxy ProxyConfiguration;
        private static EncodeContext ActiveEncode = new EncodeContext();
        private static SigloHelper.Configuration.Context ActiveConfiguration = null;
        private static readonly BasicAccount CliTokenAccount = new BasicAccount( "sasaki_yu", "NDQzYjEyMTAtODc3");

        private static readonly string DefaultNintendoAccountId = "psg_aoc_test";
        private static readonly string DefaultNintendoAccountPassword = "Password01";
        private static BasicAccount NintendoAccount = null;
        private static readonly string NetworkSettingJsonFilePath = System.IO.Path.Combine( SigloHelper.Path.GetTestsRoot(), "Aoc", "Sources", "Tests", "Download", "NetworkSettingsForCI.txt" );

        /// <summary>
        /// 実行環境に応じたプロキシ設定の取得.
        /// </summary>
        /// <param name="context">テストコンテキスト</param>
        /// <returns>プロキシ設定が無い場合は null, その他は対象の proxy instance.</returns>
        private static System.Net.WebProxy PrepareProxyConfiguration( TestContext context )
        {
            try
            {
                if ( true == context.Properties.Contains( "EnableRunTestOnIDE_FromPUX" ) &&
                    true == bool.Parse( ( string )context.Properties[ "EnableRunTestOnIDE_FromPUX" ] ) )
                {
                    return D4cHelper.ProxyHelper.Create( null );
                }
            }
            catch ( System.Exception e )
            {
                Log.WriteLine( $"Critical failure of prepare proxy configuration => [ {e.Message} ]" );
            }
            return D4cHelper.ProxyHelper.Create();
        }

        public class TestMethodLog : CsTestAssistants.ScopeLog
        {
            public TestMethodLog( [System.Runtime.CompilerServices.CallerMemberName] string testName = "" )
                : base(TestMethodLogBegin, TestMethodLogFinish, TestMethodLogFinally, testName) {}
        }

        private static void SetupNintendoAccount(TestContext context)
        {
            var id = context.Properties.Contains("NintendoAccountId") ? (string)context.Properties["NintendoAccountId"] : DefaultNintendoAccountId;
            var password = context.Properties.Contains("NintendoAccountPassword") ? (string)context.Properties["NintendoAccountPassword"] : DefaultNintendoAccountPassword;
            NintendoAccount = new BasicAccount(id, password);
        }

        /// <summary>
        /// クラスが生成され、呼び出される前に一度だけ実施。
        /// </summary>
        /// <param name="context"></param>
        [ClassInitialize]
        public static void TestClassInitialize( TestContext context )
        {
            ActiveEncode.Apply( EncodeContext.UTF8 );
            Log.Verbose = true;
            Log.Prefix = LogPrefix;

            ProxyConfiguration = PrepareProxyConfiguration( context );

            string targetName = ( string )context.Properties[ "TargetName" ];
            string platform = ( context.Properties.Contains( "Platform" ) ) ? ( string )context.Properties[ "Platform" ] : SigloHelper.Configuration.DefaultPlatform;
            string buildType = ( context.Properties.Contains( "BuildType" ) ) ? ( string )context.Properties[ "BuildType" ] : SigloHelper.Configuration.DefaultBuildType;
            ActiveConfiguration = new SigloHelper.Configuration.Context( targetName, platform, buildType );

            SetupNintendoAccount(context);

            Log.WriteLine( "=== Test Context ===" );
            Log.WriteLine( "TestDir                 : {0}", context.TestDir );
            Log.WriteLine( "TestLogsDir             : {0}", context.TestLogsDir );
            Log.WriteLine( "TestRunDirectory        : {0}", context.TestRunDirectory );
            Log.WriteLine( "TestResultsDirectory    : {0}", context.TestResultsDirectory );
            Log.WriteLine( "TestRunResultsDirectory : {0}", context.TestRunResultsDirectory );

            Log.WriteLine( "=== Configuration ===" );
            Log.WriteLine( "TargetName      : {0}", ActiveConfiguration.TargetName );
            Log.WriteLine( "TargetPlatform  : {0}", ActiveConfiguration.TargetPlatform );
            Log.WriteLine( "TargetBuildType : {0}", ActiveConfiguration.TargetBuildType );
            Log.WriteLine( "HostWebProxy    : {0}", ( null != ProxyConfiguration ) ? ProxyConfiguration.Address.ToString() : "None" );
        }

        /// <summary>
        /// クラスが破棄されるタイミングで一度だけ実施。
        /// </summary>
        [ClassCleanup]
        public static void TestClassCleanup()
        {
            ActiveEncode.Restore();
        }

        /// <summary>
        /// `TestProperty属性の "JIRA" キーの値` と `Sigloコンフィギュレーションコンテキスト` に依存した作業用ワーキングディレクトリパスを生成します.
        /// </summary>
        /// <param name="context">Sigloコンフィギュレーションコンテキスト</param>
        /// <param name="methodName">参照対象TestProperty属性を保持するメソッド名</param>
        /// <returns></returns>
        private string GenerateIntermediateDirectoryAsMethod( SigloHelper.Configuration.Context context,
            [System.Runtime.CompilerServices.CallerMemberName] string methodName = null )
        {
            // ベース作業用ディレクトリ
            string outPath = context.GetOutputTestDirectory( "testAoc_Download" );
            if ( false == string.IsNullOrEmpty( methodName ) )
            {
                string ticket = this.GetTestPropertyValue( methodName, "JIRA" );
                if ( !string.IsNullOrEmpty( ticket ) )
                {
                    outPath = System.IO.Path.Combine( outPath, ticket );
                }
            }
            Log.WriteLine( "intermediate directory : {0}", outPath );
            return outPath;
        }

        /// <summary>
        /// SDEV再初期化のためにネットワーク設定シーケンスをメソッド化
        /// </summary>
        /// <param name="executor"></param>
        private void SetupSdevEnvironments( SigloHelper.CommodityExecutor.Context executor )
        {
            // SettingsManager import `network`
            Assert.IsTrue( executor.RunSettingsManager( "Import \"" + NetworkSettingJsonFilePath + "\"" ) );
            // reset sdev
            Assert.IsTrue( executor.ResetTarget() );
            // サービスディスカバリ: TD1
            Assert.IsTrue( executor.RunDevMenuCommandSystem( "servicediscovery import-all td1pass" ) );
            // reset sdev
            Assert.IsTrue( executor.ResetTarget() );
        }

        class DownloadTestParameter
        {
            interface IConverter<T>
            {
                T Convert(string s);
            }

            private T GetOrDefault<T>(System.Collections.IDictionary dict, string key, T defaultValue, IConverter<T> converter)
            {
                if (dict.Contains(key))
                {
                    return converter.Convert((string)dict[key]);
                }
                return defaultValue;
            }

            class IntConverter : IConverter<int>
            {
                public int Convert(string s)
                {
                    return System.Convert.ToInt32(s, 10);
                }
            }
            class UInt64Converter : IConverter<UInt64>
            {
                public UInt64 Convert(string s)
                {
                    return System.Convert.ToUInt64(s, 10);
                }
            }

            class ID64Converter : IConverter<ID64>
            {
                public ID64 Convert(string s)
                {
                    return new ID64(System.Convert.ToUInt64(s, 16));
                }
            }

            private int GetOrDefault(System.Collections.IDictionary dict, string key, int defaultValue)
            {
                return GetOrDefault<int>(dict, key, defaultValue, new IntConverter());
            }

            private ID64 GetOrDefault(System.Collections.IDictionary dict, string key, ID64 defaultValue)
            {
                return GetOrDefault<ID64>(dict, key, defaultValue, new ID64Converter());
            }
            private UInt64 GetOrDefault(System.Collections.IDictionary dict, string key, UInt64 defaultValue)
            {
                return GetOrDefault<UInt64>(dict, key, defaultValue, new UInt64Converter());
            }

            public int MaxAocCount { get; private set; } = 2000;
            public ID64 ApplicationId { get; private set; } = new ID64(0x01003400023e4000);
            public int LargeAocCount { get; private set; } = 2;
            public int SmallAocCount { get { return MaxAocCount - LargeAocCount; } }
            public int MaxAocInNsp { get; private set; } = 100;
            public int NandInstallCount { get; private set; } = 1;
            public int SdInstallCount { get { return MaxAocCount - NandInstallCount; } }
            public int SdFreeSpace { get; private set; } = 5 * 1048576; // 単位は Byte ではなく KiB.  1048576KiB = 1GiB
            public UInt64 LargeAocSize { get; private set; } = 2500ul * 1024 * 1024; // 2.5GB
            public UInt64 SmallAocSize { get; private set; } = 1024 * 1024;
            public int DefaultInitialPatchCapacity { get; private set; } = 8;

            public int AocVersion { get; private set; } = 0;

            public DownloadTestParameter(TestContext context)
            {
                var prop = context.Properties;
                MaxAocCount = GetOrDefault(prop, "MaxAocCount", MaxAocCount);
                ApplicationId = GetOrDefault(prop, "ApplicationId", ApplicationId);
                LargeAocCount = GetOrDefault(prop, "LargeAocCount", LargeAocCount);
                MaxAocInNsp = GetOrDefault(prop, "MaxAocInNsp", MaxAocInNsp);
                NandInstallCount = GetOrDefault(prop, "NandInstallCount", NandInstallCount);
                SdFreeSpace = GetOrDefault(prop, "SdFreeSpace", SdFreeSpace);
                LargeAocSize = GetOrDefault(prop, "LargeAocSize", LargeAocSize);
                SmallAocSize = GetOrDefault(prop, "SmallAocSize", SmallAocSize);
                DefaultInitialPatchCapacity = GetOrDefault(prop, "DefaultInitialPatchCapacity", DefaultInitialPatchCapacity);
                AocVersion = GetOrDefault(prop, "AocVersion", AocVersion);
            }

        }

        /// <summary>
        /// SDEV再初期化
        /// </summary>
        /// <param name="executor"></param>
        private void InitializeSdev( SigloHelper.CommodityExecutor.Context executor )
        {
            // 再初期化
            ExpectEvaluator.IsTrue( executor.InvokeUpdateKernelOnSDEV() );
            // 起動待ち
            System.Threading.Thread.Sleep( ReinitAfterInterval );
            // 再設定
            SetupSdevEnvironments( executor );
        }

        /// <summary>
        /// SdCard クリーンアップ.
        /// </summary>
        /// <param name="executor"></param>
        private void CleanupSdCard( SigloHelper.CommodityExecutor.Context executor )
        {
            // format
            Assert.IsTrue( executor.RunDevMenuCommand( "sdcard format" ) );
            // cleanup
            Assert.IsTrue( executor.RunDevMenuCommand( "sdcard cleanup" ) );
            // reset sdev
            Assert.IsTrue( executor.ResetTarget() );
            // status check
            Assert.IsTrue( executor.RunDevMenuCommand( "sdcard status" ) );
        }

        private bool DoUpload()
        {
            var doUpload = TestContext.Properties["DoUpload"];
            return doUpload != null && bool.Parse((string)doUpload);
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// テストメソッド: SetupSequence
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [Priority( 1 )]
        public void SetupSequence()
        {
            Log.WriteLine( TestMethodLogBegin, "SetupSequence" );

            var context = new SigloHelper.CommodityExecutor.Context( ActiveConfiguration );

            // CLIトークン( td1, DEV6 )事前参照.
            // 並列要求時に有効期限切れの場合、取得競合で失敗するため.
            var intermediate = GenerateIntermediateDirectoryAsMethod( context );
            D4cHelper.Configuration.CliToken.QueryLatestValidateToken( intermediate, CliTokenAccount, ProxyConfiguration );

            // td1( dev6 ), ネットワーク設定インポート.
            SetupSdevEnvironments( context );

            Log.WriteLine( TestMethodLogFinish, "SetupSequence" );
        }


        private ID64 MakeAocId(ID64 appId, int index)
        {
            return new ID64(appId.Value + (ulong)(0x1000 + index));
        }

        private bool WaitTicket(SigloHelper.CommodityExecutor.Context executor, ID64 titleId)
        {
            Log.WriteLine($"=== Waiting ticket: 0x{titleId} ");
            var endTime = DateTime.Now + TimeSpan.FromMinutes(5);
            while(DateTime.Now < endTime)
            {
                System.Threading.Thread.Sleep(10 * 1000);
                var hasTicket = executor.RunDevMenuCommandSystem(
                    $"ticket list-all --simple", $"0x{titleId}0000000000000000", null);
                if(hasTicket)
                {
                    Log.WriteLine($"=== Waiting ticket: 0x{titleId}  => Success");
                    return true;
                }
                else
                {
                    var syncTicketResult = executor.RunDevMenuCommandSystem(
                        $"shop sync-ticket");
                    if(syncTicketResult)
                    {
                        Log.WriteLine($"=== SyncTicket => Success");
                    }
                    else
                    {
                        Log.WriteLine($"=== SyncTicket => Failed");
                    }
                }
            }
            Log.WriteLine($"=== Waiting ticket: 0x{titleId}  => Failed");
            return false;
        }

        [TestMethod]
        [Timeout(7 * 3600 * 1000)] // 6時間タイムアウト
        [Priority( 2 )]
        [TestProperty("JIRA", "SIGLO-47169")]
        public void TestForMaxAddOnContentInstall()
        {
            Log.WriteLine(TestMethodLogBegin, "TestForMaxAddOnContentInstall");

            // Parameter
            var p = new DownloadTestParameter(TestContext);

            // Target command executor
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            var account = NintendoAccount;

            try
            {
                // 作業用ディレクトリ
                string intermediate = GenerateIntermediateDirectoryAsMethod(executor, "TestForMaxAddOnContentInstall");

                var request = new List<TestApplication.GenerateParameter<int>>();
                var purchaseList = new List<ID64>();

                var largeAoc = new TestApplication.MultipleAddonParameter(p.ApplicationId, p.AocVersion, 1, (ushort)p.LargeAocCount, (ushort)p.LargeAocSize, p.DefaultInitialPatchCapacity);
                request.Add(largeAoc);
                purchaseList.Add(MakeAocId(p.ApplicationId, 1));

                var smallAocBeginIndex = p.LargeAocCount + 1;
                for (int i = 0; i < p.SmallAocCount; i += p.MaxAocInNsp)
                {
                    var count = System.Math.Min(p.MaxAocInNsp, p.SmallAocCount - i);
                    ushort beginIndex = (ushort)(smallAocBeginIndex + i);
                    ushort endIndex = (ushort)(beginIndex + count - 1);

                    var smallAoc = new TestApplication.MultipleAddonParameter(p.ApplicationId, p.AocVersion, beginIndex, endIndex, (ushort)p.SmallAocSize, p.DefaultInitialPatchCapacity);
                    request.Add(smallAoc);
                    purchaseList.Add(MakeAocId(p.ApplicationId, beginIndex));
                }


                if (DoUpload())
                {
                    var uploadContents = TestApplication.MakeContents(intermediate, request);

                    D4cHelper.NspUploader uploader = new D4cHelper.NspUploader(intermediate,
                        ProxyConfiguration, D4cHelper.Configuration.ServerEnvironment.DEV6);

                    foreach (var path in uploadContents.ToNspPaths())
                    {
                        var options = new D4cHelper.NspUploader.UploadOptions(D4cHelper.NspUploader.UploadOptions.Flag.UseProductContents);
                        uploader.Publish(CliTokenAccount, path, options);
                        uploader.RegisterDemoContent(path);
                    }
                }

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "debug set-boolean-fwdbg --name ns.notification --key enable_network_update false"
                    ));

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "debug set-boolean-fwdbg --name ns.notification --key enable_version_list false"
                    ));

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "debug set-boolean-fwdbg --name ns.notification --key enable_download_task_list false"
                    ));

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "gamecard erase"
                    ));

                CleanupSdCard(executor);

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application uninstall --all"
                    ));

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application clear-task-status-list"
                    ));

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "shop unlink-device-all AND " +
                    "shop unregister-device-account AND " +
                    "shop register-device-account AND " +
                    "account clear_all AND " +
                    "account add"
                    ));
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(string.Format("account link --index 0 --id {0} --password {1}", account.ID, account.PASSWORD)));

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "shop delete-all-rights AND " +
                    "shop shop-account-status AND " +
                    "shop link-device"
                    ));

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"debug fill-sdcard-free-space --margin {p.SdFreeSpace}"
                    ));

                foreach(var id in purchaseList)
                {
                    const int maxRetryCount = 100;
                    bool result = false;
                    for(int i = 0; i < maxRetryCount; ++i)
                    {
                        Log.WriteLine($"Purchase 0x{id} : TryCount = {i}");
                        result = executor.RunDevMenuCommandSystem($"shop download-demo --id 0x{id}");
                        if(result)
                        {
                            break;
                        }

                        // 失敗してもサーバ上で成功しているケースがあるのでチケット同期を待つ
                        if(WaitTicket(executor, id))
                        {
                            result = true;
                            break;
                        }
                    }
                    Log.WriteLine($"Demo 0x{id} download result: {result}");
                    ExpectEvaluator.IsTrue(result);
                }

                System.Threading.Thread.Sleep(20 * 1000);

                {
                    const int TimeOutS = 3600 * 7;
                    const int TimeOutMs = TimeOutS * 1000;
                    const int RetryCount = 3;
                    bool downloadCompleted = false;
                    for (int i = 0; i < RetryCount; ++i)
                    {
                        Log.WriteLine($"Download : TryCount = {i}");

                        try
                        {
                            ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                                "application clear-task-status-list"
                                ));

                            executor.ControlTarget("connect", new List<string>() { "--reset" });

                            ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                                "application request-download-task-list"
                                ));
                            ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                                $"application wait-download 0x{p.ApplicationId} --timeout {TimeOutMs}", TimeOutS));
                        }
                        catch
                        {
                            continue;
                        }

                        downloadCompleted = true;
                        break;
                    }
                    ExpectEvaluator.IsTrue(downloadCompleted);
                }

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"application content-meta-status --all --check-rights", null, "NoRights"));

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"addoncontent check-count -c {p.NandInstallCount}"));

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"addoncontent check-count -s sdcard -c {p.SdInstallCount}"));

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"shop unlink-device"));

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    $"application content-meta-status --all --check-rights", null, "Personalized"));

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "application uninstall --all AND " +
                    "account clear_all AND " +
                    "shop unregister-device-account"
                    ));

            }
            finally
            {
                executor.ControlTarget("reset");

                bool result = false;
                for(int i = 0; i < 20 && !result; ++i)
                {
                    result = executor.RunDevMenuCommandSystem(
                        "application uninstall --all AND " +
                        "shop unlink-device-all AND" +
                        "account clear_all AND " +
                        "shop unregister-device-account"
                        );
                }
            }

            Log.WriteLine(TestMethodLogFinish, "TestForMaxAddOnContentInstall");
        }
    }
}
